/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ImgWidget.h
 *    A widget that visualizes an Img.
 *
 * @author Erik Einhorn
 * @date   2012/02/21
 */

#ifndef _MIRA_IMGWIDGET_H_
#define _MIRA_IMGWIDGET_H_

#include <QWidget>

#include <image/Img.h>
#include <image/Colormap.h>

class QImage;
class QScrollArea;
class QLayout;

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * A widget that visualizes an Img.
 * Optionally, a mouse handler can be specified to react on user
 * interaction with the image.
 */
class ImgWidget : public QWidget
{
	Q_OBJECT;
public:
	ImgWidget(QWidget* parent = 0, bool enableScroll=true);

	~ImgWidget();

	/// Set Image the image (optionally, min and max values can be specified for scaling the pixel values)
	void setImage(const Img<>& image, double scaleMin=0.0, double scaleMax=255.0);

	/// Sets a colormap for coloring single channel images
	void setColormap(const Colormap& map);
	void clearColormap();

	/// Get the Widget containing the Pixmap.
	QWidget* getContentWidget();

	/// enable automatic resizing to content.
	void setAutoResize(bool autoResize);

	/// enable automatic scaling of the image.
	void setAutoScale(bool autoScale);

	///	enables scrollbars
	void enableScroll(bool enable);

public:
	/** @name Mouse interaction support */
	//@{

	/**
	 * The mouse listener interface that can be implemented to react
	 * on user interaction.
	 */
	class IMouseListener
	{
	public:
		virtual ~IMouseListener() {}

		virtual void onMousePressed(QMouseEvent* e) = 0;
		virtual void onMouseReleased(QMouseEvent* e) = 0;
		virtual void onMouseMoved(QMouseEvent * e) = 0;
	};

	/// Sets a mouse listener.
	void setMouseListener(IMouseListener* listener);

	//@}

protected:
	//  overwritten from QWidget
	virtual void mousePressEvent(QMouseEvent* e);

	void contentsMousePressEvent ( QMouseEvent * e );

protected:
	class ImgPainterWidget;

	QLayout* mLayout;
	QScrollArea* mScrollArea;
	ImgPainterWidget* mContentWidget;

	bool mAutoResize;
	bool mAutoScale;
	QImage mCurrentImage;

	QVector<QRgb> mColorTable;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
