/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ColormapProperty.h
 *    Declaration of ColormapProperty and related classes.
 *
 * @author Erik Einhorn
 * @date   2011/04/02
 */

#ifndef _MIRA_COLORMAPPROPERTY_H_
#define _MIRA_COLORMAPPROPERTY_H_

#include <QVector>
#include <QColor>
#include <QComboBox>

#include <serialization/PropertyNode.h>
#include <serialization/SplitReflect.h>
#include <serialization/IsTransparentSerializable.h>

#include <image/Colormap.h>

#include <visualization/GuiVisualizationExports.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * A special property class that can be used as member, if you want to
 * provide a ComboBox where the user is able to select the available and
 * registered color colormaps.
 *
 * @see Colormap
 */
class MIRA_GUI_VISUALIZATION_EXPORT ColormapProperty
{
public:

	struct ColormapDesc;

	template <typename Reflector>
	void reflectRead(Reflector& r) {
		r.delegate(mColormapIdentifier);
	}

	template <typename Reflector>
	void reflectWrite(Reflector& r) {
		r.delegate(mColormapIdentifier);
		update();
	}

	MIRA_SPLIT_REFLECT_MEMBER

public:

	/// Creates property with no selected colormap
	ColormapProperty() : mColormapDesc(NULL) {}

	/// Tries to select the specified colormap
	ColormapProperty(const std::string& colormapIdentifier);

	/// Returns true, if a valid colormap was chosen.
	bool isValid() const;

	/// Assigns colormap identifier string
	ColormapProperty& operator=(const std::string& identifier) {
		mColormapIdentifier = identifier;
		update();
		return *this;
	}

	/// Returns the set colormap identifier as string
	const std::string& str() const {
		return mColormapIdentifier;
	}

	/**
	 * Returns the chosen color colormap.
	 * @throw XInvalidParameter if no valid colormap is set.
	 */
	const Colormap&  getColormap() const;

	/**
	 * Returns the chosen color colormap as color table in Qt compliant form.
	 * @throw XInvalidParameter if no valid colormap is set.
	 */
	const QVector<QRgb>& getColorTable() const;


public:

	/**
	 * Allows to specify a filter function.
	 * The filter function can be used to filter the colormaps that are
	 * shown in the combobox and that can be specified using the property.
	 * The filter function must take a pointer to a Colormap as parameter
	 * and should return true, if the color colormap is allowed, or false, if
	 * it should be filtered out.
	 *
	 * A predefined filter function is filterContinous().
	 *
	 * Example:
	 * \code
	 * property.setFilter(ColormapProperty::filterContinuous);
	 * \endcode
	 */
	void setFilter(boost::function<bool(const Colormap*)> filterFunction);

	/**
	 * Returns true, if the specified colormap has passed the optional set
	 * filter. (for internal use)
	 */
	bool filter(const Colormap* colormap) const;

	/**
	 * A predefined filter function that filters all ContinousColormap
	 * classes.
	 * It can be set using the setFilter() method, e.g:
	 * \code
	 * property.setFilter(ColormapProperty::filterContinuous);
	 * \endcode
	 */
	static bool filterContinuous(const Colormap* colormap)
	{
		// check if colormap is a ContinousColormap
		return dynamic_cast<const ContinuousColormap*>(colormap) != NULL;
	}

private:

	void update();
	void checkValid() const;


private:

	std::string mColormapIdentifier;
	const ColormapDesc* mColormapDesc;

	boost::function<bool(const Colormap*)> mFilterFunction;
};

template <typename SerializerTag>
class IsTransparentSerializable<ColormapProperty,SerializerTag> : public std::true_type {};

///////////////////////////////////////////////////////////////////////////////

class ColormapPropertyComboBox : public QComboBox
{
Q_OBJECT
public:
	ColormapPropertyComboBox(PropertyNode* p, QWidget* parent);
public slots:
	void slotSetValue(int value);
public:
	PropertyNode* property;

private:
	class ComboBoxDelegate;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
