/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PolygonObject.h
 *    Declaration of PolygonObject.
 *
 * @author Tim Langner
 * @date   2011/09/20
 */

#ifndef _MIRA_POLYGONOBJECT_H_
#define _MIRA_POLYGONOBJECT_H_

#include <visualization/3d/VisualizationObject.h>

#include <geometry/Polygon.h>

#ifndef Q_MOC_RUN
#include <OGRE/OgreColourValue.h>
#include <OGRE/OgreSceneManager.h>
#include <OGRE/OgreSceneNode.h>
#include <OGRE/OgreManualObject.h>
#include <OGRE/OgreMaterialManager.h>
#endif

namespace mira {

///////////////////////////////////////////////////////////////////////////////

template <typename T>
class PolygonObject : public VisualizationObject
{
public:
	typedef Point<T, 2> Point2;
	typedef Point<T, 3> Point3;
	typedef boost::geometry::model::ring<Point2> Polygon2;
	typedef boost::geometry::model::ring<Point3> Polygon3;

public:
	PolygonObject(Ogre::SceneManager* sceneManager, Ogre::SceneNode* parent = NULL);
	virtual ~PolygonObject();

	virtual void setOutlineColor(const Ogre::ColourValue& color);

	// filling is only guaranteed to work correctly for convex flat polygons!
	virtual void setFillColor(const Ogre::ColourValue& color);

	void setPolygon(const Polygon2& polygon);
	void setPolygon(const Polygon3& polygon);

	void setPolygons(const std::vector<Polygon2>& polygons);
	void setPolygons(const std::vector<Polygon3>& polygons);

protected:
	void setupPolygons();

	Ogre::ManualObject* mOutlineObject;
	Ogre::ManualObject* mAreaObject;
	Ogre::ColourValue mOutlineColor;
	Ogre::ColourValue mFillColor;

	std::vector<Polygon3> mPolygons;
};

///////////////////////////////////////////////////////////////////////////////

template <typename T>
PolygonObject<T>::PolygonObject(Ogre::SceneManager* sceneManager, Ogre::SceneNode* parent) :
	VisualizationObject(sceneManager, parent),
	mOutlineColor(Ogre::ColourValue::Black), mFillColor(0.f, 0.f, 0.f, 0.f)
{
	mOutlineObject = mSceneManager->createManualObject("PolygonObjectOutline"+ toString(this));
	mOutlineObject->setCastShadows(false);
	mNode->attachObject(mOutlineObject);
	mAreaObject = mSceneManager->createManualObject("PolygonObjectArea"+ toString(this));
	mAreaObject->setCastShadows(false);
	mNode->attachObject(mAreaObject);
	setOutlineColor(mOutlineColor);
	setFillColor(mFillColor);
	setupPolygons();
}

template <typename T>
PolygonObject<T>::~PolygonObject()
{
	mNode->detachObject(mOutlineObject);
	mSceneManager->destroyManualObject(mOutlineObject);
	mNode->detachObject(mAreaObject);
	mSceneManager->destroyManualObject(mAreaObject);
}

template <typename T>
void PolygonObject<T>::setOutlineColor(const Ogre::ColourValue& color)
{
	mOutlineColor = color;
	setupPolygons();
}

template <typename T>
void PolygonObject<T>::setFillColor(const Ogre::ColourValue& color)
{
	mFillColor = color;
	setupPolygons();
}

template <typename T>
void PolygonObject<T>::setPolygon(const Polygon2& polygon)
{
	Polygon2 p = polygon;
	boost::geometry::correct(p);

	mPolygons.clear();
	mPolygons.push_back(Polygon3());

	for (std::size_t i = 0; i < p.size(); ++i)
		mPolygons.back().push_back(Point3(p[i].x(), p[i].y(), 0));

	setupPolygons();
}

template <typename T>
void PolygonObject<T>::setPolygon(const Polygon3& polygon)
{
	Polygon3 p = polygon;
	// TODO correct 3D polygon (close it) since the following won't compile
	//boost::geometry::correct(p);

	mPolygons.clear();
	mPolygons.push_back(p);

	setupPolygons();
}

template <typename T>
void PolygonObject<T>::setPolygons(const std::vector<Polygon2>& polygons)
{
	mPolygons.clear();
	foreach(const Polygon2& polygon, polygons)
	{
		Polygon2 p = polygon;
		boost::geometry::correct(p);

		mPolygons.push_back(Polygon3());
		for (std::size_t i = 0; i < p.size(); ++i)
			mPolygons.back().push_back(Point3(p[i].x(), p[i].y(), 0));
	}

	setupPolygons();
}

template <typename T>
void PolygonObject<T>::setPolygons(const std::vector<Polygon3>& polygons)
{
	mPolygons = polygons;
	setupPolygons();
}

template <typename T>
void PolygonObject<T>::setupPolygons()
{
	std::size_t count=0;
	foreach(const Polygon3& polygon, mPolygons)
		count += polygon.size();

	mOutlineObject->clear();
	mOutlineObject->estimateVertexCount(count * 2);

	mOutlineObject->begin("TransparentNoLight", Ogre::RenderOperation::OT_LINE_LIST);

	foreach(const Polygon3& polygon, mPolygons)
	{
		for (std::size_t i = 1; i < polygon.size(); ++i)
		{
			const Point3& a = polygon[i-1];
			const Point3& b = polygon[i];

			mOutlineObject->position(Ogre::Vector3(a.x(), a.y(), a.z()));
			mOutlineObject->colour(mOutlineColor);
			mOutlineObject->position(Ogre::Vector3(b.x(), b.y(), b.z()));
			mOutlineObject->colour(mOutlineColor);
		}
	}

	mOutlineObject->end();

	mAreaObject->clear();

	foreach(const Polygon3& polygon, mPolygons)
	{
		mAreaObject->begin("TransparentNoLightTwoSided",
				           Ogre::RenderOperation::OT_TRIANGLE_FAN);
		for (std::size_t i = 0; i < polygon.size(); ++i)
		{
			const Point3& p = polygon[i];
			mAreaObject->position(Ogre::Vector3(p.x(), p.y(), p.z()));
			mAreaObject->colour(mFillColor);
		}
		mAreaObject->end();
	}
}

///////////////////////////////////////////////////////////////////////////////

}

#endif
