/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RemoteTestUnits.C
 *    Units for remote connection testcase
 *
 * @author Erik Einhorn
 * @date   2012/04/26
 */

#include <fw/Unit.h>

using namespace mira;

namespace mira { namespace test { 

///////////////////////////////////////////////////////////////////////////////

/**
 * Unit for remote connection testcase
 */
class RemoteTestUnit1 : public Unit
{
MIRA_OBJECT(RemoteTestUnit1)

public:

	RemoteTestUnit1() : Unit(Duration::milliseconds(250)) {
		mValue = 0;
		mEchoValue = -1;
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		Unit::reflect(r);
		r.method("echo", &RemoteTestUnit1::echo, this, "");
	}

protected:

	virtual void initialize() {
		mIntOut = publish<int>("Int");
		publishService(*this);
	}

	virtual void process(const Timer& timer) {
		if(mValue>50) {

			if(mEchoValue==-1) {
				std::cout << "Never got an RPC Echo" << std::endl;
				MIRA_FW.requestTermination(-1);
				return;
			}

			MIRA_FW.requestTermination();
			return;
		}

		mValue++;
		mIntOut.post(mValue);
	}

	bool echo(int val)
	{
		std::cout << "Got back: " << val << std::endl;
		mEchoValue = val;
		if(val!=mValue) {
			std::cout << "RPC Echo failed" << std::endl;
			MIRA_FW.requestTermination(-1);
		}
		return true;
	}


private:

	Channel<int> mIntOut;
	int mValue;
	int mEchoValue;
};



/**
 * Unit for remote connection testcase
 */
class RemoteTestUnit2 : public Unit
{
MIRA_OBJECT(RemoteTestUnit2)

public:

	RemoteTestUnit2() : Unit(Duration::milliseconds(100)) {}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		Unit::reflect(r);
	}

protected:

	virtual void initialize() {
		subscribe<int>("Int", &RemoteTestUnit2::onValue);
	}

	virtual void process(const Timer& timer) {

	}

	void onValue(ChannelRead<int> val) {
		std::cout << "Got: " << val->value() << std::endl;

		bool res = false;
		try {
			auto r = callService<bool>("RemoteTestUnit1","echo", val->value());
			res = r.get();
		} catch(...) {}

		if(!res) {
			std::cout << "Expected RPC to return 'true'" << std::endl;
			MIRA_FW.requestTermination(-1);
		}
	}


private:

};


///////////////////////////////////////////////////////////////////////////////

}}

MIRA_CLASS_SERIALIZATION(mira::test::RemoteTestUnit1, mira::Unit );
MIRA_CLASS_SERIALIZATION(mira::test::RemoteTestUnit2, mira::Unit );
