/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RemoteConnection.C
 *    Implementation of RemoteConnection.h.
 *
 * @author Tim Langner
 * @date   2010/11/18
 */

#include <fw/RemoteConnection.h>

#include <boost/array.hpp>

#include <utils/StringAlgorithms.h>

#include <serialization/adapters/std/list>
#include <serialization/adapters/std/set>

#include <fw/Framework.h>
#include <fw/MicroUnit.h>
#include <fw/UnitManager.h>
#include <fw/RemoteModule.h>
#include <fw/ChannelBufferSlotFlags.h>

// max allowed size of a message is limited to 256MB, this limit is used
// only in checkMessageHeader() to prevent overflows of possible Denial
// of Service attacks.
#define MAX_MESSAGE_SIZE 256*1024*1024

namespace mira {

///////////////////////////////////////////////////////////////////////////////

void KnownFramework::validate() const
{
	getHostPort();
}

boost::tuple<std::string, uint16> KnownFramework::getHostPort() const
{
	std::size_t splitPos = address.rfind(':');
	if (splitPos == std::string::npos)
		MIRA_THROW(XInvalidParameter, "'" << address << "' is not a valid address.");
	std::string host, port;
	host = address.substr(0, splitPos);
	port = address.substr(splitPos+1, address.size() - splitPos - 1);
	return boost::make_tuple(host, fromString<uint16>(port));
}

///////////////////////////////////////////////////////////////////////////////

void RemoteConnection::RPCRemoteFinishHandler::onRPCfinished(const Buffer<uint8>& answer)
{
	boost::mutex::scoped_lock lock(mConnectionMutex);
	if ( mConnection != NULL )
	{
		FrameworkMessageHeader header(answer.size(), RPC_RESPONSE_MSG);
		boost::array<boost::asio::const_buffer, 2> buffers =
			{{
				header.asBuffer(),
				boost::asio::buffer(answer.data(), answer.size())
			}};
		mConnection->write(buffers);
	}
}

void RemoteConnection::RPCRemoteFinishHandler::setConnection(RemoteConnection* iConnection)
{
	boost::mutex::scoped_lock lock(mConnectionMutex);
	mConnection = iConnection;
}

///////////////////////////////////////////////////////////////////////////////

void RemoteConnection::RPCRemoteRequestHandler::onRPCrequested(const Buffer<uint8>& request)
{
	boost::mutex::scoped_lock lock(mConnectionMutex);
	if ( mConnection != NULL )
	{
		FrameworkMessageHeader header(request.size(), RPC_REQUEST_MSG);
		boost::array<boost::asio::const_buffer, 2> buffers =
			{{
				header.asBuffer(),
				boost::asio::buffer(request.data(), request.size())
			}};
		mConnection->write(buffers);
	}
}

void RemoteConnection::RPCRemoteRequestHandler::setConnection(RemoteConnection* iConnection)
{
	boost::mutex::scoped_lock lock(mConnectionMutex);
	mConnection = iConnection;
}

///////////////////////////////////////////////////////////////////////////////

/// Constructs a remote connection that uses its own io service
RemoteConnection::RemoteConnection() :
	masterSlaveOffset(Duration::seconds(0)),
	mSocket(mService)
{
	init();
}

/// Constructs a remote connection that uses a given io service
RemoteConnection::RemoteConnection(boost::asio::io_service& service) :
	mService(service),
	mSocket(mService)
{
	init();
}

RemoteConnection::~RemoteConnection()
{
	mRPCFinishHandler->setConnection(NULL);
	mRPCRequestHandler->setConnection(NULL);
	foreach(const std::string& service, publishedServices)
		MIRA_FW.getRPCManager().removeRemoteService(service);
	if ( authority )
		delete authority;

	// remove all remote authorities from our local AuthorityManager
	foreach(const auto& a, mRemoteAuthorities)
		MIRA_FW.getAuthorityManager().removeAuthority(a.second.get());
	mRemoteAuthorities.clear();

	// ensure, that both ping threads are stopped
	if(MIRA_FW.getRemoteModule()->isPingTimeoutEnabled()) {
		mProcessPingThread.interrupt();
		mProcessPingThread.join();
		mCheckPingTimeoutThread.interrupt();
		mCheckPingTimeoutThread.join();
	}
}

bool RemoteConnection::isLocal() const
{
	try
	{
		if (mSocket.local_endpoint().address() == mSocket.remote_endpoint().address())
			return true;
	}
	catch(...)
	{}
	return false;
}

void RemoteConnection::start()
{
	if(MIRA_FW.getRemoteModule()->isPingTimeoutEnabled()) {
		// Ensure, that the last ping time is initialized properly.
		mPingLastReceived = Time::now();

		// Start a thread, which send pings and syncTime
		mProcessPingThread =
				boost::thread(boost::bind(&RemoteConnection::processPingThread,
				               this));

		// Start a thread, which checks if the connection is alive.
		mCheckPingTimeoutThread =
				boost::thread(boost::bind(&RemoteConnection::checkPingTimeoutThread,
				              this));
	}
}

void RemoteConnection::stop()
{
	if(MIRA_FW.getRemoteModule()->isPingTimeoutEnabled()) {
		mProcessPingThread.interrupt();
		mProcessPingThread.join();
		mCheckPingTimeoutThread.interrupt();
		mCheckPingTimeoutThread.join();
	}

	boost::mutex::scoped_lock lock(mStopMutex);
	if (mStopped)
		return;
	mStopped = true;
	//mSocket.shutdown(boost::asio::ip::tcp::socket::shutdown_both);
	mSocket.close();
	lock.unlock();
	onStop();
}

void RemoteConnection::syncTime()
{
	// Only sync time, if the frameworks are fully connected.
	if (mAuthState == AUTHSTATE_ACCEPTED)
	{
		if ( (Time::now()-mLastPTP) > Duration::seconds(30) )
		{
			mLastPTP = Time::now();

			// only do time synchronization if connection is to another host
			if (isLocal())
				synchronizeFrameworks();
			else
				sendPTP();
		}
	}
}

void RemoteConnection::sendPTP()
{
	// Send ptp sync packet
	writeMessage(PTP_SYNC);
	// get the time when the packet was send
	uint64 timestamp = Time::now().toUnixNS();
	// Send ptp follow up packet containing the time stamp when the sync packet was send
	FrameworkMessageHeader header(sizeof(uint64), PTP_FOLLOW_UP);
	boost::array<boost::asio::const_buffer, 2> buffers =
		{{
			header.asBuffer(),
			boost::asio::buffer((void*)&timestamp, sizeof(uint64))
		}};
	write(buffers);
}

void RemoteConnection::ping()
{
	if ( (Time::now()-mPingLastSend) > MIRA_FW.getRemoteModule()->getPingInterval() )
	{
		mPingLastSend = Time::now();
		if (remoteVersion >= 0x00030001)
			writeMessage(PING_MSG);
	}
}

bool RemoteConnection::hasPingTimeout() const
{
	// PING_MSG was added in protocol version 0x00030001
	if (remoteVersion < 0x00030001)
		return false;

	return (Time::now()-mPingLastReceived) > MIRA_FW.getRemoteModule()->getPingTimeout();
}

void RemoteConnection::publishChannels(const ChannelTypeMap& channels)
{
	Buffer<uint8> buffer;
	BinaryBufferSerializer bs(&buffer);
	bs.serialize(channels);
	MIRA_LOG(DEBUG) << "Framework: Publishing channels to remote framework ID='"
		<< frameworkID << "'";
	FrameworkMessageHeader header(buffer.size(), PUBLISH_CHANNEL_MSG);
	boost::array<boost::asio::const_buffer, 2> buffers =
		{{
			header.asBuffer(),
			boost::asio::buffer(buffer.data(), buffer.size())
		}};
	write(buffers);
}

void RemoteConnection::unpublishChannel(const std::string& channel)
{
	MIRA_LOG(DEBUG) << "Framework: Unpublishing channel to remote framework ID='"
		<< frameworkID << "'";
	FrameworkMessageHeader header(channel.size(), UNPUBLISH_CHANNEL_MSG);
	boost::array<boost::asio::const_buffer, 2> buffers =
		{{
			header.asBuffer(),
			boost::asio::buffer(channel.data(), channel.size())
		}};
	write(buffers);
}

void RemoteConnection::publishAuthorities(const AuthorityDescriptions& authorities)
{
	Buffer<uint8> buffer;
	BinaryBufferSerializer bs(&buffer);
	bs.serialize(authorities);
	MIRA_LOG(DEBUG) << "Framework: Publishing authorities to remote framework ID='"
		<< frameworkID << "'";
	FrameworkMessageHeader header(buffer.size(), PUBLISH_AUTHORITY_MSG);
	boost::array<boost::asio::const_buffer, 2> buffers =
		{{
			header.asBuffer(),
			boost::asio::buffer(buffer.data(), buffer.size())
		}};
	write(buffers);
}

void RemoteConnection::unpublishAuthorities(const AuthorityDescriptions& authorities)
{
	Buffer<uint8> buffer;
	BinaryBufferSerializer bs(&buffer);
	bs.serialize(authorities);
	MIRA_LOG(DEBUG) << "Framework: Unpublishing authorities to remote framework ID='"
		<< frameworkID << "'";
	FrameworkMessageHeader header(buffer.size(), UNPUBLISH_AUTHORITY_MSG);
	boost::array<boost::asio::const_buffer, 2> buffers =
		{{
			header.asBuffer(),
			boost::asio::buffer(buffer.data(), buffer.size())
		}};
	write(buffers);
}

void RemoteConnection::migrateUnit(const std::string& id)
{
	FrameworkMessageHeader header(id.size(), REQUEST_MIGRATION_MSG);
	boost::array<boost::asio::const_buffer, 2> buffers =
		{{
			header.asBuffer(),
			boost::asio::buffer(id.data(), id.size())
		}};
	write(buffers);
}

bool RemoteConnection::hasAuthority(const std::string& id) const
{
	if (mRemoteAuthorities.find(id) != mRemoteAuthorities.end())
		return true;
	return false;
}

void RemoteConnection::publishServices(const StringSet& services)
{
	Buffer<uint8> buffer;
	BinaryBufferSerializer bs(&buffer);

	bs.serialize((uint32)services.size());
	foreach(const std::string& s, services)
	{
		const RPCServer::Service& service = MIRA_FW.getRPCManager().getLocalService(s);
		bs.serialize(service,false);
	}

	MIRA_LOG(DEBUG) << "Framework: Publishing services to remote framework ID='"
		<< frameworkID << "'";
	FrameworkMessageHeader header(buffer.size(), PUBLISH_SERVICE_MSG);
	boost::array<boost::asio::const_buffer, 2> buffers =
		{{
			header.asBuffer(),
			boost::asio::buffer(buffer.data(), buffer.size())
		}};
	write(buffers);
}

void RemoteConnection::unpublishServices(const StringSet& services)
{
	Buffer<uint8> buffer;
	BinaryBufferSerializer bs(&buffer);
	bs.serialize(services);
	MIRA_LOG(DEBUG) << "Framework: Unpublishing services to remote framework ID='"
		<< frameworkID << "'";
	FrameworkMessageHeader header(buffer.size(), UNPUBLISH_SERVICE_MSG);
	boost::array<boost::asio::const_buffer, 2> buffers =
		{{
			header.asBuffer(),
			boost::asio::buffer(buffer.data(), buffer.size())
		}};
	write(buffers);
}

void RemoteConnection::valueChanged(ChannelRead<void> value, ServiceLevel& serviceLevel)
{
	// if data is coming from remote, don't send it again (avoid loops)
	if(value.getFlags() & SlotFlags::FROM_REMOTE)
		return;
	sendData(value, serviceLevel);
}

void RemoteConnection::parseMessage()
{
	MIRA_FW.getRemoteModule()->updateIncomingStats(mHeader.messageSize);
	switch((FrameworkMessageType)mHeader.messageType)
	{
		case PTP_SYNC:
		{
			mPTPSyncSlave = mHeaderReceived;
			break;
		}
		case PTP_FOLLOW_UP:
		{
			receivedPTPFollowUP(*reinterpret_cast<uint64*>(mMessage.data()));
			break;
		}
		case PTP_DELAY_RESPONSE:
		{
			receivedPTPDelayResponse(*reinterpret_cast<uint64*>(mMessage.data()));
			break;
		}
		case PTP_DELAY_REQUEST:
		{
			receivedPTPDelayRequest(mHeaderReceived.toUnixNS());
			break;
		}
		case PTP_FINISH:
		{
			receivedPTPFinish();
			break;
		}
		// The remote framework notifies us about a published channel
		case PUBLISH_CHANNEL_MSG:
		{
			receivedPublishChannelMsg();
			break;
		}
		case UNPUBLISH_CHANNEL_MSG:
		{
			receivedUnpublishChannelMsg();
			break;
		}
		// The remote framework notifies us about a published authority
		case PUBLISH_AUTHORITY_MSG:
		{
			receivedPublishAuthorityMsg();
			break;
		}
		// The remote framework notifies us about an authority that gets unpublished
		case UNPUBLISH_AUTHORITY_MSG:
		{
			receivedUnpublishAuthorityMsg();
			break;
		}
		case REQUEST_MIGRATION_MSG:
		{
			receivedRequestMigrationMsg();
			break;
		}
		case MIGRATION_MSG:
		{
			receivedMigrationMsg();
			break;
		}
		case MIGRATION_SINK_SUCCESS_MSG:
		{
			receivedMigrationSinkSuccessMsg();
			break;
		}
		case MIGRATION_SINK_FAILURE_MSG:
		{
			receivedMigrationSinkFailureMsg();
			break;
		}
		case MIGRATION_FINISHED_MSG:
		{
			receivedMigrationFinishedMsg();
			break;
		}
		// The remote framework notifies us about a published service
		case PUBLISH_SERVICE_MSG:
		{
			receivedPublishServiceMsg();
			break;
		}
		// The remote framework notifies us about a service that gets unpublished
		case UNPUBLISH_SERVICE_MSG:
		{
			receivedUnpublishServiceMsg();
			break;
		}
		// The remote framework subscribes on a channel
		case SUBSCRIBE_CHANNEL_MSG:
		{
			receivedSubscribeChannelRequest();
			break;
		}
		// The remote framework unsubscribes from a channel
		case UNSUBSCRIBE_CHANNEL_MSG:
		{
			uint8* ptr = &mMessage[0];
			std::string channelID(ptr, ptr + mMessage.size());
			MIRA_LOG(DEBUG) << "Framework: Remote framework ID='"
				<< frameworkID << "' unsubscribes from channel='" << channelID << "'";
			receivedUnsubscribeChannelRequest(channelID);
			break;
		}
		// Data of a channel we subscribed on has changed
		case WRITE_CHANNEL_MSG:
		{
			receivedWriteChannelMsg();
			break;
		}
		case RPC_REQUEST_MSG:
		{
			MIRA_FW.getRPCManager().handleRemoteRequest(mMessage, mRPCFinishHandler);
			break;
		}
		case RPC_RESPONSE_MSG:
		{
			MIRA_FW.getRPCManager().handleRemoteResponse(mMessage);
			break;
		}
		case TYPE_METADB_MSG:
		{
			receivedTypeMetaMsg();
			break;
		}
		case CHANNEL_META_MSG:
		{
			receivedChannelMetaMsg();
			break;
		}
		case PING_MSG:
		{
			receivedPingMsg();
			break;
		}
		default:
			MIRA_LOG(WARNING) << "Framework: Unknown message type from remote framework ID='"
				<< frameworkID << "' type=" << (int)mHeader.messageType;
			break;
	}
}

void RemoteConnection::init()
{
	remoteID = boost::uuids::nil_uuid();
	remoteVersion = 0;
	authority = NULL;
	synchronized = false;
	mLastPTP = Time::unixEpoch();
	mPingLastSend = Time::unixEpoch();
	mPingLastReceived = Time::unixEpoch();
	mRPCFinishHandler.reset(new RPCRemoteFinishHandler(this));
	mRPCRequestHandler.reset(new RPCRemoteRequestHandler(this));
	mAuthState = AUTHSTATE_NONE;
	mStopped = false;
}

void RemoteConnection::receivedPTPFollowUP(uint64 timestamp)
{
	mPTPSyncMaster = Time::fromUnixNS(timestamp);
	writeMessage(PTP_DELAY_REQUEST);
	mPTPDelaySlave = Time::now();
}

void RemoteConnection::receivedPTPDelayResponse(uint64 timestamp)
{
	mPTPDelayMaster = Time::fromUnixNS(timestamp);

	// adapt masterSlaveOffset ONLY if PTP sync is enabled!
	if(MIRA_FW.getRemoteModule()->isPTPSyncEnabled()) {
		masterSlaveOffset = (mPTPSyncSlave - mPTPSyncMaster) -
		                    (((mPTPSyncSlave - mPTPSyncMaster) +
		                    (mPTPDelayMaster - mPTPDelaySlave))/2);

		MIRA_LOG(DEBUG) << "Framework: delay to remote framework ID='"
			<< frameworkID << "' is " << masterSlaveOffset;
	}

	write(FrameworkMessageHeader(0, PTP_FINISH).asBuffer());
}

void RemoteConnection::receivedPTPDelayRequest(uint64 timestamp)
{
	// Send ptp delay response packet containing the time stamp
	// when delay request packet was received
	FrameworkMessageHeader header(sizeof(uint64), PTP_DELAY_RESPONSE);
	boost::array<boost::asio::const_buffer, 2> buffers =
		{{
			header.asBuffer(),
			boost::asio::buffer((void*)&timestamp, sizeof(uint64))
		}};
	write(buffers);
}

void RemoteConnection::receivedPTPFinish()
{
	synchronizeFrameworks();
	mLastPTP = Time::now();
}

void RemoteConnection::receivedSubscribeChannelRequest()
{
	BinaryBufferDeserializer bd(&mMessage);
	ServiceLevel serviceLevel;
	bd.deserialize(serviceLevel);
	std::string resolvedID = authority->resolveName(serviceLevel.channelID);
	MIRA_LOG(DEBUG) << "Framework: Got subscription from remote framework ID='"
			<< frameworkID << "' for channel='" << resolvedID << "'";
	if ( subscriptions.count(resolvedID) == 0 )
	{
		subscriptions[resolvedID] = 0;

		authority->subscribe<void>(resolvedID,
		                           boost::bind(&RemoteConnection::valueChanged,
		                                       this, _1, serviceLevel));

		try
		{
			// send last data that is contained in the channel so that the connected
			// framework has the newest data
			Channel<void> channel = authority->getChannel<void>(resolvedID);
			ChannelRead<void> value = channel.read();
			sendData(value, serviceLevel);
		}
		catch (XInvalidRead&)
		{
			// No data in the channel so we will wait for the next
			// channel callback.
		}
	}
}

void RemoteConnection::receivedUnsubscribeChannelRequest(const std::string& channelID)
{
	std::string resolvedID = authority->resolveName(channelID);
	if ( subscriptions.count(resolvedID) != 0 )
	{
		subscriptions.erase(resolvedID);
		authority->unsubscribe<void>(resolvedID);
	}
}

void RemoteConnection::receivedPublishChannelMsg()
{
	BinaryBufferDeserializer bd(&mMessage);
	std::map<std::string, Typename> channels;
	bd.deserialize(channels);

	foreach(const auto& channel, channels)
	{
		// If we have not known about the fact, that the remote framework has
		// a publisher for this channel test if we need to subscribe
		std::string channelID = MIRA_FW.getNameRegistry().resolve(channel.first,
		                                                         authority->getNamespace());
		if ( !MIRA_FW.getChannelManager().hasPublished(authority->getGlobalID(), channelID) )
		{
			MIRA_LOG(DEBUG) << "Framework: Got notification from remote framework ID='"
					<< frameworkID << "' about a published channel='" << channel.first
					<< "'" << " type='" << channel.second << "'";

			// tell framework that we act as a publisher of this channel
			// (however, we will not subscribe at the other framework
			// unless we have a subscriber for it)
			authority->publish<void>(channel.first, channel.second);

			if ( MIRA_FW.getChannelManager().hasSubscriber(channel.first) )
			{
				ServiceLevel serviceLevel = MIRA_FW.getRemoteModule()->getServiceLevel(channel.first);
				Buffer<uint8> slBuffer;
				BinaryBufferSerializer bs(&slBuffer);
				bs.serialize(serviceLevel);
				FrameworkMessageHeader header(slBuffer.size(), SUBSCRIBE_CHANNEL_MSG);
				boost::array<boost::asio::const_buffer, 2> buffers =
					{{
						header.asBuffer(),
						boost::asio::buffer(slBuffer.data(), slBuffer.size())
					}};
				write(buffers);
			}
		}
	}
}

void RemoteConnection::receivedUnpublishChannelMsg()
{
	std::string channelID((char*)mMessage.data(), mMessage.size());
	// if we had known about the fact, that the remote framework had
	// published this channel before we can unsubscribe that channel safely
	// from that framework since there will be no more data updates.
	if ( MIRA_FW.getChannelManager().hasPublished(authority->getGlobalID(), channelID) )
	{
		FrameworkMessageHeader header(channelID.length(), UNSUBSCRIBE_CHANNEL_MSG);
		boost::array<boost::asio::const_buffer, 2> buffers =
		{{
			header.asBuffer(),
			boost::asio::buffer(channelID)
		}};
		write(buffers);
		// we also no longer are a publisher of this channel
		authority->unpublish(channelID);
	}	
}

void RemoteConnection::receivedPublishAuthorityMsg()
{
	BinaryBufferDeserializer bd(&mMessage);
	AuthorityDescriptions authorities;
	bd.deserialize(authorities);

	// add authorities as RemoteAuthorities to our local AuthorityManager
	foreach(const AuthorityDescription& a, authorities)
	{
		// already published by this connection -> ignore
		if (mRemoteAuthorities.count(a.getGlobalID()) > 0)
			continue;
		RemoteAuthority* ra = new RemoteAuthority(a);
		try
		{
			MIRA_FW.getAuthorityManager().addAuthority(ra);
		}
		catch(XInvalidParameter& ex)
		{
			delete ra;
			MIRA_LOG(WARNING) << ex.message();
			continue;
		}
		mRemoteAuthorities[ra->getGlobalID()].reset(ra);
	}
}

void RemoteConnection::receivedUnpublishAuthorityMsg()
{
	BinaryBufferDeserializer bd(&mMessage);
	AuthorityDescriptions authorities;
	bd.deserialize(authorities);

	// remove authorities from our local AuthorityManager
	foreach(const AuthorityDescription& a, authorities)
	{
		auto it = mRemoteAuthorities.find(a.getGlobalID());
		if(it==mRemoteAuthorities.end())
			continue;

		MIRA_FW.getAuthorityManager().removeAuthority(it->second.get());
		mRemoteAuthorities.erase(it);
	}
}

void RemoteConnection::receivedPublishServiceMsg()
{
	BinaryBufferDeserializer bd(&mMessage);

	uint32 size;
	bd.deserialize(size);
	for(uint32 i=0; i<size; ++i)
	{
		RPCManager::RemoteService serviceInfo;
		bd.deserialize(serviceInfo,false);

		publishedServices.insert(serviceInfo.name);
		MIRA_LOG(DEBUG) << "Framework: Got notification from remote framework ID='"
			<< frameworkID << "' about a published service='" 
			<< serviceInfo.name << "'.";

		MIRA_FW.getRPCManager().addRemoteService(serviceInfo, mRPCRequestHandler);
	}
}

void RemoteConnection::receivedUnpublishServiceMsg()
{
	BinaryBufferDeserializer bd(&mMessage);
	StringSet services;
	bd.deserialize(services);

	foreach(const std::string& service, services)
	{
		publishedServices.erase(service);
		MIRA_LOG(DEBUG) << "Framework: Got notification from remote framework ID='"
			<< frameworkID << "' about a service='" << service 
			<< "' that gets unpublished";
		MIRA_FW.getRPCManager().removeRemoteService(service);
	}
}

void RemoteConnection::receivedWriteChannelMsg()
{
	BinaryBufferDeserializer bd(&mMessage);
	std::string channelID;
	StampedHeader sHeader;
	uint32 dataSize;
	bd.deserialize(channelID);
	bd.deserialize(sHeader);
	bd.deserialize(dataSize);
	Buffer<uint8> data(dataSize);
	try {
		boost::asio::read(mSocket, boost::asio::buffer(data.data(), data.size()));
	} catch(boost::system::system_error& e) {
		if(MIRA_FW.getRemoteModule()->isPingTimeoutEnabled()) {
			MIRA_THROW(XIO, "Error while reading channel message from socket of RemoteConnection: " << e.what())
		} else {
			MIRA_LOG(ERROR) << "Error while reading from socket of RemoteConnection: " << e.what();
			stop();
			return; // abort
		}
	}
	MIRA_FW.getRemoteModule()->updateIncomingStats(dataSize);
	Channel<void> channel = authority->getChannel<void>(channelID);
	ChannelWrite<void> d = channel.write();

	try
	{
		// Correct the timestamp by the latest slave master offset
		d->timestamp = sHeader.timestamp + masterSlaveOffset;
		d->frameID = sHeader.frameID;
		d->sequenceID = sHeader.sequenceID;
		d.writeSerializedValue(std::move(data));

		d.addFlags(SlotFlags::FROM_REMOTE); // mark that the data is coming from remote
	}
	catch(Exception& ex)
	{
		MIRA_LOG(ERROR) << ex.message() << " While receiving channel write message from remote framework.";
		d.discard();
	}
}

void RemoteConnection::receivedRequestMigrationMsg()
{
	std::string id((char*)mMessage.data(), mMessage.size());
	XMLDom xml;
	MIRA_LOG(DEBUG) << "Received migration request for unit '" << id << "'";
	mMigrationUnit = MIRA_FW.getUnitManager()->getUnit(id);
	if (!mMigrationUnit)
		return; // TODO handle this case e.g. send a error response
	mMigrationUnit->stop();
	XMLDom::iterator root = xml.root();
	root = "unit";
	root.add_attribute("id", mMigrationUnit->getID());
	root.add_attribute("ns", mMigrationUnit->getNamespace());
	XMLSerializer s(root);
	s.serialize("instance", mMigrationUnit);
	std::string xmlData = xml.saveToString();
	FrameworkMessageHeader header(xmlData.size(), MIGRATION_MSG);
	boost::array<boost::asio::const_buffer, 2> buffers =
		{{
			header.asBuffer(),
			boost::asio::buffer(xmlData.data(), xmlData.size())
		}};
	write(buffers);
}

void RemoteConnection::receivedMigrationMsg()
{
	try
	{
		std::string xmlData((char*)mMessage.data(), mMessage.size());
		XMLDom xml;
		xml.loadFromString(xmlData);
		mMigrationID = xml.root().get_attribute<std::string>("id");
		mMigrationNS = xml.root().get_attribute<std::string>("ns");
		XMLDeserializer d(xml.root());
		d.deserialize("instance", mMigrationUnit);
		write(FrameworkMessageHeader(0, MIGRATION_SINK_SUCCESS_MSG).asBuffer());
	}
	catch(Exception& ex)
	{
		MIRA_LOG(ERROR) << "Migration failed: " << ex.message();
		// migration failed
		mMigrationUnit.reset();
		write(FrameworkMessageHeader(0, MIGRATION_SINK_FAILURE_MSG).asBuffer());
	}
}

void RemoteConnection::receivedMigrationSinkSuccessMsg()
{
	if (!mMigrationUnit)
		return;
	// unit could be migrated/restored in remote framework
	// so we can remove it from our framework
	std::string id = mMigrationUnit->getGlobalID();
	mMigrationUnit.reset();
	MIRA_FW.getUnitManager()->removeUnit(id);
	write(FrameworkMessageHeader(0, MIGRATION_FINISHED_MSG).asBuffer());
}

void RemoteConnection::receivedMigrationSinkFailureMsg()
{
	if (!mMigrationUnit)
		return;
	// migration failed so start unit again
	mMigrationUnit->start();
	mMigrationUnit.reset();
}

void RemoteConnection::receivedMigrationFinishedMsg()
{
	if (!mMigrationUnit)
		return;
	// remove authority from manager and list if unpublishAuthorityMsg was
	// not already received
	ResourceName ns(mMigrationNS);
	std::string id = ns / mMigrationID;
	auto it = mRemoteAuthorities.find(id);
	if(it!=mRemoteAuthorities.end())
	{
		MIRA_FW.getAuthorityManager().removeAuthority(it->second.get());
		mRemoteAuthorities.erase(it);
	}
	// checkin the migrated unit and start it
	mMigrationUnit->checkin(mMigrationNS, mMigrationID);
	mMigrationUnit->start();
	MIRA_FW.getUnitManager()->addUnit(mMigrationUnit, mMigrationNS, mMigrationID);
	mMigrationUnit.reset();
}

void RemoteConnection::receivedTypeMetaMsg()
{
	BinaryBufferDeserializer bd(&mMessage);
	MetaTypeDatabase db;
	bd.deserialize(db);
	MIRA_FW.getMetaDatabase()->merge(db);
}

void RemoteConnection::receivedChannelMetaMsg()
{
	BinaryBufferDeserializer bd(&mMessage);
	std::string channel;
	TypeMetaPtr meta;
	bd.deserialize(channel);
	bd.deserialize(meta);
	authority->getChannel<void>(channel).setTypeMeta(meta);
}

void RemoteConnection::receivedPingMsg()
{
	mPingLastReceived = Time::now();
}

void RemoteConnection::sendData(ChannelRead<void> value, ServiceLevel& serviceLevel)
{
	// if we haven't sent meta information for this channel yet...
	bool& alreadySent = subscriptions[value.getChannelID()];
	if (!alreadySent)
	{
		Channel<void> channel = authority->getChannel<void>(value.getChannelID());
		TypeMetaPtr meta = channel.getTypeMeta();
		// check if there is any meta information and...
		if (meta)
		{
			// check if we already have sent detailed informations about
			// this meta type (if other channels have the same type we
			// will only send detailed informations once)
			if (sentMetaInformation.count(meta->getTypename()) == 0)
			{
				try
				{
					Buffer<uint8> buffer;
					BinaryBufferSerializer bs(&buffer);
					// create meta database for all types needed by channel type
					MetaTypeDatabase db = MIRA_FW.getMetaDatabase()->getDependentTypesDB(meta);
					bs.serialize(db);
					FrameworkMessageHeader header(buffer.size(), TYPE_METADB_MSG);
					boost::array<boost::asio::const_buffer, 2> buffers =
						{{
							header.asBuffer(),
							boost::asio::buffer(buffer.data(), buffer.size())
						}};
					write(buffers);
					sentMetaInformation.insert(meta->getTypename());
				}
				catch(Exception& ex)
				{
					MIRA_LOG_EXCEPTION(WARNING,ex) << "Exception:\n";
				}
			}
			// now send channel type meta information
			Buffer<uint8> buffer;
			BinaryBufferSerializer bs(&buffer);
			bs.serialize(value.getChannelID());
			bs.serialize(meta);
			FrameworkMessageHeader header(buffer.size(), CHANNEL_META_MSG);
			boost::array<boost::asio::const_buffer, 2> buffers =
				{{
					header.asBuffer(),
					boost::asio::buffer(buffer.data(), buffer.size())
				}};
			write(buffers);
			alreadySent = true;
		}
	}
	Buffer<uint8> dataHeader;
	BinaryBufferSerializer bs(&dataHeader);
	const Buffer<uint8>& data = value.readSerializedValue(serviceLevel.codecs);
	bs.serialize(value.getChannelID());
	bs.serialize((StampedHeader)(*value));
	bs.serialize((uint32)data.size());

	FrameworkMessageHeader header(dataHeader.size(), WRITE_CHANNEL_MSG);
	boost::array<boost::asio::const_buffer, 3> buffers =
		{{
			header.asBuffer(),
			boost::asio::buffer(dataHeader.data(), dataHeader.size()),
			boost::asio::buffer(data.data(), data.size())
		}};
	write(buffers);
}

void RemoteConnection::synchronizeFrameworks()
{
	if ( !synchronized )
	{
		publishChannels(MIRA_FW.getChannelManager().getPublishedChannels());
		publishServices(MIRA_FW.getRPCManager().getLocalServices());

		auto l = MIRA_FW.getAuthorityManager().getAuthorities();
		AuthorityDescriptions d;
		foreach(const std::string& a, l)
		{
			try
			{
				// only publish local authorities
				if (MIRA_FW.getAuthorityManager().isLocal(a))
					d.push_back(MIRA_FW.getAuthorityManager().getDescription(a));
			} catch(...)
			{}
		}

		publishAuthorities(d);
		synchronized = true;
	}
}

void RemoteConnection::updateOutgoingStats(std::size_t size)
{
	MIRA_FW.getRemoteModule()->updateOutgoingStats(size);
}

bool RemoteConnection::checkMessageHeader() const
{
	return (mHeader.messageSize < MAX_MESSAGE_SIZE) &&
	       (mHeader.messageType < MAX_MSG_COUNT);
}

void RemoteConnection::sendConnectDenied(const std::string& msg)
{
	writeMessage(CONNECT_DENIED_MSG, msg);
}

void RemoteConnection::processPingThread()
{
	while(!mProcessPingThread.interruption_requested()) {
		syncTime();

		// Only sending ping message, if the framework are synchronized.
		if (synchronized)
			ping();

		MIRA_SLEEP(10); //10ms
	}
}

void RemoteConnection::checkPingTimeoutThread()
{
	while(!mCheckPingTimeoutThread.interruption_requested()) {
		MIRA_SLEEP(1000);
		if (hasPingTimeout()) {
			bool socketError = false;
			MIRA_LOG(ERROR) << "Disconnect framework " << frameworkID
			                << " (from " << address.address << ") after "
			                << "ping timeout.";
			try {
				mSocket.shutdown(boost::asio::ip::tcp::socket::shutdown_both);
			} catch (boost::system::system_error& ex) {
				socketError = true;
				MIRA_LOG(ERROR) << "An exception occurred during shutdown the "
				                << "socket for remote framework " << frameworkID << ": " << ex.what();
			}
			try {
				mSocket.close();
			} catch (boost::system::system_error& ex) {
				socketError = true;
				MIRA_LOG(ERROR) << "An exception occurred during closing the "
				                << "socket for remote framework " << frameworkID << ": " << ex.what();
			}

			// If an error occurred during closing the socket, we stop() here.
			if (socketError)
				stop();
		}
	}
}

///////////////////////////////////////////////////////////////////////////////

}
