/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ErrorService.h
 *    Service used in framework to collect all errors and store them in a
 *    SQLite database.
 *
 * @author Tim Langner
 * @date   2010/11/08
 */

#ifndef _MIRA_ERRORSERVICE_H_
#define _MIRA_ERRORSERVICE_H_

#include <serialization/adapters/std/vector>

#include <fw/Framework.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/// Forward decl.
class SQLiteDB;

/**
 * @ingroup FWModule
 * Class that allows to store errors persistently in a SQLite database.
 */
class MIRA_FRAMEWORK_EXPORT ErrorService
{
public:
	///////////////////////////////////////////////////////////////////////////

	/**
	 * Error informations
	 */
	struct Error
	{
		std::string category;
		std::string authority;
		std::string trText;
		std::string message;
		uint32 count;
		Time timestamp;

		template<typename Reflector>
		void reflect(Reflector& r)
		{
			r.member("Category", category, "The error category.");
			r.member("Authority", authority,
			         "The authority that reports the error.");
			r.member("TrText", trText,
			         "The error text that is used for translation of errors.");
			r.member("Message", message,
			         "The error message that will not be translated.");
			r.member("Count", count,
			         "Number of error occurrences.");
			r.member("TimeStamp", timestamp,
			         "Timestamp of last occurrence of the error.");
		}

		Error() : count(0) {}
	};

	/// A vector of errors.
	typedef std::vector<Error> ErrorVector;

	///////////////////////////////////////////////////////////////////////////

	ErrorService(Authority& authority, const std::string& errorDBFilename);

	/// Reflect method for serialization
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.interface("IError");
		r.method("setError", &ErrorService::setError, this, "Set an error");
		r.method("resetError", &ErrorService::resetError, this, "Reset an error");
		r.method("resetErrors", &ErrorService::resetErrors, this, "Reset all errors");
		r.method("getErrorCount", &ErrorService::getErrorCount, this, "Get number of errors");
		r.method("getErrors", &ErrorService::getErrors, this, "Get all errors");
		r.method("getErrorText", &ErrorService::getErrorText, this, "Get a translated error text");
	}

	/**
	 * Stores an error in a category for the given authority in the database.
	 * If the same error exists in the database
	 * @param[in] category Error category
	 * @param[in] authority The authority that reports the error
	 * @param[in] errorTxt The error text that is used for translation of errors
	 * @param[in] message The error message that will not be translated
	 */
	void setError(const std::string& category, const std::string& authority,
	              const std::string& errorTxt, const std::string& message);

	/**
	 * Reset errors in a category for the given authority.
	 * @param[in] category The category to clear the errors
	 * @param[in] authority The authority that reported the error
	 */
	void resetError(const std::string& category, const std::string& authority);

	/**
	 * Reset all errors for the given authority.
	 * @param[in] authority The authority that reported the errors
	 */
	void resetErrors(const std::string& authority);

	/**
	 * Get number of errors for the given authority.
	 * @param[in] authority The authority that reported the errors
	 */
	uint32 getErrorCount(const std::string& authority);

	/**
	 * Get all errors from the database.
	 */
	ErrorVector getErrors();

	/**
	 * Get a translation of an error text for the specified language
	 */
	std::string getErrorText(const std::string& error, const std::string& language);

protected:
	void checkDB();
	boost::mutex mMutex;

	Authority& mAuthority;
	std::string mErrorDBFilename;
	boost::shared_ptr<SQLiteDB> mDB;
};

//////////////////////////////////////////////////////////////////////////////

}

#endif
