/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Channel.h
 *    Framework channel classes. Channels are the central communication
 *    objects for exchanging data in the framework.
 *
 * @author Erik Einhorn
 * @date   2010/09/06
 */

#ifndef _MIRA_CHANNEL_H_
#define _MIRA_CHANNEL_H_

#ifndef Q_MOC_RUN
#include <boost/shared_ptr.hpp>
#include <boost/type_traits/is_polymorphic.hpp>
#endif

#include <error/Exceptions.h>

#include <factory/TypeId.h>

#include <utils/EnumToFlags.h>
#include <utils/IteratorRangeContainer.h>

#include <fw/AbstractChannel.h>
#include <fw/ChannelSubscriber.h>
#include <fw/ChannelReadWrite.h>
#include <fw/ChannelReadInterval.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * An exception that occurs whenever a channel has no data
 */
class XInvalidRead : public XRuntime
{
public:
	XInvalidRead(std::string msg, const char* file=NULL, int line=0) throw() :
		XRuntime(msg, file, line) {}
};

///////////////////////////////////////////////////////////////////////////////

/// forward declaration
template <typename T>
class Channel;

///////////////////////////////////////////////////////////////////////////////

///@cond INTERNAL

/**
 * Concrete channels extend AbstractChannels by a type and provide
 * access methods to the underlying data (slots)
 */
template<typename T>
class ConcreteChannel : public AbstractChannel
{
public:

	typedef ChannelBuffer<T> Buffer;
	typedef typename Buffer::ValueType ValueType;
	typedef typename Buffer::Slot Slot;

	typedef ChannelSubscriber<T, ChannelRead<T> > Subscriber;
	typedef boost::shared_ptr<Subscriber> SubscriberPtr;

public:

	ConcreteChannel(const std::string& id) :
		AbstractChannel(id, new ChannelBuffer<T>)
	{
		// make sure, that our ConcreteChannel class has the same size
		// as AbstractChannel and that we are not polymorphic. Both is
		// important for our channel_cast to work as expected.
		static_assert(sizeof(ConcreteChannel)==sizeof(AbstractChannel),
		              "ConcreteChannel must have the same size as AbstractChannel");

		static_assert(boost::is_polymorphic<ConcreteChannel>::value==false,
		              "ConcreateChannel and AbstractChannel must not be polymorphic");
	}

public:

	/**
	 * Adds a subscriber that is not subscribed using a channel subscriber
	 * i.e. a subscriber without a callback
	 */
	void addSubscriber()
	{
		boost::mutex::scoped_lock lock(mSubscribersMutex);
		mNrOfSubscribersWithoutChannelSubscriber++;
	}

	/**
	 * Adds the specified subscriber to this channel and calls
	 * its attach() method.
	 */
	void addSubscriber(SubscriberPtr subscriber)
	{
		boost::mutex::scoped_lock lock(mSubscribersMutex);
		mSubscribers.push_back(subscriber);
		subscriber->attachChannel(this);
	}

	/**
	 * Removes a subscriber that has not subscribed using a channel subscriber
	 * i.e. a callack
	 */
	void removeSubscriber()
	{
		boost::mutex::scoped_lock lock(mSubscribersMutex);
		if (mNrOfSubscribersWithoutChannelSubscriber>0)
			--mNrOfSubscribersWithoutChannelSubscriber;
	}

	/**
	 * Removes the specified subscriber from this channel and calls its
	 * detach() method
	 */
	void removeSubscriber(SubscriberPtr subscriber)
	{
		AbstractChannel::removeSubscriber(subscriber);
	}

public:

	/**
	 * Obtains read access to the latest data element of this channel.
	 * @throw XInvalidRead when there is no data in the channel
	 * @return ChannelRead object that has already locked the data.
	 *         One can unlock the data manually or it is unlocked automatically
	 *         when the ChannelRead object goes out of scope.
	 */
	ChannelRead<T> read()
	{
		ChannelBufferBase::Slot* s = mBuffer->readNewestSlot();
		if(s==NULL)
			MIRA_THROW(XInvalidRead, "Cannot read from channel '"
			          << mID << "' since there is no data");
		return ChannelRead<T>(this, Buffer::castSlot(s));
	}

	/**
	 * Obtains read access to the element at the specified timestamp.
	 * If no slot with the exact timestamp exists
	 * (most of the time there will be no such slot) a slot will be chosen
	 * according to the query mode (see SlotQueryMode).
	 * @throw XInvalidRead when there is no data in the channel
	 * @return ChannelRead object that has already locked the data.
	 *         One can unlock the data manually or it is unlocked automatically
	 *         when the ChannelRead object goes out of scope.
	 */
	ChannelRead<T> read(const Time& timestamp, SlotQueryMode mode = NEAREST_SLOT,
	                    const Duration& tolerance = Duration::infinity())
	{
		ChannelBufferBase::Slot* s = mBuffer->readSlotAtTime(timestamp, mode);
		if(s==NULL)
			MIRA_THROW(XInvalidRead, "Cannot read slot from channel '"
			           << mID << "' at the specified time: '" << timestamp << "'");

		if(mira::abs(s->timestamp()-timestamp) > tolerance)
		{
			// !!DO NOT REMOVE THIS DUMMY!! It is needed to unlock the slot automatically!
			ChannelRead<T> dummy(this, Buffer::castSlot(s));
			MIRA_THROW(XInvalidRead, "Cannot read slot from channel '"
			           << mID << "', the nearest slot at timestamp '"
			           << s->timestamp() << "' is too far away from the specified time: '"
			           << timestamp << "'");
		}
		return ChannelRead<T>(this, Buffer::castSlot(s));
	}

	/**
	 * Obtains read access to an interval of elements before and after the
	 * requested timestamp. It is required that nrSlots >= olderSlots + newerSlots.
	 * The following rules are applied:
	 * - The interval will contain nrSlots number of elements
	 * - The interval will contain olderSlots number of slots with
	 *   a timestamp < requested timestamp (if they exist)
	 * - The interval will contain newerSlots number of slots with
	 *   a timestamp >= requested timestamp (if they exist)
	 * - If the number of available older + newer slots is < nrSlots more slots
	 *   are added according to the specified fillMode.
	 *   - If fillMode == PREFER_OLDER, first older slots are added
	 *     to the interval as needed. If there are not enough older slots,
	 *     newer slots are added to the interval as needed.
	 *   - If fillMode == PREFER_NEWER, first newer slots are added
	 *     to the interval as needed. If there are not enough newer slots,
	 *     older slots are added to the interval as needed.
	 * @throw XRuntime when there are not enough slots available.
	 * @return ChannelReadInterval object that has already locked all the data
	 *         in the interval. One can unlock the data manually or it is
	 *         unlocked automatically when the ChannelReadInterval
	 *         object goes out of scope.
	 */
	ChannelReadInterval<T> readInterval(const Time& timestamp, std::size_t nrSlots,
	                                    std::size_t olderSlots, std::size_t newerSlots,
	                                    IntervalFillMode fillMode = PREFER_NEWER)
	{
		std::list<ChannelBufferBase::Slot*> slotList;
		try
		{
			mBuffer->readInterval(timestamp, nrSlots, olderSlots, newerSlots,
			                      fillMode, slotList);
		}
		catch(Exception &ex)
		{
			MIRA_RETHROW(ex, "while trying to read interval ["
			             << olderSlots << " < " << timestamp << " <= " << newerSlots << "] of " << nrSlots
			             << " slots from channel '" << mID << "'")
		}
		return ChannelReadInterval<T>(this, slotList);
	}

	/**
	 * Obtains read access to an interval of elements in a given time span.
	 * - The first element in the interval will be the first slot with a timestamp > from
	 * - The last element in the interval will be the last slot with a timestamp <= to
	 * So the slot with timestamp = from is excluded and slot with
	 * timestamp = to is included in the interval.
	 *
	 * The exclusion of the slot whose timestamp is exactly equal to 'from'
	 * allows to read the data from the channel continuously by repeatedly
	 * calling the readInterval() method, without reading a single data slot
	 * twice. You just need to pass the timestamp of the last slot, that you
	 * have read, as parameter 'from" in the next call to readInterval().
	 *
	 * @return ChannelReadInterval object that has already locked all the data
	 *         in the interval. One can unlock the data manually or it is
	 *         unlocked automatically when the ChannelReadInterval
	 *         object goes out of scope.
	 */
	ChannelReadInterval<T> readInterval(const Time& from,
	                                    const Time& to = Time::eternity())
	{
		std::list<ChannelBufferBase::Slot*> slotList;
		try
		{
			mBuffer->readInterval(from, to, slotList);
		}
		catch(Exception &ex)
		{
			MIRA_RETHROW(ex, "while trying to read interval (" << from << "," << to
			             << "] from channel '" << this->getID() << "'")
		}
		return ChannelReadInterval<T>(this, slotList);
	}

	/**
	 * Obtains exclusive write access to the next free data slot of this
	 * channel.
	 * If no such slot exists the queue will grow until its maximum size.
	 * If all slots are locked and the queue has maximum size this method will
	 * block until a slot is released.
	 * @return ChannelWrite object that has already locked the data.
	 *         One can unlock the data manually or it is unlocked automatically
	 *         when the ChannelWrite object goes out of scope.
	 *         All subscribers will be notified.
	 */
	ChannelWrite<T> write()
	{
		ChannelBufferBase::Slot* s = mBuffer->requestWriteSlot();
		return ChannelWrite<T>(this, Buffer::castSlot(s));
	}

public:
	friend class Channel<T>;
};

///////////////////////////////////////////////////////////////////////////////


/**
 * Class with a single static cast method that casts an arbitrary
 * AbstractChannel to a ConcreteChannel.
 * The class is needed since C++ does not allow partial function template
 * specialization.
 */
template<typename T>
struct ChannelCaster {
	static ConcreteChannel<T>* cast(AbstractChannel* p)
	{
		// promote the channel if necessary (this will throw an exception, if
		// the channel cannot be promoted to the desired type)
		promote_channel<T>(p);

		assert(p->isTyped()); // at the point p will always be typed

		// check the type of our buffer's slot manually
		if(typeId<T>()!=p->getTypeId())
			MIRA_THROW(XBadCast, "Cannot cast channel '" << p->getID()
			          << "' to typed Channel of type '" << typeName<T>() << "'");

		// cast us "hard" using a static_cast.
		// This is safe, since we have checked the types above
		return static_cast<ConcreteChannel<T>*>(p);
	}
};

///////////////////////////////////////////////////////////////////////////////

// specialization to cast to untyped ConcreteChannel (void)
template<>
struct ChannelCaster<void> {
	static ConcreteChannel<void>* cast(AbstractChannel* p)
	{
		// cast us "hard" using a static_cast.
		// This is safe, since all channels can be casted to untyped channel
		return static_cast<ConcreteChannel<void>* >(p);
	}
};

///////////////////////////////////////////////////////////////////////////////

// specialization to cast polymorphic channels
template<typename T>
struct ChannelCaster<T*> {
	static ConcreteChannel<T*>* cast(AbstractChannel* p)
	{
		static_assert(std::is_base_of<mira::Object,T>::value,
		              "Pointers that are used in channels must be derived from "
		              "mira::Object. Pointers to other classes cannot be used.");

		// promote the channel if necessary (this will throw an exception, if the
		// the channel cannot be promoted to the desired type)
		promote_channel<T*>(p);

		// channel must be typed after promotion
		assert(p->getBuffer()->isPolymorphic());

		// cast us "hard" using a static_cast.
		// This is safe, since we have checked the types above
		return static_cast<ConcreteChannel<T*>* >(p);
	}
};

///////////////////////////////////////////////////////////////////////////////

template<typename T>
inline ConcreteChannel<T>* channel_cast(AbstractChannel* p)
{
	return ChannelCaster<T>::cast(p);
}

///@endcond

///////////////////////////////////////////////////////////////////////////////

/// forward decl for friend decl in Channel<T>
template <typename T>
class Channel;

/// @cond INTERNAL
template<typename T>
inline Channel<T> channel_cast(Channel<void> channel);
/// @endcond


/**
 * Flags specifying the access rights of an authority to a channel
 * Can be combined.
 */
enum ChannelAccessFlags {
	CHANNEL_ACCESS_NONE  = 0x00,	///< No access at all (Authority is not a publisher nor a subscriber)
	CHANNEL_ACCESS_READ  = 0x01,	///< Read access (Authority is a subscriber)
	CHANNEL_ACCESS_WRITE = 0x02		///< Write access (Authority is a publisher)
};
MIRA_ENUM_TO_FLAGS(ChannelAccessFlags)

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup FWModule
 * A channel is a named data container that stores data of a certain type
 * within a buffer. Each data element in the channel is called a slot.
 * The channel manages the access to the data and ensures safe non-blocking
 * access across different threads.
 * Use @ref read() and @ref write() to obtain access to the channel data.
 * Write access to the channel is non blocking. This means that data can be
 * written to the channel while other threads still maintain read access to the
 * data.
 * For write access the channel returns the next free slot that is not locked
 * by any pending write or read requests. Write and read access is allowed via
 * ChannelRead and ChannelWrite objects only. They automatically take care
 * that the data is unlocked at the right time and that the Channel is informed
 * about changed data.
 *
 * The signaling process is as follows:
 *
 * -# some @ref ChannelWrite finished the write process (went out of
 *    scope, called its finish method etc.).
 * -# the Channel calls @ref ChannelSubscriber::signal() of all registered
 *    ChannelSubsribers to inform them that new data is available.
 * -# Each ChannelSubscriber calls @ref SubscriberHandler::signal() of its
 *    associated SubscriberHandler.
 * -# The SubscriberHandler stores the ChannelSubscriber in an internal
 *    queue.
 * -# Within the SubscriberHandler thread the SubscriberHandler calls
 *    ChannelSubscriber::invoke() of the ChannelSubscriber.
 * -# the ChannelSubscriber finally calls the registered function that
 *    was assigned when it was constructed in @ref Authority::subscribe()
 *
 * This procedure ensures that type information won't get lost although
 * the call. ChannelSubscriber are queued within the SubscriberHandler to
 * decouple the threads.
 *
 * Implementation notes:
 *
 *  - must store shared pointer to BufferItem in mBuffer,
 *    since copying of ValueType can be expensive during
 *    a "push_back" and shared_mutex is not copyable.
 *
 *
 * This class is a proxy that allows restricted (based on access rights)
 * access to ConcreteChannel to read or/and write data to channels.
 */
template <typename T>
class Channel
{
public:

	/**
	 * Create channel proxy that is not assigned to a channel.
	 */
	Channel() : mChannel(), mAccessFlags(CHANNEL_ACCESS_NONE) {}

	/**
	 * Is used by Framework to create a channel proxy object
	 */
	Channel(ConcreteChannel<T>* channel, ChannelAccessFlags accessFlags) :
		mChannel(channel), mAccessFlags(accessFlags) {
		assert(mChannel!=NULL);
	}

	/**
	 * Reset the proxy object so that it becomes invalid.
	 */
	void reset() {
		mChannel=NULL;
		mAccessFlags = CHANNEL_ACCESS_NONE;
	}

	/**
	 *  Returns true if the proxy object is valid.
	 *  Meaning if it points to a channel.
	 */
	bool isValid() const {
		return mChannel!=NULL;
	}

	/**
	 * Checks if the channel is valid otherwise throws an exception.
	 *
	 * @throw XAccessViolation If channel is not valid.
	 */
	void validate() const {
		if(mChannel==NULL)
			MIRA_THROW(XAccessViolation, "No channel assigned to the channel proxy. "
			           "You can obtain a channel using Authority::getChannel()!");
	}

	/**
	 * @name Channel Information.
	 * The following methods allow to obtain information on the channel (name, type, etc).
	 */
	//@{

	/**
	 * Return the channel ID, its name.
	 */
	const std::string& getID() const {
		validate();
		return mChannel->getID();
	}

	/**
	 * Returns the type id of the channel.
	 * The type id is unique only within the same process. Use
	 * getTypename() to get a portable type information.
	 */
	int getTypeId() const {
		validate();
		return mChannel->getTypeId();
	}

	/**
	 * Returns true, if the channel is typed and false, if it is untyped
	 */
	bool isTyped() const { return getTypeId()>=0; }

	/**
	 * Return the typename of the channel. Used to obtain type of the contained
	 * binary data of untyped channels.
	 */
	Typename getTypename() const {
		validate();
		return mChannel->getTypename();
	}

	/**
	 * Set the typename of this channel. Can be used when the channel is
	 * untyped to specify the type of contained binary data.
	 */
	void setTypename(const Typename& type) {
		validate();
		mChannel->setTypename(type);
	}

	/**
	 * Return the type meta information for this channel
	 */
	TypeMetaPtr getTypeMeta() const {
		validate();
		return mChannel->getTypeMeta();
	}

	/**
	 * Set the type meta information for this channel
	 */
	void setTypeMeta(TypeMetaPtr meta) {
		validate();
		mChannel->setTypeMeta(meta);
	}

	//@}

public:

	/**
	 * @name Content information.
	 * The following methods provide information on the channels content.
	 */
	//@{

	/**
	 * Returns if the channel is empty (no data has been published)
	 */
	bool isEmpty() const {
		validate();
		return mChannel->getNrOfSlots() == 0;
	}

	/**
	 * Returns true, if this channel has at least one subscriber
	 */
	bool hasSubscriber() const {
		validate();
		return mChannel->hasSubscriber();
	}

	/**
	 * Returns true, if this channel has at least one publisher
	 */
	bool hasPublisher() const {
		validate();
		return mChannel->hasPublisher();
	}

	/**
	 * Returns true, if this channel has one solo slot only
	 */
	bool hasSoloSlot() const {
		validate();
		return mChannel->getBuffer()->getMaxSlots() == 1;
	}

	/**
	 * Returns the upper limit of slots that are allowed for this channel.
	 */
	std::size_t getMaxSlots() const {
		validate();
		return mChannel->getBuffer()->getMaxSlots();
	}

	/**
	 * Returns the number slots that are guaranteed to be kept in the channel buffer.
	 */
	std::size_t getMinSlots() const {
		validate();
		return mChannel->getBuffer()->getMinSlots();
	}

	/**
	 * Returns the timeframe that specifies how long a slot is
	 * guaranteed to be kept in the channel buffer.
	 */
	Duration getStorageDuration() const {
		validate();
		return mChannel->getBuffer()->getStorageDuration();
	}

	/**
	 * Returns how many slots (i.e. data objects) are currently stored in the channel buffer.
	 */
	std::size_t getNrOfSlots() const {
		validate();
		return mChannel->getBuffer()->getSize();
	}

	//@}

public:

	/**
	 * @name Accessing the channels data.
	 * The Channel provides different methods for reading and writing data
	 * in different flavors.
	 */
	//@{

	/**
	 * Waits until data in this channel becomes available. If a
	 * timeout is specified, the method times out if no data becomes
	 * available after the given time.
	 * @throw XAccessViolation if channel has no read access rights
	 * @param[in] timeout Wait for this duration or forever if Duration::infinity() (default)
	 * @return A valid ChannelRead, if data became available, or an invalid
	 *         ChannelRead, if the method timed out or the thread was
	 *         interrupted.
	 */
	ChannelRead<T> waitForData(const Duration& timeout = Duration::infinity()) const
	{
		validateReadAccess();

		Time end;
		if(!timeout.isValid() || timeout.isInfinity())
			end = Time::eternity();
		else
			end = Time::now() + timeout;

		while(!boost::this_thread::interruption_requested())
		{
			try {
				return mChannel->read();
			} catch(XInvalidRead& ex) {}
			if(Time::now()>end) // handle timeout
				break;
			MIRA_SLEEP(50)
		}
		return ChannelRead<T>();
	}


	/**
	 * Waits until this channel has at least one publisher. If a
	 * timeout is specified, the method times out if no publisher becomes
	 * available after the given time.
	 * @throw XAccessViolation if channel has no read access rights
	 * @param[in] timeout Wait for this duration or forever if Duration::infinity() (default)
	 * @return Returns true, if the wait was successfully and a publisher is available.
	 *         If the wait timed out, false is returned.
	 */
	bool waitForPublisher(const Duration& timeout = Duration::infinity()) const
	{
		validateReadAccess();

		Time end;
		if(!timeout.isValid() || timeout.isInfinity())
			end = Time::eternity();
		else
			end = Time::now() + timeout;

		while(!boost::this_thread::interruption_requested())
		{
			if(hasPublisher())
				return true;
			if(Time::now()>end) // handle timeout
				break;
			MIRA_SLEEP(50)
		}
		return false; // timeout
	}


	/**
	 * Obtains read access to the latest data element of this channel.
	 * @throw XAccessViolation if channel has no read access rights
	 * @throw XInvalidRead when there is no data in the channel
	 * @return ChannelRead object that has already locked the data.
	 *         One can unlock the data manually or it is unlocked automatically
	 *         when the ChannelRead object goes out of scope.
	 */
	ChannelRead<T> read() {
		validateReadAccess();
		return mChannel->read();
	}

	/**
	 * Obtains read access to the element at the specified timestamp.
	 * If no slot with the exact timestamp exists
	 * (most of the time there will be no such slot) a slot will be chosen
	 * according to the query mode (see SlotQueryMode).
	 *
	 * Internally calls ConcreteChannel::read().
	 *
	 * @param[in] timestamp Timestamp of the desired slot.
	 * @param[in] mode Specifies whether the nearest, newer or older slot
	 *                 to the given timestamp should be used (default: nearest slot)
	 * @param[in] tolerance The max. allowed difference between the given
	 *                      timestamp and the best slot that was found (default: infinity)
	 *
	 * @throw XAccessViolation if channel has no read access rights
	 * @throw XInvalidRead when there is no data in the channel
	 * @return ChannelRead object that has already locked the data.
	 *         One can unlock the data manually or it is unlocked automatically
	 *         when the ChannelRead object goes out of scope.
	 */
	ChannelRead<T> read(const Time& timestamp, SlotQueryMode mode=NEAREST_SLOT,
	                    const Duration& tolerance = Duration::infinity()) {
		validateReadAccess();
		return mChannel->read(timestamp, mode, tolerance);
	}

	/**
	 * Obtains read access to the element with the specified sequenceID within
	 * the given search interval
	 * If no slot with the exact sequenceID exists within the search interval an
	 * XInvalidRead exception is thrown.
	 *
	 * Internally uses readInterval(newestSlotTime - searchInterval, newestSlotTime)
	 *
	 * @param[in] sequenceID The sequenceID of the desired slot.
	 * @param[in] searchInterval The interval which is searched for the sequenceID (default: 1 second)
	 * @throw XAccessViolation if channel has no read access rights
	 * @throw XInvalidRead when there is no such sequenceID within the search
	 *                     interval the channel
	 * @return ChannelRead object that has already locked the data.
	 *         One can unlock the data manually or it is unlocked automatically
	 *         when the ChannelRead object goes out of scope.
	 */
	ChannelRead<T> read(uint32 sequenceID,
	                    const Duration& searchInterval = Duration::seconds(1)) {
		validateReadAccess();
		Time newestSlotTime = mChannel->read()->timestamp;
		ChannelReadInterval<T> interval = readInterval(newestSlotTime - searchInterval, newestSlotTime);
		typename ChannelReadInterval<T>::const_iterator iter = interval.begin();
		for(; iter != interval.end(); ++iter){
			if(iter->sequenceID == sequenceID)
				return (ChannelRead<T>)iter;
		}
		MIRA_THROW(XInvalidRead, "No slot with sequenceID "<<sequenceID<<
		           " found within searchInterval of channel");
	}

	/**
	 * Same as above, but always takes the nearest slot.
	 *
	 * Internally calls ConcreteChannel::read().
	 */
	ChannelRead<T> read(const Time& timestamp, const Duration& tolerance) {
		validateReadAccess();
		return mChannel->read(timestamp, NEAREST_SLOT, tolerance);
	}

	/**
	 * Obtains read access to an interval of elements before and after the
	 * requested timestamp. It is required that nrSlots >= olderSlots + newerSlots.
	 * The following the rules are applied:
	 * - The interval will contain nrSlots number of elements
	 * - The interval will contain olderSlots number of slots with a
	 *   timestamp < requested timestamp (if they exist)
	 * - The interval will contain newerSlots number of slots with a
	 *   timestamp >= requested timestamp (if they exist)
	 * - If the number of available older + newer slots is < nrSlots more slots
	 *   are added according to the specified fillMode.
	 *   - If fillMode == PREFER_OLDER, first older slots are added to the
	 *     interval as needed. If there are not enough older slots,
	 *     newer slots are added to the interval as needed.
	 *   - If fillMode == PREFER_NEWER, first newer slots are added to the
	 *     interval as needed. If there are not enough newer slots,
	 *     older slots are added to the interval as needed.
	 *
	 *
	 * Internally calls ConcreteChannel::readInterval().
	 *
	 * @throw XAccessViolation if channel has no read access rights
	 * @throw XRuntime when there are not enough slots available.
	 * @return ChannelReadInterval object that has already locked all the data
	 *         in the interval. One can unlock the data manually or it is
	 *         unlocked automatically when the ChannelReadInterval object
	 *         goes out of scope.
	 */
	ChannelReadInterval<T> readInterval(const Time& timestamp,
	                                    std::size_t nrSlots,
	                                    std::size_t olderSlots,
	                                    std::size_t newerSlots,
	                                    IntervalFillMode fillMode = PREFER_NEWER) {
		validateReadAccess();
		return mChannel->readInterval(timestamp, nrSlots, olderSlots,
		                              newerSlots, fillMode);
	}

	/**
	 * Obtains read access to an interval of elements in a given time span.
	 * - The first element in the interval will be the first slot with a timestamp > from
	 * - The last element in the interval will be the last slot with a timestamp <= to
	 * So slot with timestamp = from is excluded and slot with timestamp = to
	 * is included in the interval.
	 *
	 * Internally calls ConcreteChannel::readInterval().
	 *
	 * @throw XAccessViolation if channel has no read access rights
	 * @return ChannelReadInterval object that has already locked all the data
	 *         in the interval. One can unlock the data manually or it is
	 *         unlocked automatically when the ChannelReadInterval object
	 *         goes out of scope.
	 */
	ChannelReadInterval<T> readInterval(const Time& from,
	                                    const Time& to=Time::eternity()) {
		validateReadAccess();
		return mChannel->readInterval(from, to);
	}

	/**
	 * Obtains exclusive write access to the next free data slot of this
	 * channel.
	 * If no such slot exists the queue will grow until its maximum size.
	 * If all slots are locked and the queue has maximum size this method will
	 * block until a slot is released.
	 *
	 * Internally calls ConcreteChannel::write().
	 *
	 * @throw XAccessViolation if channel has no write access rights
	 * @return ChannelWrite object that has already locked the data.
	 *         One can unlock the data manually or it is unlocked automatically
	 *         when the ChannelWrite object goes out of scope.
	 *         All subscribers will be notified.
	 */
	ChannelWrite<T> write() {
		validateWriteAccess();
		return mChannel->write();
	}

	/**
	 * Same as write above with an additional parameter copyLatestData
	 * that allows you to specify whether the channel should copy the latest
	 * data available into the write slot.
	 * This is suitable whenever the caller wants to modify the latest data
	 * partially since it allows to omit an additional read call.
	 * To use this feature the caller must obtain read access, i.e. it must
	 * be a subscriber of the channel.
	 *
	 * Internally calls ConcreteChannel::write().
	 *
	 * @throw XAccessViolation if channel has no write access rights
	 * @return ChannelWrite object that has already locked the data.
	 *         One can unlock the data manually or it is unlocked automatically
	 *         when the ChannelWrite object goes out of scope.
	 *         All subscribers will be notified.
	 */
	ChannelWrite<T> write(bool copyLatestData) {
		validateWriteAccess();

		// check if we should and need to copy data into the write slot
		if(copyLatestData
			&& (mChannel->getBuffer()->getMaxSlots()>1) // only necessary if we may have more than one slot
			&& (mChannel->getBuffer()->getSize()>0))    // and only if we have at least one slot yet
		{
			ChannelRead<T> latest = mChannel->read();
			Stamped<T> tmp(*latest);
			latest.finish();
			ChannelWrite<T> writeSlot = mChannel->write();
			*writeSlot = tmp;
			return writeSlot;

		} else
			return mChannel->write();
	}

	/**
	 * Returns the latest data from the channel by value.
	 * This method is provided to allow easy read access to the channel, however
	 * internally it leads to unnecessary copying of data and should be used
	 * for lightweight objects only. For large objects like range scans or
	 * images use the read() method to obtain read access without any
	 * performance penalties.
	 *
	 * Internally calls ConcreteChannel::read().
	 *
	 * @throw XAccessViolation if channel has no read access rights
	 * @return Latest stamped data from the channel
	 */
	Stamped<T> get() {
		validateReadAccess();
		ChannelRead<T> value = mChannel->read();
		return *value;
	}

	/**
	 * Returns the data at the given time by value.
	 * This method is provided to allow easy read access to the channel, however
	 * internally it leads to unnecessary copying of data and should be used
	 * for lightweight objects only. For large objects like range scans or
	 * images use the read() method to obtain read access without any
	 * performance penalties.
	 *
	 * Internally calls ConcreteChannel::read().
	 *
	 * @throw XAccessViolation if channel has no read access rights
	 * @return Latest stamped data from the channel
	 */
	Stamped<T> get(const Time& timestamp, SlotQueryMode mode=NEAREST_SLOT,
	               const Duration& tolerance = Duration::infinity()) {
		validateReadAccess();
		ChannelRead<T> value = mChannel->read(timestamp, mode, tolerance);
		return *value;
	}

	/**
	 * Same as above, but always takes the nearest slot.
	 *
	 * Internally calls ConcreteChannel::read().
	 */
	Stamped<T> get(const Time& timestamp, Duration tolerance) {
		validateReadAccess();
		ChannelRead<T> value = mChannel->read(timestamp, NEAREST_SLOT, tolerance);
		return *value;
	}

private:

	///@cond INTERNAL

	/// Struct is used in get(...,Filter&&)
	struct GetTime
	{
		GetTime(const Time& iT0) : t0(iT0) {}

		//typedef const Time& result_type;
		typedef float result_type;
		result_type operator()(const Stamped<T>& p) const {
			//return p.timestamp;
			return (p.timestamp-t0).totalMilliseconds();
		}

		Time t0;
	};

	/// Struct is used in get(...,Filter&&)
	struct GetValue
	{
		typedef const T& result_type;
		result_type operator()(const Stamped<T>& p) const {
			return p;
		}
	};

	/// @endcond

public:

	/**
	 * Returns the data at the given time by value. Moreover, this variant
	 * of the get method allows to specify a filter that is applied on the
	 * data, e.g. linear interpolation, etc.
	 *
	 * The following example returns the value at timestamp t, where the
	 * value is linearly interpolated between two samples in the channel
	 * near the timestamp t:
	 * \code
	 * myValue = myChannel.get(t, LinearInterpolator());
	 * \endcode
	 *
	 * If you do not need filtering or interpolation you should use the get()
	 * method without a filter or the read() method for large objects to obtain
	 * direct read access without any performance penalties.
	 * @throw XAccessViolation if channel has no read access rights
	 * @return Stamped interpolated data
	 */
	template <typename Filter>
	Stamped<T> get(const Time& timestamp, Filter&& filter) {
		validateReadAccess();

		ChannelReadInterval<T> data;
		try
		{
			data = readInterval(timestamp, filter.samples(),
			                    filter.samplesBefore(),
			                    filter.samplesAfter());
			// check filter requirements
			if (!filter.canExtrapolate())
			{
				int samplesBefore = filter.samplesBefore();
				int samplesAfter = filter.samplesAfter();
				// a rbegin in ChannelReadInterval could speed this up
				for (auto it = data.begin(); it != data.end(); ++it)
				{
					if (it->timestamp <= timestamp)
						--samplesBefore;
					else if (it->timestamp >= timestamp)
						--samplesAfter;
				}

				if (samplesBefore > 0)
					MIRA_THROW(XRuntime, "Filter::samplesBefore: Requirements not meet. Missing " << samplesBefore << " items.");
				if (samplesAfter > 0)
					MIRA_THROW(XRuntime, "Filter::samplesAfter: Requirements not meet. Missing " << samplesAfter << " items.");
			}
		}
		catch(Exception&)
		{
			// read interval failed so test if the channel only contains a single data element
			// if so return it as a special case for interpolation of channels that will only
			// receive a single update during their lifetime
			if (mChannel->getNrOfSlots() == 1)
				return get();
			// more than one data element in the channel but still not sufficient for interpolation
			// rethrow is our only option
			throw;
		}

		const Time t0 = data.begin()->timestamp;

		typedef typename ChannelReadInterval<T>::const_iterator const_iterator;
		typedef boost::transform_iterator<GetTime, const_iterator> GetTimeIterator;
		GetTimeIterator begin1 = GetTimeIterator(data.begin(), GetTime(t0));
		GetTimeIterator end1   = GetTimeIterator(data.end()  , GetTime(t0));

		typedef boost::transform_iterator<GetValue, const_iterator> GetValueIterator;

		GetValueIterator begin2=GetValueIterator(data.begin(), GetValue());
		GetValueIterator end2  =GetValueIterator(data.end()  , GetValue());

		typedef mira::IteratorRangeContainer<GetTimeIterator> TimeContainer;
		typedef mira::IteratorRangeContainer<GetValueIterator> ValueContainer;
		TimeContainer  timeContainer(begin1, end1);
		ValueContainer valueContainer(begin2, end2);

		return makeStamped(filter.template apply<float, T>(timeContainer,
		                                                   valueContainer,
		                                                   (timestamp-t0).totalMilliseconds()),
		                                                   timestamp);
	}

	/**
	 * Writes the specified data into the channel.
	 * This method is provided to allow easy write access to the channel, however
	 * internally it leads to unnecessary copying of data and should be used
	 * for lightweight objects only. For large objects like range scans or
	 * images use the write() method to obtain write access without any
	 * performance penalties.
	 * @throw XAccessViolation if channel has no write access rights
	 */
	void post(const Stamped<T>& value) {
		validateWriteAccess();
		ChannelWrite<T> writeSlot = mChannel->write();
		*writeSlot = value;
	}

	/**
	 * Writes the specified data with the specified time stamp into the channel.
	 * This method is provided to allow easy write access to the channel, however
	 * internally it leads to unnecessary copying of data and should be used
	 * for lightweight objects only. For large objects like range scans or
	 * images use the write() method to obtain write access without any
	 * performance penalties.
	 * @throw XAccessViolation if channel has no write access rights
	 */
	template <typename U>
	void post(const U& value, const Time& timestamp = Time::now()) {
		validateWriteAccess();
		ChannelWrite<T> writeSlot = mChannel->write();
		*writeSlot = makeStamped(value, timestamp);
	}

	//@}

public:

	/// @cond INTERNAL
	template<typename U>
	friend Channel<U> channel_cast(Channel<void> channel);
	/// @endcond

public:

	/// @cond INTERNAL
	void dbgDump(bool brief=true) { mChannel->dbgDump(brief); }
	/// @endcond

private:

	void validateReadAccess() const
	{
		validate();
		if((mAccessFlags & CHANNEL_ACCESS_READ)==0)
			MIRA_THROW(XAccessViolation, "You are not allowed to read from channel '"
			           << mChannel->getID() << "'. Forgot to subscribe?");
	}

	void validateWriteAccess() const
	{
		validate();
		if((mAccessFlags & CHANNEL_ACCESS_WRITE)==0)
			MIRA_THROW(XAccessViolation, "You are not allowed to write to channel '"
			           << mChannel->getID() << "'. Forgot to publish?");
	}

private:

	ConcreteChannel<T>* mChannel;
	ChannelAccessFlags mAccessFlags;
};

///////////////////////////////////////////////////////////////////////////////

///@cond INTERNAL
template<typename U>
inline Channel<U> channel_cast(Channel<void> channel)
{
	channel.validate();
	return Channel<U>(channel_cast<U>(channel.mChannel), channel.mAccessFlags);
}
///@endcond

///////////////////////////////////////////////////////////////////////////////

} // namespace

#include "impl/ChannelReadWrite.hpp"
#include "impl/ChannelReadInterval.hpp"

#endif
