/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ChannelSynchronizer.h
 *    Filter for synchronized callbacks on subscribed channels.
 *
 * @author Tim Langner, Ronny Stricker
 * @date   2011/04/18
 */

#ifndef _MIRA_CHANNELSYNCHRONIZER_H_
#define _MIRA_CHANNELSYNCHRONIZER_H_

#ifndef Q_MOC_RUN
#include <boost/preprocessor/seq.hpp>
#include <boost/preprocessor/tuple/elem.hpp>
#include <boost/preprocessor/repetition/repeat.hpp>
#include <boost/preprocessor/comparison/not_equal.hpp>
#include <boost/preprocessor/repetition/for.hpp>
#include <boost/preprocessor/comma_if.hpp>
#include <boost/preprocessor/arithmetic/inc.hpp>

#include <boost/tuple/tuple.hpp>
#endif

#include <fw/Framework.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

#ifdef DOXYGEN
// This is a fake class to generate a doxygen documentation
/**
 * Class that can be registered as a filter when subscribing to more than 
 * one channel to only get a callback when all channels have data with 
 * a similar timestamp (differ only in the specified tolerance).
 * To synchronize 3 or 4 channels, you have to use the ChannelSynchronizer3 and
 * ChannelSynchronizer4 template class. To synchronize arbitrary number of
 * channels, just define MIRA_CHANNEL_SYNCHRONIZER(MySynchronizerName, 5).
 * Where MySynchronizerName defines the name of the template class which will
 * be generated, the number will define the number of channels, which the
 * synchronizer is using. After that you can use your MySynchronizerName class
 * exactly the same way.
 *
 * \code
 * MyUnit : public MicroUnit
 * {
 * 		void initialize()
 * 		{
 * 			myFilter.subscribe(*this, "IntChannelA", "IntChannelB",
 * 			                   &MyUnit::onCallback, this, Duration::milliseconds(500));
 * 		}
 *
 * 		void onCallback(ChannelRead<int> channelA, ChannelRead<int> channelB)
 * 		{}
 *
 * 		ChannelSynchronizer<int, int> myFilter;
 * }
 * \endcode
 */
template <typename A, typename B>
class ChannelSynchronizer
{
public:

	ChannelSynchronizer() :
		mLastA(Time::unixEpoch()),
		mLastB(Time::unixEpoch()){}

	/**
	 * Call this instead of Authority::unsubscribe(...) to unsubscribe
	 * all the channels of the synchronizer
	 */
	void unsubscribe(Authority& authority)\
	{
		authority.unsubscribe<type0>(mChannel0.getID()); authority.unsubscribe<type1>(mChannel1.getID());
	}

	/**
	 * Call this instead of Authority::subscribe()
	 * Function provided for convenience (if no callback function is required).
	 */
	void subscribe(Authority& authority,
					const std::string& channelIDA,
					const std::string& channelIDB,
					const Duration& t = Duration::seconds())
	{
		mFunction = NULL;
		mChannelA = authority.subscribe<A>(channelIDA, &ChannelSynchronizer<A,B>::callbackA, this, t);
		mChannelB = authority.subscribe<B>(channelIDB, &ChannelSynchronizer<A,B>::callbackB, this, t);
		mTolerance = t;
	}

	/**
	 * Call this instead of Authority::subscribe()
	 * @param authority The authority that does the basic subscription
	 * @param channelIDA The ID of the channel A we are subscribing on
	 * @param channelIDB The ID of the channel B we are subscribing on
	 * @param fn The function that should be called when the filter matches
	 * @param t Maximum time difference of the channels message timestamps for
	 *          still being synchronous
	 */
	void subscribe(Authority& authority,
	               const std::string& channelIDA,
	               const std::string& channelIDB,
	               boost::function<void (ChannelRead<A>, ChannelRead<B>)> fn,
	               const Duration& t = Duration::seconds(0))
	{
		mFunction = fn;
		mChannelA = authority.subscribe<A>(channelIDA, &ChannelSynchronizer<A,B>::callbackA, this, t);
		mChannelB = authority.subscribe<B>(channelIDB, &ChannelSynchronizer<A,B>::callbackB, this, t);
		mTolerance = t;
	}

	/**
	 * Same as above but with a function and object pointer. Provided for convenience.
	 */
	template<typename Class>
	void subscribe(Authority& authority,
	               const std::string& channelIDA,
	               const std::string& channelIDB,
	               void (Class::*f)(ChannelRead<A>, ChannelRead<B>),
	               Class* obj,
	               const Duration& t = Duration::seconds(0))
	{
		subscribe(authority, channelIDA, channelIDB, boost::bind(f, obj, _1, _2), t);
	}

	/// Called when channel A has new data
	void callbackA(ChannelRead<A> a)
	{
		ChannelRead<B> b;
		try
		{
			b = mChannelB.read(a.getTimestamp(), mTolerance);
		}
		catch(...)
		{ return; }
		call(a, b);
	}

	/// Called when channel B has new data
	void callbackB(ChannelRead<B> b)
	{
		ChannelRead<A> a;
		try
		{
			a = mChannelA.read(b.getTimestamp(), mTolerance);
		}
		catch(...)
		{ return; }
		call(a, b);
	}

	void call(ChannelRead<A> a, ChannelRead<B> b)
	{
		if (mLastA == a.getTimestamp())
			return;
		if (mLastB == b.getTimestamp())
			return;
		mLastA = a.getTimestamp();
		mLastB = b.getTimestamp();
		mChannelReadA = a;
		mChannelReadB = b;
		if (mFunction)
			mFunction(a, b);
	}

	/**
	 * Return (synchronized) ChannelRead objects.
	 * The ChannelRead objects are always synchronized by the channel with the lowest
	 * update frequency. Please note that the ChannelRead objects may be invalid if
	 * no data has been published yet.
	 */
	boost::tuple<ChannelRead<A>, ChannelRead<B> > read()
	{
		return boost::make_tuple(mChannelReadA,mChannelReadB);
	}

	/**
	 * Return true if all ChannelRead objects contain valid data.
	 */
	bool isValid() const {
		return ( mChannelReadA.isValid() && mChannelReadB.isValid() );
	}

	/**
	 * Return the latest (synchronized) element once it is available.
	 * If the functions times out, invalid channelRead objects will be returned.
	 * Please note that the function does not regard synchronized elements that
	 * were available BEFORE this function was called.
	 */
	boost::tuple< ChannelRead<type0> , ChannelRead<type1>> waitForData(const Duration& timeout = Duration::infinity())
	{
		/* validateReadAccess is private :( */
		/*BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_VALIDATE,mChannel)*/

		Time end;
		if(!timeout.isValid() || timeout.isInfinity())
			end = Time::eternity();
		else
			end = Time::now() + timeout;

		while(!boost::this_thread::interruption_requested())
		{
			Time oldestTimeStamp = Time::now();
			Time timestamp;
			try {
				ChannelRead<typeA> a;
				ChannelRead<typeB> b;
				timestamp = mChannelA.read().getTimestamp();
				if ( timestamp < oldestTimeStamp )
					oldestTimeStamp = timestamp;
				timestamp = mChannelB.read().getTimestamp();
				if ( timestamp < oldestTimeStamp )
					oldestTimeStamp = timestamp;
				a = mChannelA.read( oldestTimeStamp, mTolerance );
				b = mChannelB.read( oldestTimeStamp, mTolerance );
				mChannelReadA = a;
				mChannelReadB = b;
				return read();
			} catch(XInvalidRead& ex) {}
			if(Time::now()>end) /* handle timeout */
				break;
			mira::sleep(mira::Duration::milliseconds(50));
		}
		return boost::make_tuple( mChannelRead0 , mChannelRead1);
	}

	bool waitForPublisher(const Duration& timeout = Duration::infinity()) const
	{
		return mChannelA.waitForPublisher( timeout ) && mChannelB.waitForPublisher( timeout );
	}

	Time mLastA;
	Time mLastB;
	Channel<A> mChannelA;
	Channel<B> mChannelB;
	ChannelRead<A> mChannelReadA;
	ChannelRead<B> mChannelReadB;
	Duration mTolerance;
	boost::function<void (ChannelRead<A>, ChannelRead<B>)> mFunction;
};

#else

///////////////////////////////////////////////////////////////////////////////

// create data0, data1, ...
#define MIRA_FW_INTERNAL_NUMBER(z, n, data) \
		BOOST_PP_COMMA_IF(n) BOOST_PP_CAT(data,n)

#define MIRA_FW_INTERNAL_INC_NUMBER(z, n, data) \
		BOOST_PP_COMMA_IF(n) BOOST_PP_CAT(data,BOOST_PP_INC(n))

// create ChannelRead<type0>, ChannelRead<type1>, ...
#define MIRA_FW_INTERNAL_CHANNEL(z, n, data) \
		BOOST_PP_COMMA_IF(n) ChannelRead<BOOST_PP_CAT(type,n)>

// create ChannelRead<type0> c0, ChannelRead<type1> c1, ...
#define MIRA_FW_INTERNAL_CHANNEL_VAR(z, n, data) \
		BOOST_PP_COMMA_IF(n) ChannelRead<BOOST_PP_CAT(type,n)> BOOST_PP_CAT(c,n)

// create mChannel0.waitForPublisher( timeout ) && mChannel1.waitForPublisher( timeout )
#define MIRA_FW_INTERNAL_WAIT_FOR_PUBLISHER(z, n, data) \
		BOOST_PP_IF(n,&&,) BOOST_PP_CAT(mChannel,n).waitForPublisher( timeout )


// create data1.isValid() && data2.isValid() ...
#define MIRA_FW_INTERNAL_ISVALID(z,n,data) \
		BOOST_PP_EXPR_IF(n,&&) BOOST_PP_CAT(data,n).isValid()

// create Time mLast0; Channel<type0> mChannel0;
// Time mLast1; Channel<type1> mChannel1; ...
#define MIRA_FW_INTERNAL_MEMBER(z, n, data) \
	Time BOOST_PP_CAT(mLast,n);\
	Channel<BOOST_PP_CAT(type,n)> BOOST_PP_CAT(mChannel,n);\
	ChannelRead<BOOST_PP_CAT(type,n)> BOOST_PP_CAT(mChannelRead,n);\

// create: ChannelRead<type0> c0;
// ChannelRead<type1> c1;
#define MIRA_FW_INTERNAL_DECL_CHANNEL(z, n, data) \
	ChannelRead<BOOST_PP_CAT(type,n)> BOOST_PP_CAT(c,n);\

// create: auto c0 = mChannel0.read( cN.getTimestamp(), mTolerance);
// auto c1 = mChannel1.read( cN.getTimestamp(), mTolerance);
#define MIRA_FW_INTERNAL_GET_CHANNEL(z, n, data) \
	BOOST_PP_CAT(c,n) = BOOST_PP_CAT(mChannel,n).read(BOOST_PP_CAT(c,data).getTimestamp(), mTolerance);\


// create: if (mLast0 == c0.getTimestamp()) return;
// if (mLast1 == c1.getTimestamp()) return; ...
#define MIRA_FW_INTERNAL_CALL_RET(z, n, data) \
	if (BOOST_PP_CAT(mLast,n) == BOOST_PP_CAT(c,n).getTimestamp()) return;\

#define MIRA_FW_INTERNAL_GET_CHANNELREAD(z, n, data) \
	BOOST_PP_CAT(mChannelRead,n) = BOOST_PP_CAT(c,n);

// create mLast0(Time::unixEpoch()), mLast1(Time::unixEpoch()), ...
#define MIRA_FW_INTERNAL_INITIALIZE(z, n, data) \
	BOOST_PP_COMMA_IF(n) BOOST_PP_CAT(mLast,n)(Time::unixEpoch())

// check if element 0 of state is not equal to element 1
#define MIRA_FW_INTERNAL_FOR_PRED(r, state) \
	BOOST_PP_NOT_EQUAL( \
			BOOST_PP_TUPLE_ELEM(3,0,state), \
			BOOST_PP_INC(BOOST_PP_TUPLE_ELEM(3,1,state)) \
	) \

// increment element 0 of state
#define MIRA_FW_INTERNAL_FOR_OP(r, state) \
	( \
			BOOST_PP_INC(BOOST_PP_TUPLE_ELEM(3,0,state)), \
			BOOST_PP_TUPLE_ELEM(3,1,state), \
			BOOST_PP_TUPLE_ELEM(3,2,state) \
	)\

// create: auto c0 = mChannel0.read( cN.getTimestamp(), mTolerance);
// auto c1 = mChannel1.read( cN.getTimestamp(), mTolerance);
#define MIRA_FW_INTERNAL_FOR_MACRO(r,state) \
	MIRA_FW_INTERNAL_GET_CHANNEL(,BOOST_PP_TUPLE_ELEM(3,0,state),BOOST_PP_TUPLE_ELEM(3,2,state))

// create: ChannelRead<type0> c0;
// ChannelRead<type1> c1;
#define MIRA_FW_INTERNAL_DECL_FOR_MACRO(r,state) \
	MIRA_FW_INTERNAL_DECL_CHANNEL(,BOOST_PP_TUPLE_ELEM(3,0,state),BOOST_PP_TUPLE_ELEM(3,2,state))

#define MIRA_FW_INTERNAL_GENCALLBACK(z, n, data) \
	/* void callback0(ChannelReady<type0> c0 ) { ... */\
	void BOOST_PP_CAT(callback,n)(ChannelRead<BOOST_PP_CAT(type,n)> BOOST_PP_CAT(c,n)) { \
		BOOST_PP_REPEAT(n,MIRA_FW_INTERNAL_DECL_CHANNEL,n)\
		BOOST_PP_FOR((BOOST_PP_INC(n),BOOST_PP_DEC(data),n), MIRA_FW_INTERNAL_FOR_PRED, MIRA_FW_INTERNAL_FOR_OP, MIRA_FW_INTERNAL_DECL_FOR_MACRO  )\
		try {\
			/* get elements of channel with id < n */ \
			BOOST_PP_REPEAT(n,MIRA_FW_INTERNAL_GET_CHANNEL,n)\
			\
			/* get elements of channel with id > n */ \
			BOOST_PP_FOR((BOOST_PP_INC(n),BOOST_PP_DEC(data),n), MIRA_FW_INTERNAL_FOR_PRED, MIRA_FW_INTERNAL_FOR_OP, MIRA_FW_INTERNAL_FOR_MACRO  )\
		}\
		catch(...) {return;} \
		/* now we can call the function with all channel elements */\
		call(BOOST_PP_REPEAT(data,MIRA_FW_INTERNAL_NUMBER,c));\
	}\

// create: mLast0 = c0.getTimestamp();
// mLast1 = c1.getTimestamp(); ...
#define MIRA_FW_INTERNAL_CALL_ASSIGN(z, n, data) \
	BOOST_PP_CAT(mLast,n) = BOOST_PP_CAT(c,n).getTimestamp();\

// create: timestamp = mChannel0.read().getTimestamp();
//		   if ( timestamp < oldestTimeStamp ) oldestTimeStamp = timestamp;
//		   timestamp = mChannelB.read().getTimestamp();
//		   if ( timestamp < oldestTimeStamp ) oldestTimeStamp = timestamp;
//         ...
#define MIRA_FW_INTERNAL_OLDESTTIMESTAMP(z, n, data) \
	timeStamp = BOOST_PP_CAT(mChannel,n).read().getTimestamp();\
	if ( timeStamp < oldestTimeStamp ) oldestTimeStamp = timeStamp;\

// create: c0 = mChannel0.read( oldestTimeStamp, mTolerance );
//		   c1 = mChannel1.read( oldestTimeStamp, mTolerance );...
#define MIRA_FW_INTERNAL_READ_FROM_OLD(z, n, data) \
	BOOST_PP_CAT(c,n) = BOOST_PP_CAT(mChannel,n).read( oldestTimeStamp, mTolerance );\

// data contains 2 tuple elements 1: number of template arguments, 2: Class name
#define MIRA_FW_INTERNAL_SUBSCRIBE(z, n, data) \
	BOOST_PP_CAT(mChannel,n) = authority.subscribe<BOOST_PP_CAT(type,n)>\
	(BOOST_PP_CAT(channelID,n), &BOOST_PP_TUPLE_ELEM(2,1,data)<\
			BOOST_PP_REPEAT(BOOST_PP_TUPLE_ELEM(2,0,data),MIRA_FW_INTERNAL_NUMBER,type)>::BOOST_PP_CAT(callback,n), this, t);\


#define MIRA_FW_INTERNAL_UNSUBSCRIBE(z, n, data) \
	authority.unsubscribe<BOOST_PP_CAT(type,n)>(BOOST_PP_CAT(mChannel,n).getID()); \


// Use this macro to generate a synchronizer class with the given name
// and the given number of channel types
#define MIRA_CHANNEL_SYNCHRONIZER( TNAME, TNUM ) \
	template < \
		BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_NUMBER,typename type) \
		> class TNAME\
	{\
	public:\
	\
	/* generate constructor which initializes last callback times*/ \
	TNAME() : BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_INITIALIZE,)\
	{} \
	\
	void unsubscribe(Authority& authority)\
	{\
		BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_UNSUBSCRIBE,(TNUM,TNAME))\
	}\
	\
	void subscribe(Authority& authority,\
			BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_NUMBER,const std::string& channelID)\
			\
			,const Duration& t = Duration::seconds(0))\
	{\
		mFunction = NULL;\
		BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_SUBSCRIBE,(TNUM,TNAME))\
		mTolerance = t;\
	}\
	\
	void subscribe(Authority& authority,\
			BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_NUMBER,const std::string& channelID)\
			\
			,boost::function<void (BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_CHANNEL,))> fn\
			,const Duration& t = Duration::seconds(0))\
	{\
		mFunction = fn;\
		BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_SUBSCRIBE,(TNUM,TNAME))\
		mTolerance = t;\
	}\
	\
	/* generate subscribe method provided for convenience*/\
	template<typename Class>\
	void subscribe(Authority& authority,\
			BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_NUMBER,const std::string& channelID)\
			\
			, void (Class::*f)( BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_CHANNEL,) )\
			, Class* obj\
			, const Duration& t = Duration::seconds(0)\
			)\
	{\
		subscribe(authority,\
				BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_NUMBER,channelID),\
				boost::bind(f, obj, BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_INC_NUMBER,_)),t);\
	}\
	/* generate callback functions for every channel*/ \
	BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_GENCALLBACK,TNUM)\
	\
	void call(BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_CHANNEL_VAR,))\
	{\
		BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_CALL_RET,)\
		BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_CALL_ASSIGN,)\
		BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_GET_CHANNELREAD,)\
		if (mFunction) mFunction( BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_NUMBER,c));\
	}\
	\
	boost::tuple<BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_CHANNEL,)> read() { \
		if (!(BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_ISVALID,mChannelRead))) \
			MIRA_THROW(XInvalidRead, "Cannot read from synchronized channel "\
					    << "' since there is no data");\
		return boost::make_tuple(BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_NUMBER,mChannelRead));\
	}\
	\
	bool isValid() const { \
		return (BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_ISVALID,mChannelRead) );\
	}\
	\
	boost::tuple<BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_CHANNEL,)> waitForData(const Duration& timeout = Duration::infinity()) \
	{\
		/* validateReadAccess is private :( */ \
		/*BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_VALIDATE,mChannel)*/\
		\
		Time end;\
		if(!timeout.isValid() || timeout.isInfinity())\
			end = Time::eternity();\
		else\
			end = Time::now() + timeout;\
		\
		while(!boost::this_thread::interruption_requested())\
		{\
			Time oldestTimeStamp = Time::now();\
			Time timeStamp;\
			try {\
				BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_DECL_CHANNEL,)\
				BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_OLDESTTIMESTAMP,)\
				BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_READ_FROM_OLD,)\
				BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_GET_CHANNELREAD,)\
				return read();\
			} catch(XInvalidRead& ex) {}\
			if(Time::now()>end) /* handle timeout */\
				break;\
			MIRA_SLEEP(50);\
		}\
		return boost::make_tuple(BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_NUMBER,mChannelRead));\
	}\
	\
	bool waitForPublisher(const Duration& timeout = Duration::infinity()) const\
	{\
		return BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_WAIT_FOR_PUBLISHER,);\
	}\
	\
	BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_MEMBER,)\
	Duration mTolerance; \
	boost::function<void (BOOST_PP_REPEAT(TNUM,MIRA_FW_INTERNAL_CHANNEL,))> mFunction; \
	};\
	\


#define MIRA_FW_GEN_CHANNEL_SYNCHRONIZER( z, n, data) \
	MIRA_CHANNEL_SYNCHRONIZER(BOOST_PP_CAT(ChannelSynchronizer,n), n)\
	/* NEXT SYNCHRONIZER */\

// generate predefined channel synchronizer for 2 channels
MIRA_CHANNEL_SYNCHRONIZER(ChannelSynchronizer, 2)

// generate predefined channel synchronizer for 3 to 9 channels
BOOST_PP_REPEAT_FROM_TO(3,10,MIRA_FW_GEN_CHANNEL_SYNCHRONIZER,)

#undef MIRA_FW_INTERNAL_NUMBER
#undef MIRA_FW_INTERNAL_INC_NUMBER
#undef MIRA_FW_INTERNAL_CHANNEL
#undef MIRA_FW_INTERNAL_CHANNEL_VAR
#undef MIRA_FW_INTERNAL_WAIT_FOR_PUBLISHER
#undef MIRA_FW_INTERNAL_ISVALID
#undef MIRA_FW_INTERNAL_MEMBER
#undef MIRA_FW_INTERNAL_DECL_CHANNEL
#undef MIRA_FW_INTERNAL_GET_CHANNEL
#undef MIRA_FW_INTERNAL_CALL_RET
#undef MIRA_FW_INTERNAL_GET_CHANNELREAD
#undef MIRA_FW_INTERNAL_INITIALIZE
#undef MIRA_FW_INTERNAL_FOR_PRED
#undef MIRA_FW_INTERNAL_FOR_OP
#undef MIRA_FW_INTERNAL_FOR_MACRO
#undef MIRA_FW_INTERNAL_FOR_DECL_MACRO
#undef MIRA_FW_INTERNAL_GENCALLBACK
#undef MIRA_FW_INTERNAL_CALL_ASSIGN
#undef MIRA_FW_INTERNAL_OLDESTTIMESTAMP
#undef MIRA_FW_INTERNAL_READ_FROM_OLD
#undef MIRA_FW_INTERNAL_SUBSCRIBE
#undef MIRA_FW_INTERNAL_UNSUBSCRIBE
#undef MIRA_CHANNEL_SYNCHRONIZER
#undef MIRA_FW_GEN_CHANNEL_SYNCHRONIZER

#endif

///////////////////////////////////////////////////////////////////////////////

}

#endif
