/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file mirafind.C
 *    .
 *
 * @author Erik Einhorn, Tim Langner
 * @date   2011/12/31
 */

#include <iostream>

#include <utils/ProgramOptions.h>
#include <utils/PackageFinder.h>
#include <utils/PathFinder.h>

using namespace mira;

void printUsage()
{
	std::cout <<
			"mirafind [options] fileOrPackage\n\n"
			"Searches for the specified file or package(s) within the paths specified by the\n"
			"MIRA_PATH environment variable. If the file/package was found, the process\n"
			"exits with exitcode 0 and prints the full path on stdout.\n"
			"If no file/package was found, the process exists with exitcode -1 and\n"
			"stdout stays empty.\n"
			"\n"
			"Options:\n"
			"  -p: Find a package (options d, r and f are ignored), use empty fileOrPackage to find all packages\n"
			"  -d: Show the directory only and cut off the file name\n"
			"  -r: Search recursively all sub-directories of the paths given in MIRA_PATH\n"
			"  -f: Only gives the first file that is found instead of all (if more than one exist)\n"
			"  -n: Do not end the output by a newline (useful whenever the output is used in scripts)\n"
			<< std::endl;
}

int main(int argc, char** argv)
{
	if(argc<2) {
		printUsage();
		return -1;
	}

	ProgramOptions options(argc, argv);

	options.getDescriptions().add_options()
		("file", boost::program_options::value<std::string>(), "The file to find")
		("package,p", "search for package")
		("directory,d", "Output directory only (without the file's name)")
		("recursive,r", "search recursively")
		("first-only,f", "return first file only")
		("no-newline,n", "do not end the output by newlines");

	options.getPositionalDescriptions().add("file", -1);

	ProgramOptions::VariableMap vmap = options.getOptions();
	Path file;
	bool directory = vmap.count("directory") > 0;
	bool recursive = vmap.count("recursive") > 0;
	bool noNewline = vmap.count("no-newline") > 0;
	bool firstOnly = vmap.count("first-only") > 0;
	bool package = vmap.count("package") > 0;
	if(!vmap.count("file") && !package) {
		printUsage();
		return -1;
	}

	if (vmap.count("file"))
		file = vmap["file"].as<std::string>();

	try {
		if (package) {
			if (!file.empty())
			{
				std::cout << findPackage(file.string()).string();
				if(!noNewline)
					std::cout << std::endl;
			}
			else
			{
				// get a fresh package list (reindex cache)
				std::map<std::string, Path> packages = findPackages(Duration::seconds(0));
				if(packages.empty())
					throw -1;
				auto it = packages.begin();
				bool first = true;
				for (; it != packages.end(); ++it)
				{
					if (!first && noNewline)
						std::cout << " ";
					std::cout << it->first << " " << it->second.string();
					if(!noNewline)
						std::cout << std::endl;
					first = false;
				}
			}
		}
		else
		if(recursive) {
			PathVector v = findProjectFiles(file, true);
			if(v.empty())
				throw -1;

			for(std::size_t i=0; i<v.size(); ++i)
			{
				const Path& path = v[i];
				if(directory) std::cout << path.parent_path().string();
				else          std::cout << path.string();

				if(!noNewline)
					std::cout << std::endl;
				else if(i!=(v.size()-1) && !firstOnly) // separate by spaces
					std::cout << " ";

				if(firstOnly)
					break;
			}

		} else {
			Path path = findProjectFile(file);

			if(directory) std::cout << path.parent_path().string();
			else          std::cout << path.string();

			if(!noNewline)
				std::cout << std::endl;
		}

	} catch(...) {
		std::cerr << "No such file: " << file << std::endl;
		return -1;
	}

	return 0;
}
