/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ToStringTest.C
 *    Test cases for ToString.h.
 *
 * @author Erik Einhorn
 * @date   2010/10/15
 */

#include <boost/test/unit_test.hpp>

#include <utils/ToString.h>

using namespace mira;
using namespace std;

#define toStringTester(T, v, s) {                                     \
	string toS = toString<T>(v);                                      \
	cout << "value: " << v << ", string: " << s << " - toString(v): " \
	     << toS << endl;                                              \
	BOOST_CHECK_EQUAL(toS,  s);                                       \
}

#define fromStringTester(T, v, s) {                                     \
	T fromS = fromString<T>(s);                                         \
	cout << "value: " << v << ", string: " << s << " - fromString(s): " \
	     << fromS << endl;                                              \
	BOOST_CHECK_EQUAL(fromS, v);                                        \
}

#define Tester(T, v, s)         \
		toStringTester(T,v,s)   \
		fromStringTester(T,v,s)

BOOST_AUTO_TEST_CASE( ToStringTest )
{
	float       tZeroF = 0.0f;
	double      tZeroD = 0.0;
	long double tZeroL = 0.0;

	Tester(int, 10, "10");
	Tester(int, -5, "-5");

	Tester(char, 10, "10");
	Tester(char, -5, "-5");

	Tester(float, 1.25, "1.25");    // values chosen so that we can check for float equality
	Tester(float, -99.75, "-99.75");// (sums of powers of 2, exact binary representation)
	Tester(float, log(tZeroF), "-inf");
	Tester(float, 1.0f/tZeroF, "inf");
	toStringTester(float, 0.0f/tZeroF, "nan");
	BOOST_CHECK(boost::math::isnan(fromString<float>("nan")));

	Tester(double, 1.25, "1.25"); 
	Tester(double, -99.75, "-99.75");
	Tester(double, log(tZeroF), "-inf");
	Tester(double, 1.0/tZeroD, "inf");
	toStringTester(double, 0.0/tZeroD, "nan");
	BOOST_CHECK(boost::math::isnan(fromString<double>("nan")));

	Tester(long double, 1.25, "1.25");
	Tester(long double, -99.75, "-99.75");
	Tester(long double, log(tZeroF), "-inf");
	Tester(long double, 1.0/tZeroL, "inf");
	toStringTester(long double, 0.0/tZeroL, "nan");
	BOOST_CHECK(boost::math::isnan(fromString<long double>("nan")));


	BOOST_CHECK_THROW(fromString<float>("unknown"), XIO); // invalid conversion
	BOOST_CHECK_THROW(fromString<char>("128"), XIO);      // overflow
	BOOST_CHECK_THROW(fromString<unsigned char>("-1"), XIO);  // underflow
	// The following test does not work on all systems.
	// We assume that the implementation of std::stringstream differs
	// (on some systems an error flag is set, on others its not)
	//BOOST_CHECK_THROW(fromString<unsigned int>("-1"), XIO);   // invalid conversion
	BOOST_CHECK_THROW(fromString<short>("99999"), XIO);   // invalid conversion
}
