/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TransformTest.C
 *    testcases for transform framework.
 *
 * @author Erik Einhorn, Michael Volkhardt
 * @date   2011/09/06
 */

#include <iostream>

#include <platform/Platform.h>

#if defined(MIRA_GNUC_VERSION)
# if MIRA_GNUC_VERSION >= 40600
#  pragma GCC diagnostic push
#  pragma GCC diagnostic ignored "-Wdeprecated-declarations"
# endif
#endif

#include <Eigen/Eigen>

#if defined(MIRA_GNUC_VERSION)
# if MIRA_GNUC_VERSION >= 40600
#  pragma GCC diagnostic pop
# endif
#endif

#include <math/Angle.h>

#include <boost/test/unit_test.hpp>

#include <transform/RigidTransform.h>
#include <transform/TransformCast.h>

#include <serialization/Print.h>

using namespace mira;
using namespace std;

/**
 * Checks if the conversation EulerAngles->Quaternion->Matrix->EulerAngles
 * preserves the angles.
 */
void checkConvertAnglesQuaternionMatrixAngles(float yaw, float pitch, float roll)
{
	// EulerAngles->Quaternion
	Eigen::Quaternion<float> q(Eigen::AngleAxis<float>(yaw,   Eigen::Matrix<float,3,1>::UnitZ()) *
	                           Eigen::AngleAxis<float>(pitch, Eigen::Matrix<float,3,1>::UnitY()) *
	                           Eigen::AngleAxis<float>(roll,  Eigen::Matrix<float,3,1>::UnitX()));

	// Quaternion->Matrix->EulerAngles
	//Eigen::Matrix<float,3,1> euler =q.toRotationMatrix().eulerAngles(2, 1, 0);
	Eigen::Matrix<float,3,1> euler =eulerAngles(q.toRotationMatrix(), 2, 1, 0);

	BOOST_CHECK_CLOSE(euler(0), yaw, 0.01f);
	BOOST_CHECK_CLOSE(euler(1), pitch, 0.01f);
	BOOST_CHECK_CLOSE(euler(2), roll, 0.01f);
}

BOOST_AUTO_TEST_CASE( RotationConvertPreTest )
{
	checkConvertAnglesQuaternionMatrixAngles(0.0f,0.0f,0.0f);
	checkConvertAnglesQuaternionMatrixAngles(0.1f,0.0f,0.0f);
	checkConvertAnglesQuaternionMatrixAngles(0.0f,0.1f,0.0f);
	checkConvertAnglesQuaternionMatrixAngles(0.0f,0.0f,0.1f);
	checkConvertAnglesQuaternionMatrixAngles(-0.1f,0.0f,0.0f);
	checkConvertAnglesQuaternionMatrixAngles(0.0f,-0.1f,0.0f);
	checkConvertAnglesQuaternionMatrixAngles(0.0f,0.0f,-0.1f);
}

BOOST_AUTO_TEST_CASE( ConversationTest ) {

	// RigidTransform 2D -> 3D
	{
		RigidTransform2f t2f(1.0f, 2.0f, 3.0f);
		cout << "t2f:" << print(t2f) << endl;

		RigidTransform3f t3f = transform_cast<RigidTransform3f>(t2f);
		cout << "t3f" << print(t3f) << endl;

		BOOST_CHECK_CLOSE(t3f.x(), t2f.x(), 0.01f);
		BOOST_CHECK_CLOSE(t3f.y(), t2f.y(), 0.01f);
		BOOST_CHECK_CLOSE(t3f.yaw(), t2f.phi(), 0.01f);
		BOOST_CHECK_SMALL(t3f.z(), 0.01f);
	}

	// RigidTransform 3D -> 2D
	{
		RigidTransform3f::TranslationType t(5, 6, 7);
		RigidTransform3f::RotationType r(
				Eigen::AngleAxis<float>(2,
						Eigen::Matrix<float, 3, 1>::UnitZ()));

		RigidTransform3f t3f(t, r);
		cout << "t3f:" << print(t3f) << endl;

		RigidTransform2f t2f = transform_cast<RigidTransform2f>(t3f);
		cout << "t2f:" << print(t2f) << endl;

		BOOST_CHECK_CLOSE(t3f.x(), t2f.x(), 0.01f);
		BOOST_CHECK_CLOSE(t3f.y(), t2f.y(), 0.01f);
		BOOST_CHECK_CLOSE(t3f.yaw(), t2f.phi(), 0.01f);
	}

	// RigidTransform 32 -> 3D -> 2D
	{
		RigidTransform2f t2f(1.0f, 2.0f, -0.1f);
		cout << "t2f:" << print(t2f) << endl;

		RigidTransform3f t3f = transform_cast<RigidTransform3f>(t2f);
		cout << "t3f" << print(t3f) << endl;

		RigidTransform2f t2bf = transform_cast<RigidTransform2f>(t3f);

		BOOST_CHECK_CLOSE(t2f.x(), t2bf.x(), 0.01f);
		BOOST_CHECK_CLOSE(t2f.y(), t2bf.y(), 0.01f);
		BOOST_CHECK_CLOSE(t2f.phi(), t2bf.phi(), 0.01f);

	}

	// RigidTransformCov 2D -> 3D
	{
		Eigen::Matrix3f cov = Eigen::Matrix3f::Zero();
		cov << 1, 2, 3, 4, 5, 6, 7, 8, 9;

		RigidTransformCov2f t2f(1.0f, 2.0f, 3.0f, cov);
		cout << "t2f:" << print(t2f) << endl;

		RigidTransformCov3f t3f = transform_cast<RigidTransformCov3f>(t2f);
		cout << "t3f:" << print(t3f) << endl;

		BOOST_CHECK_CLOSE(t3f.x(), t2f.x(), 0.01f);
		BOOST_CHECK_CLOSE(t3f.y(), t2f.y(), 0.01f);
		BOOST_CHECK_CLOSE(t3f.yaw(), t2f.phi(), 0.01f);
		BOOST_CHECK_SMALL(t3f.z(), 0.01f);

		RigidTransformCov3f::YawPitchRollCovMatrixType t3fEulerCov =
		        quaternionCovToYawPitchRollCov(t3f.cov, t3f.r);

		BOOST_CHECK_CLOSE(t3fEulerCov(0,0), t2f.cov(0,0), 0.01f);
		BOOST_CHECK_CLOSE(t3fEulerCov(1,1), t2f.cov(1,1), 0.01f);
		BOOST_CHECK_CLOSE(t3fEulerCov(3,3), t2f.cov(2,2), 0.01f);
		BOOST_CHECK_CLOSE(t3fEulerCov(0,1), t2f.cov(0,1), 0.01f);
		BOOST_CHECK_CLOSE(t3fEulerCov(1,0), t2f.cov(1,0), 0.01f);
		BOOST_CHECK_CLOSE(t3fEulerCov(0,3), t2f.cov(0,2), 0.01f);
		BOOST_CHECK_CLOSE(t3fEulerCov(3,0), t2f.cov(2,0), 0.01f);
		BOOST_CHECK_CLOSE(t3fEulerCov(1,3), t2f.cov(1,2), 0.01f);
		BOOST_CHECK_CLOSE(t3fEulerCov(3,1), t2f.cov(2,1), 0.01f);
	}

	// RigidTransformCov 3D -> 2D
	{
		RigidTransform3f::TranslationType t(5, 6, 7);
		RigidTransform3f::RotationType r(
				Eigen::AngleAxis<float>(2,
						Eigen::Matrix<float, 3, 1>::UnitZ()));

		Eigen::Matrix<float,6,6> cov = Eigen::Matrix<float,6,6>::Identity() * 3;
		RigidTransformCov3f t3f(t, r, cov);
		cout << "t3f:" << print(t3f) << endl;

		RigidTransformCov2f t2f = transform_cast<RigidTransformCov2f>(t3f);
		cout << "t2f:" << print(t2f) << endl;

		BOOST_CHECK_CLOSE(t3f.x(), t2f.x(), 0.01f);
		BOOST_CHECK_CLOSE(t3f.y(), t2f.y(), 0.01f);
		BOOST_CHECK_CLOSE(t3f.yaw(), t2f.phi(), 0.01f);
	}

	// RigidTransform 2D -> RigidTransformCov 2D
	{
		RigidTransform2f t2f(1.0f, 2.0f, 3.0f);
		cout << "t2f:" << print(t2f) << endl;

		RigidTransformCov2f tc2f = transform_cast<RigidTransformCov2f>(t2f);
		cout << "tc2f:" << print(tc2f) << endl;

		BOOST_CHECK_CLOSE(tc2f.x(), t2f.x(), 0.01f);
		BOOST_CHECK_CLOSE(tc2f.y(), t2f.y(), 0.01f);
		BOOST_CHECK_CLOSE(tc2f.phi(), t2f.phi(), 0.01f);
	}

	// RigidTransformCov 2D -> RigidTransform 2D
	{
		RigidTransformCov2f tc2f(1.0f, 2.0f, 3.0f, Eigen::Matrix3f::Identity());
		cout << "tc2f:" << print(tc2f) << endl;

		RigidTransform2f t2f = transform_cast<RigidTransform2f>(tc2f);
		cout << "t2f:" << print(t2f) << endl;

		BOOST_CHECK_CLOSE(tc2f.x(), t2f.x(), 0.01f);
		BOOST_CHECK_CLOSE(tc2f.y(), t2f.y(), 0.01f);
		BOOST_CHECK_CLOSE(tc2f.phi(), t2f.phi(), 0.01f);
	}

	// RigidTransform 2D -> RigidTransformCov 3D
	{
		RigidTransform2f t2f(1.0f, 2.0f, 3.0f);
		cout << "t2f:" << print(t2f) << endl;

		RigidTransformCov3f t3f = transform_cast<RigidTransformCov3f>(t2f);
		cout << "t3f:" << print(t3f) << endl;

		BOOST_CHECK_CLOSE(t3f.x(), t2f.x(), 0.01f);
		BOOST_CHECK_CLOSE(t3f.y(), t2f.y(), 0.01f);
		BOOST_CHECK_CLOSE(t3f.yaw(), t2f.phi(), 0.01f);
		BOOST_CHECK_SMALL(t3f.z(), 0.01f);
	}

	// RigidTransform 3D -> RigidTransformCov 2D
	{
		RigidTransform3f::TranslationType t(5, 6, 7);
		RigidTransform3f::RotationType r(
				Eigen::AngleAxis<float>(2,
						Eigen::Matrix<float, 3, 1>::UnitZ()));

		RigidTransform3f t3f(t, r);
		cout << "t3f:" << print(t3f) << endl;

		RigidTransformCov2f t2f = transform_cast<RigidTransformCov2f>(t3f);
		cout << "t2f:" << print(t2f) << endl;

		BOOST_CHECK_CLOSE(t3f.x(), t2f.x(), 0.01f);
		BOOST_CHECK_CLOSE(t3f.y(), t2f.y(), 0.01f);
		BOOST_CHECK_CLOSE(t3f.yaw(), t2f.phi(), 0.01f);
	}

	// RigidTransformCov 2D -> RigidTransform 3D
	{
		Eigen::Matrix3f cov = Eigen::Matrix3f::Zero();
		cov << 1, 2, 3, 4, 5, 6, 7, 8, 9;

		RigidTransformCov2f t2f(1.0f, 2.0f, 3.0f, cov);
		cout << "t2f:" << print(t2f) << endl;

		RigidTransform3f t3f = transform_cast<RigidTransform3f>(t2f);
		cout << "t3f:" << print(t3f) << endl;

		BOOST_CHECK_CLOSE(t3f.x(), t2f.x(), 0.01f);
		BOOST_CHECK_CLOSE(t3f.y(), t2f.y(), 0.01f);
		BOOST_CHECK_CLOSE(t3f.yaw(), t2f.phi(), 0.01f);
		BOOST_CHECK_SMALL(t3f.z(), 0.01f);
	}

	// RigidTransformCov 3D -> RigidTransform 2D
	{
		RigidTransform3f::TranslationType t(5, 6, 7);
		RigidTransform3f::RotationType r(
				Eigen::AngleAxis<float>(2,
						Eigen::Matrix<float, 3, 1>::UnitZ()));

		Eigen::Matrix<float,6,6> cov = Eigen::Matrix<float,6,6>::Identity() * 3;
		RigidTransformCov3f t3f(t, r, cov);
		cout << "t3f:" << print(t3f) << endl;

		RigidTransform2f t2f = transform_cast<RigidTransform2f>(t3f);
		cout << "t2f:" << print(t2f) << endl;

		BOOST_CHECK_CLOSE(t3f.x(), t2f.x(), 0.01f);
		BOOST_CHECK_CLOSE(t3f.y(), t2f.y(), 0.01f);
		BOOST_CHECK_CLOSE(t3f.yaw(), t2f.phi(), 0.01f);
	}
}

BOOST_AUTO_TEST_CASE( RigidTransformTypeCast)
{
	RigidTransform<float,2> f2(1.0f, 2.0f, 3.0f);
	RigidTransform<double,2> d2 = f2.cast<double>();

	BOOST_CHECK_CLOSE(d2.x(),1.0,0.001);
	BOOST_CHECK_CLOSE(d2.y(),2.0,0.001);
	BOOST_CHECK_CLOSE(d2.phi(),3.0,0.001);

	RigidTransform<float,3> f3(1.0f, 2.0f, 3.0f, 0.1f, 0.2f, 0.3f);
	RigidTransform<double,3> d3 = f3.cast<double>();

	BOOST_CHECK_CLOSE(d3.x(),1.0,0.001);
	BOOST_CHECK_CLOSE(d3.y(),2.0,0.001);
	BOOST_CHECK_CLOSE(d3.z(),3.0,0.001);
	BOOST_CHECK_CLOSE(d3.yaw(),0.1,0.001);
	BOOST_CHECK_CLOSE(d3.pitch(),0.2,0.001);
	BOOST_CHECK_CLOSE(d3.roll(),0.3,0.001);

	Eigen::Matrix3f cov2;
	cov2 << 2.0f, 0.0f, 0.0f,
	        0.0f, 3.0f, 0.0f,
	        0.0f, 0.0f, 4.0f;
	RigidTransformCov<float,2> fc2(1.0f, 2.0f, 3.0f, cov2);
	RigidTransformCov<double,2> dc2 = fc2.cast<double>();

	BOOST_CHECK_CLOSE(dc2.x(),1.0,0.001);
	BOOST_CHECK_CLOSE(dc2.y(),2.0,0.001);
	BOOST_CHECK_CLOSE(dc2.phi(),3.0,0.001);
	BOOST_CHECK_CLOSE(dc2.cov(0,0),2.0,0.001);
	BOOST_CHECK_CLOSE(dc2.cov(1,1),3.0,0.001);
	BOOST_CHECK_CLOSE(dc2.cov(2,2),4.0,0.001);

	Eigen::Matrix<float,6,6> cov3;
	cov3 << 2.0f, 0.0f, 0.0f, 0.0f, 0.0f, 0.0f,
	        0.0f, 3.0f, 0.0f, 0.0f, 0.0f, 0.0f,
	        0.0f, 0.0f, 4.0f, 0.0f, 0.0f, 0.0f,
	        0.0f, 0.0f, 0.0f, 0.3f, 0.0f, 0.0f,
	        0.0f, 0.0f, 0.0f, 0.0f, 0.4f, 0.0f,
	        0.0f, 0.0f, 0.0f, 0.0f, 0.0f, 0.5f;

	RigidTransformCov<float,3> fc3(1.0f, 2.0f, 3.0f, 0.1f, 0.2f, 0.3f, cov3);;
	RigidTransformCov<double,3> dc3 = fc3.cast<double>();

	BOOST_CHECK_CLOSE(dc3.x(),1.0,0.001);
	BOOST_CHECK_CLOSE(dc3.y(),2.0,0.001);
	BOOST_CHECK_CLOSE(dc3.z(),3.0,0.001);
	BOOST_CHECK_CLOSE(dc3.yaw(),0.1,0.001);
	BOOST_CHECK_CLOSE(dc3.pitch(),0.2,0.001);
	BOOST_CHECK_CLOSE(dc3.roll(),0.3,0.001);

	BOOST_CHECK_CLOSE(dc3.cov(0,0),2.0,0.001);
	BOOST_CHECK_CLOSE(dc3.cov(1,1),3.0,0.001);
	BOOST_CHECK_CLOSE(dc3.cov(2,2),4.0,0.001);

}



