/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file BinaryStreamBenchmark.C
 *    Benchmarks for all flavors of binary streams.
 *
 * @author Erik Einhorn
 */

#include <sstream>
#include <fstream>

#include <utils/Time.h>
#include <stream/BinaryStream.h>
#include <stream/BufferStream.h>

using namespace std;
using namespace mira;

//const int size = 50000000;
const int size = 100000;

template <typename BinaryStream>
void fillData(BinaryStream& s)
{
	for(int i=0; i<size;++i)
		s << i;
}


////////////////////////////////////////////////////////////////////////////
// BinarySTLStream based on stringstream
Duration benchmarkStringStream()
{
	stringstream s;
	BinaryStlOstream bos(s);

	Time start = Time::now();
	fillData(bos);
	const string& d = s.str();
	Duration duration = Time::now()-start;
	cout << d.size() << endl;
	return duration;
}

////////////////////////////////////////////////////////////////////////////
// BinarySTLStream based on BufferStream
Duration benchmarkBufferStream(bool reserve)
{
	BufferStream::container_type buf;
	if(reserve)
		buf.reserve(sizeof(int)*size);
	BufferStream s(buf);
	BinaryStlOstream bos(s);

	Time start = Time::now();
	fillData(bos);
	Duration duration = Time::now()-start;
	cout << buf.size() << endl;
	return duration;
}

////////////////////////////////////////////////////////////////////////////
// BinaryBufferStream
Duration benchmarkFastBufferStream(bool reserve)
{
	BinaryBufferOstream::buffer_type buf;
	if(reserve)
		buf.reserve(sizeof(int)*size);
	BinaryBufferOstream bos(&buf);

	Time start = Time::now();
	fillData(bos);
	Duration duration = Time::now()-start;
	cout << buf.size() << endl;
	return duration;
}


float speedup(const Duration& x, const Duration& ref)
{
	int p = 100*ref.totalMicroseconds() / x.totalMicroseconds();
	return (float)p / 100.0f;
}


int main()
{
	Duration timeStringStream = benchmarkStringStream();

	Duration timeBufferStreamNoReserve = benchmarkBufferStream(false);
	Duration timeBufferStreamReserve = benchmarkBufferStream(true);

	Duration timeFastBufferStreamNoReserve = benchmarkFastBufferStream(false);
	Duration timeFastBufferStreamReserve = benchmarkFastBufferStream(true);

	Duration ref = timeStringStream;

	cout << "std::stringstream:                 " <<
			timeStringStream << ", speedup: " << speedup(timeStringStream,ref) << "x (compared to stringstream)" << endl;

	cout << "mira::BufferStream:                 " <<
			timeBufferStreamNoReserve << ", speedup: " << speedup(timeBufferStreamNoReserve,ref) << "x" << endl;
	cout << "mira::BufferStream (with reserve):  " <<
			timeBufferStreamReserve << ", speedup: " << speedup(timeBufferStreamReserve,ref) << "x" << endl;

	cout << "BinaryBufferStream:                " <<
			timeFastBufferStreamNoReserve << ", speedup: " << speedup(timeFastBufferStreamNoReserve,ref) << "x" << endl;
	cout << "BinaryBufferStream (with reserve): " <<
			timeFastBufferStreamReserve << ", speedup: " << speedup(timeFastBufferStreamReserve,ref) << "x" << endl;

	return 0;
}
