/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SerializationContainerTest.C
 *    Test for serialization framework.
 *
 * @author Erik Einhorn
 */

#include <boost/test/unit_test.hpp>

#include "CommonTest.h"

#include <serialization/adapters/std/vector>
#include <serialization/adapters/std/list>
#include <serialization/adapters/std/deque>
#include <serialization/adapters/std/pair>
#include <serialization/adapters/std/set>
#include <serialization/adapters/std/map>

using namespace std;
using namespace mira;

template<typename Container>
class ClassSequence {
public:

	typedef typename Container::value_type type;

	ClassSequence() {}

	ClassSequence(bool) {
		container.push_back(boost::lexical_cast<type>(1));
		container.push_back(boost::lexical_cast<type>(2));
		container.push_back(boost::lexical_cast<type>(3));
		container.push_back(boost::lexical_cast<type>(4));
		container.push_back(boost::lexical_cast<type>(5));
	}

	void check() {
		BOOST_CHECK_EQUAL(container.size(), 5);

		typename Container::iterator it = container.begin();
		for (size_t i = 0; i < container.size(); ++i, ++it) {
			BOOST_CHECK_EQUAL(*it, boost::lexical_cast<type>(i + 1));
		}
	}

	template<typename Reflector>
	void reflect(Reflector& r) {
		r.member("container", container, "");
	}

public:

	Container container;
};

//////////////////////////////////////////////////////////////////////////////

template<typename Container>
class ClassSet {
public:

	typedef typename Container::value_type type;

	ClassSet() {}
	ClassSet(bool) {
		container.insert(boost::lexical_cast<type>(1));
		container.insert(boost::lexical_cast<type>(2));
		container.insert(boost::lexical_cast<type>(3));
		container.insert(boost::lexical_cast<type>(4));
		container.insert(boost::lexical_cast<type>(5));
	}

	void check() {
		BOOST_CHECK_EQUAL(container.size(), 5);

		typename Container::iterator it = container.begin();
		for (size_t i = 0; i < container.size(); ++i, ++it) {
			BOOST_CHECK_EQUAL(*it, boost::lexical_cast<type>(i + 1));
		}
	}

	template<typename Reflector>
	void reflect(Reflector& r) {
		r.member("container", container, "");
	}

public:

	Container container;
};

//////////////////////////////////////////////////////////////////////////////

template<typename Container>
class ClassSequenceWithPtr {
public:

	typedef typename Container::value_type type;

	ClassSequenceWithPtr() :
		ptr(NULL) {
	}
	ClassSequenceWithPtr(bool) {
		container.push_back(boost::lexical_cast<type>(1));
		container.push_back(boost::lexical_cast<type>(2));
		container.push_back(boost::lexical_cast<type>(3));
		container.push_back(boost::lexical_cast<type>(4));
		container.push_back(boost::lexical_cast<type>(5));

		ptr = &container.back();
	}

	void check() {
		BOOST_CHECK_EQUAL(container.size(), 5);

		typename Container::iterator it = container.begin();
		for (size_t i = 0; i < container.size(); ++i, ++it) {
			BOOST_CHECK_EQUAL(*it, boost::lexical_cast<type>(i + 1));
		}

		BOOST_CHECK_EQUAL(ptr, &container.back());
	}

	template<typename Reflector>
	void reflect(Reflector& r) {
		r.member("container", container, "");
		r.member("ptr", ptr, "");
	}

public:

	Container container;
	type* ptr;
};

//////////////////////////////////////////////////////////////////////////////


template<typename Container>
class ClassSetWithPtr {
public:

	typedef typename Container::value_type type;

	ClassSetWithPtr() :
		ptr(NULL) {
	}
	ClassSetWithPtr(bool) {
		container.insert(boost::lexical_cast<type>(1));
		container.insert(boost::lexical_cast<type>(2));
		container.insert(boost::lexical_cast<type>(3));
		container.insert(boost::lexical_cast<type>(4));
		container.insert(boost::lexical_cast<type>(5));

		ptr = const_cast<type*>(&(*container.begin()));
	}

	void check() {
		BOOST_CHECK_EQUAL(container.size(), 5);

		typename Container::iterator it = container.begin();
		for (size_t i = 0; i < container.size(); ++i, ++it) {
			BOOST_CHECK_EQUAL(*it, boost::lexical_cast<type>(i + 1));
		}

		BOOST_CHECK_EQUAL(ptr, &(*container.begin()));
	}

	template<typename Reflector>
	void reflect(Reflector& r) {
		r.member("container", container, "");
		r.member("ptr", ptr, "");
	}

public:

	Container container;
	type* ptr;
};

//////////////////////////////////////////////////////////////////////////////

template<typename Container>
class ClassMap {
public:

	typedef typename Container::value_type type;
	typedef typename Container::key_type key;
	typedef typename Container::mapped_type value;

	ClassMap() :
		ptr(NULL) {
	}
	ClassMap(bool) {
		container[boost::lexical_cast<key>(1)] = boost::lexical_cast<value>(1) + boost::lexical_cast<value>(1);
		container[boost::lexical_cast<key>(2)] = boost::lexical_cast<value>(2) + boost::lexical_cast<value>(2);
		container[boost::lexical_cast<key>(3)] = boost::lexical_cast<value>(3) + boost::lexical_cast<value>(3);
		container[boost::lexical_cast<key>(4)] = boost::lexical_cast<value>(4) + boost::lexical_cast<value>(4);
		container[boost::lexical_cast<key>(5)] = boost::lexical_cast<value>(5) + boost::lexical_cast<value>(5);

		ptr = &container[boost::lexical_cast<key>(3)];
	}

	void check() {
		BOOST_CHECK_EQUAL(container.size(), 5);

		typename Container::iterator it = container.begin();
		for (size_t i = 0; i < container.size(); ++i, ++it) {
			BOOST_CHECK_EQUAL(it->first, boost::lexical_cast<key>(i + 1));
			BOOST_CHECK_EQUAL(it->second, boost::lexical_cast<value>(i + 1) + boost::lexical_cast<value>(i + 1));
		}

		BOOST_CHECK_EQUAL(ptr, &container[boost::lexical_cast<key>(3)]);
	}

	template<typename Reflector>
	void reflect(Reflector& r) {
		r.member("container", container, "");
		r.member("ptr", ptr, "");
	}

public:

	Container container;
	value* ptr;
};

//////////////////////////////////////////////////////////////////////////////

class ClassArray {
public:
	ClassArray() {
	}
	ClassArray(bool) {
		array[0] = 1;
		array[1] = 2;
		array[2] = 3;
		array[3] = 4;
		array[4] = 5;

	}

	void check() {
		 for(size_t i=0; i<5; ++i)
			 BOOST_CHECK_EQUAL(array[i], (i+1));
	}

	template<typename Reflector>
	void reflect(Reflector& r) {
		r.member("array", array, "");
	}

public:
	int array[5];
};

//////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE( TestClassSequenceVector )
{
	testAll<ClassSequence<vector<int> > >("Containers",1);
}

BOOST_AUTO_TEST_CASE( TestClassSequenceVector2 )
{
	testAll<ClassSequenceWithPtr<vector<string> > >("Containers",2);
}

BOOST_AUTO_TEST_CASE( TestClassSequenceList )
{
	testAll<ClassSequence<list<int> > >("Containers",3);
}

BOOST_AUTO_TEST_CASE( TestClassSequenceList2 )
{
	testAll<ClassSequenceWithPtr<list<string> > >("Containers",4);
}

BOOST_AUTO_TEST_CASE( TestClassSequenceDeque )
{
	testAll<ClassSequence<deque<int> > >("Containers",5);
}

BOOST_AUTO_TEST_CASE( TestClassSequenceDeque2 )
{
	testAll<ClassSequenceWithPtr<deque<string> > >("Containers",6);
}

BOOST_AUTO_TEST_CASE( TestClassSet )
{
	testAll<ClassSet<set<int> > >("Containers",7);
}

//BOOST_AUTO_TEST_CASE( TestClassSet2 )
//{
//	BOOST_CHECK_THROW(testXml<ClassSetWithPtr<set<string> > >("Containers",8),
//	                  mira::XIO);
//}

BOOST_AUTO_TEST_CASE( TestClassMultiset )
{
	testAll<ClassSet<multiset<int> > >("Containers",9);
}

//BOOST_AUTO_TEST_CASE( TestClassMultiset2 )
//{
//	BOOST_CHECK_THROW(testXml<ClassSetWithPtr<multiset<string> > >("Containers",10),
//	                  mira::XIO);
//}

BOOST_AUTO_TEST_CASE( TestClassMap )
{
	testAll<ClassMap<map<int, string> > >("Containers",11);
}

BOOST_AUTO_TEST_CASE( TestClassMap2)
{
	testAll<ClassMap<map<string, string> > >("Containers",12);
}

BOOST_AUTO_TEST_CASE( TestArray)
{
	testAll<ClassArray >("Containers",13);
}

//////////////////////////////////////////////////////////////////////////////

class ClassInvalidArraySize {
public:
	ClassInvalidArraySize() {
	}
	ClassInvalidArraySize(bool) {
	}

	void check() {
	}

	template<typename Reflector>
	void reflect(Reflector& r) {
		r.member("array", array, "");
	}

public:
	int array[3];
};

BOOST_AUTO_TEST_CASE( TestInvalidArraySize )
{
	string xml =
	"<Root>"
	"	<Test>"
	"      <array>"
	"	      <item>123</item>"
	"	      <item>345</item>"
	"	   </array>"
	"   </Test>"
	"</Root>";

	BOOST_CHECK_THROW(testXmlFromString<ClassInvalidArraySize>(xml), mira::XIO);

}

//////////////////////////////////////////////////////////////////////////////

// disable multi_array test for windows (see ticket #130)
#ifndef MIRA_WINDOWS
#include <serialization/adapters/boost/multi_array.hpp>

class ClassMultiArray {
public:
	ClassMultiArray() {
	}
	ClassMultiArray(bool) {
		array.resize(boost::extents[3][3][3]);
		uint32 i = 0;
		for (auto it = array.origin(); it!= array.origin() + array.num_elements(); ++it,++i) {
			*it=i;
		}

	}

	void check() {

	}

	template<typename Reflector>
	void reflect(Reflector& r) {
		r.member("array", array, "");
	}

public:
	boost::multi_array<int, 3> array;

};

BOOST_AUTO_TEST_CASE( TestMultiArray)
{
	testAll<ClassMultiArray >("Containers",12);
}
#endif
///////////////////////////////////////////////////////////////////////////////

template<typename SeqContainer, typename SeqTrackingContainer,
         typename SetContainer, typename MapContainer>
class ClassCombined {
public:

	ClassCombined() {}
	ClassCombined(bool) : sequence1(true), sequence2(true),
	                      set1(true), set2(true),
	                      sequenceWithPtr1(true), sequenceWithPtr2(true),
	                      map1(true), map2(true),
	                      array1(true), array2(true),
	                      multiArray1(true), multiArray2(true) {}

	void check() {
		sequence1.check();
		sequence2.check();
		set1.check();
		set2.check();
		sequenceWithPtr1.check();
		sequenceWithPtr2.check();
		map1.check();
		map2.check();
		array1.check();
		array2.check();
		multiArray1.check();
		multiArray2.check();
	}

	template<typename Reflector>
	void reflect(Reflector& r) {
		r.member("Sequence1", sequence1, "");
		r.member("Set1", set1, "");
		r.member("SequenceWithPtr1", sequenceWithPtr1, "");
		r.member("Map1", map1, "");
		r.member("Array1", array1, "");
		r.member("MultiArray1", multiArray1, "");
		r.member("Sequence2", sequence2, "");
		r.member("Set2", set2, "");
		r.member("SequenceWithPtr2", sequenceWithPtr2, "");
		r.member("Map2", map2, "");
		r.member("Array2", array2, "");
		r.member("MultiArray2", multiArray2, "");
	}

public:

	ClassSequence<SeqContainer> sequence1, sequence2;
	ClassSet<SetContainer> set1, set2;
	ClassSequenceWithPtr<SeqTrackingContainer> sequenceWithPtr1, sequenceWithPtr2;
	ClassMap<MapContainer> map1, map2;
	ClassArray array1, array2;
	ClassMultiArray multiArray1, multiArray2;
};

BOOST_AUTO_TEST_CASE( TestContainersCombined)
{
	testAll<ClassCombined<vector<int>, vector<string>, set<int>, map<int, string> > >("Containers",14);
	testAll<ClassCombined<vector<string>, vector<string>, set<string>, map<string, string> > >("Containers",15);
}

///////////////////////////////////////////////////////////////////////////////
