/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file BasicClasses.h
 *
 * @author Erik Einhorn
 */

#ifndef _MIRA_BASISCLASSES_H_
#define _MIRA_BASISCLASSES_H_

#include <string>
#include <vector>

#ifndef Q_MOC_RUN
#include <boost/test/unit_test.hpp>
#include <boost/test/floating_point_comparison.hpp>
#endif

#include <math/Math.h>

#include <serialization/GetterSetter.h>

using namespace std;
using namespace mira;


class ClassNormal
{
public:

	ClassNormal() : cA(0), c5(0), i(0), f(0), fnan(0), finf(0), fninf(0), d(0) {}

	// set member to some values
	ClassNormal(bool) : cA('A'), c5(5), i(12), f(1.23456f),
						fnan(std::numeric_limits<float>::quiet_NaN()),
						finf(std::numeric_limits<float>::infinity()),
						fninf(-std::numeric_limits<float>::infinity()),
						d(1.23456789),
						s("This is a string")
	{
		v.push_back(1);
		v.push_back(2);
		v.push_back(3);
		v.push_back(4);
		v.push_back(5);

		vv.push_back(v);
		vv.push_back(v);
		vv.push_back(v);
		for(int i=0; i<5; ++i)
			array[i]=2*i;
	}

	// returns true, if the values are as required
	void check() const {
		BOOST_CHECK_EQUAL(cA, 'A');
		BOOST_CHECK_EQUAL(c5, 5);
		BOOST_CHECK_EQUAL(i, 12);
		BOOST_CHECK(boost::math::isnan(fnan));
		BOOST_CHECK(boost::math::isinf(finf));
		BOOST_CHECK_EQUAL(fninf, -std::numeric_limits<float>::infinity());
		BOOST_CHECK_CLOSE(f, 1.23456f, 0.000001f);
		BOOST_CHECK_CLOSE(d, 1.23456789,  0.0000000001);
		BOOST_CHECK_EQUAL(s, "This is a string");

		BOOST_REQUIRE_EQUAL(v.size(), 5);
		BOOST_CHECK_EQUAL(v[0], 1);
		BOOST_CHECK_EQUAL(v[1], 2);
		BOOST_CHECK_EQUAL(v[2], 3);
		BOOST_CHECK_EQUAL(v[3], 4);
		BOOST_CHECK_EQUAL(v[4], 5);

		BOOST_REQUIRE_EQUAL(vv.size(), 3);
		for(int i=0; i<3; ++i)
		{
			BOOST_REQUIRE_EQUAL(vv[i].size(), 5);
			BOOST_CHECK_EQUAL(vv[i][0], 1);
			BOOST_CHECK_EQUAL(vv[i][1], 2);
			BOOST_CHECK_EQUAL(vv[i][2], 3);
			BOOST_CHECK_EQUAL(vv[i][3], 4);
			BOOST_CHECK_EQUAL(vv[i][4], 5);
		}

		for(int i=0; i<5; ++i)
			BOOST_CHECK_EQUAL(array[i],2*i);
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("cA", cA, "");
		r.member("c5", c5, "");
		r.member("i", i, "An integer");
		r.member("f", f, "A float");
		r.member("fnan", fnan, "A float containing NaN");
		r.member("finf", finf, "A float containing Inf");
		r.member("fninf", fninf, "A float containing -Inf");
		r.member("d", d, "A double");
		r.member("array", array, "");
		r.member("s", s, "A string");
		r.member("v", v, "A vector of ints");
		r.member("vv", vv, "A vector of vector of ints");
	}

//private:

	char cA, c5;
	int i;
	float f, fnan, finf, fninf;
	double d;

	int array[5];
	std::string s;
	std::vector<int> v;
	std::vector<vector<int> > vv;

};

class ClassWithSetter
{
public:

	ClassWithSetter() : i(0), j(0){}
	ClassWithSetter(bool) : i(0), j(0) {}

	// returns true, if the values are as required
	void check() {
		BOOST_CHECK_EQUAL(i, 12345);
		BOOST_CHECK_EQUAL(j, 54321);
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("i", i, setter(&ClassWithSetter::setI, this), "");
		r.member("j", j, setter(&ClassWithSetter::setJ, this), "");
	}

	void setI(const int& v)
	{
		i = 12345;
	}

	void setJ(int v)
	{
		j = 54321;
	}

private:

	int i, j;
};

class ClassWithGetterAndSetter
{
public:

	ClassWithGetterAndSetter() : i(0) {}
	ClassWithGetterAndSetter(bool) : i(0) {}

	// returns true, if the values are as required
	void check() {
		BOOST_CHECK_EQUAL(i, 54321);
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("i", getter(&ClassWithGetterAndSetter::getI, this),
					  setter(&ClassWithGetterAndSetter::setI, this), "");
	}

	int getI() const
	{
		return 12345;
	}

	void setI(int v)
	{
		BOOST_CHECK_EQUAL(v, 12345);
		i = 54321;
	}

private:

	int i;
};

class ClassWithVersionA
{
public:

	ClassWithVersionA() : i(0), j(0) {}
	ClassWithVersionA(bool) : i(123), j(543) {}

	void check() {
		BOOST_CHECK_EQUAL(i, 123);
		BOOST_CHECK_EQUAL(j, 543);
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.requireVersion(4);
		r.member("i", i, "");
		r.member("j", j, "");
	}

private:
	int i;
	int j;
};

class ClassWithVersionB
{
public:

	ClassWithVersionB() : i(0), j(0) {}
	ClassWithVersionB(bool) : i(123), j(543) {}

	void check() {
		BOOST_CHECK_EQUAL(i, 123);
		if(mVersion==4)
			BOOST_CHECK_EQUAL(j, 543);
		else
			BOOST_CHECK_EQUAL(j, 777);
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		int version = r.version(4);
		r.member("i", i, "");

		if(version>=4)
			r.member("j", j, "");
		else
			j=777;

		mVersion = version; // for check
	}

private:
	int i;
	int j;
	int mVersion;
};


class ClassWithoutDefaultValue
{
public:

	ClassWithoutDefaultValue() : i(0), j(0) {}
	ClassWithoutDefaultValue(bool) : i(123), j(543) {}

	void check() {
		BOOST_CHECK_EQUAL(i, 123);
		BOOST_CHECK_EQUAL(j, 543);
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("i", i, "");
		r.member("j", j, "");
	}

private:
	int i;
	int j;
};

class ClassWithDefaultValue
{
public:

	ClassWithDefaultValue() : i(0), j(0) {}
	ClassWithDefaultValue(bool) : i(123), j(543) {}

	void check() {
		BOOST_CHECK_EQUAL(i, 123);
		BOOST_CHECK_EQUAL(j, 543);
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("i", i, "");
		r.member("j", j, "", 543);
	}

private:
	int i;
	int j;
};

#endif
