/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RPCManagerTest.C
 *    Test case for RPCManager and RPC.
 *
 * @author Erik Einhorn
 * @date   2012/03/20
 */

#include <serialization/JSONSerializer.h>
#include <serialization/Print.h>
#include <serialization/adapters/std/vector>
#include <rpc/RPCManager.h>

#include <utils/Hexdump.h>

#include <boost/test/unit_test.hpp>
#include <boost/function.hpp>
#include <boost/bind.hpp>

/////////////////////////////////////////////////////////////////////////////////
using namespace mira;
using namespace std;



class Service1
{
public:

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.interface("IInterface1");
		r.method("method1", &Service1::method1, this, "");
		r.method("method2", &Service1::method2, this, "");
	}

	int method1(float f, int value, const std::string& s)
	{
		std::cout << "Service1::method1" << std::endl;
		return 10+value;
	}

	int method2(float f, int value, const std::string& s)
	{
		std::cout << "Service1::method2" << std::endl;
		return 20+value;
	}

};


class Service2
{
public:

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.interface("IInterface2");
		r.method("foo", &Service2::foo, this, "");
	}

	int foo(int value)
	{
		std::cout << "Service2::foo: " << value << std::endl;
		return 30+value;
	}

};

class Service3
{
public:

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.interface("IInterface1");
		r.method("method1", &Service3::method1, this, "");
		r.method("method2", &Service3::method2, this, "");
	}

	int method1(float f, int value, const std::string& s)
	{
		std::cout << "Service3::method1" << std::endl;
		return 40+value;
	}

	int method2(float f, int value, const std::string& s)
	{
		std::cout << "Service3::method2" << std::endl;
		return 50+value;
	}
};

///////////////////////////////////////////////////////////////////////////////

class RPCHandler1 : public AbstractRPCHandler
{
public:
	virtual void queueRequest(AbstractDeferredInvokerPtr invoker)
	{
		invoker->invoke();
	}

	virtual boost::thread::id getThreadID() const
	{
		return boost::this_thread::get_id();
	}
};

class FinishHandler : public DeferredInvokerFinishHandler
{
public:
	virtual void onRPCfinished(const std::string& callId)
	{
		std::cout << "FinishHandler::onRPCfinished: " << callId << std::endl;
	}
};

class RPCHandler2 : public AbstractRPCHandler
{
public:

	RPCHandler2()
	{
		thread = boost::thread(&RPCHandler2::process, this);
	}

	virtual ~RPCHandler2()
	{
		thread.interrupt();
		thread.join();
	}

	virtual void queueRequest(AbstractDeferredInvokerPtr invoker)
	{
		queue.push(invoker);
	}

	virtual boost::thread::id getThreadID() const
	{
		return thread.get_id();
	}

	void process()
	{
		while(!boost::this_thread::interruption_requested())
		{
			boost::mutex::scoped_lock lock(queueMutex);
			if(!queue.empty())
			{
				AbstractDeferredInvokerPtr invoker = queue.front();
				queue.pop();
				lock.unlock();

				invoker->invoke();

			} else
				lock.unlock();
			boost::this_thread::sleep(Duration::milliseconds(100));
		}
	}

	boost::mutex queueMutex;
	std::queue<AbstractDeferredInvokerPtr> queue;

	boost::thread thread;

};


class RemoteFinishHandler3 : public RPCManager::RemoteFinishHandler
{
public:

	RemoteFinishHandler3(RPCManager& iLocalmgr) :
		localmgr(iLocalmgr)
	{}


	virtual void onRPCfinished(const Buffer<uint8>& response)
	{
		std::cout << "RemoteFinishHandler3::onRPCfinished, size: " << response.size()  << std::endl;
		hexdump(std::cout, response);
		localmgr.handleRemoteResponse(response);
	}

	RPCManager& localmgr;
};

class RemoteRequestHandler3 : public RPCManager::RemoteRequestHandler
{
public:

	RemoteRequestHandler3(RPCManager& iRemotemgr, RPCManager& localmgr) :
		finishHandler(new RemoteFinishHandler3(localmgr)),
		remotemgr(iRemotemgr)
	{}

	virtual void onRPCrequested(const Buffer<uint8>& request)
	{
		std::cout << "RemoteRequestHandler3::onRPCrequested, size: " << request.size() << std::endl;
		hexdump(std::cout, request);
		remotemgr.handleRemoteRequest(request, finishHandler);
	}

	RPCManager::RemoteFinishHandlerPtr finishHandler;
	RPCManager& remotemgr;
};


BOOST_AUTO_TEST_CASE(FullTest)
{
	RPCManager mgr;

	AbstractRPCHandlerPtr handler1(new RPCHandler1);
	AbstractRPCHandlerPtr handler2(new RPCHandler2);

	Service1 service1;
	Service2 service2;

	std::cout << "<<<<<<<<<<<<<< Local tests <<<<<<<<<<<<<<<<<<<<<<<" << std::endl;
	mgr.addLocalService("Service1", service1, handler1);
	mgr.addLocalService("Service2", service2, handler2);

	std::cout << "Local services: " << std::endl;
	const std::set<std::string>& sl = mgr.getLocalServices();
	foreach(const std::string& v, sl)
		std::cout << "  " << v << std::endl;

	std::cout << "Remote services: " << std::endl;
	const std::set<std::string>& sr = mgr.getRemoteServices();
	foreach(const std::string& v, sr)
		std::cout << "  " << v << std::endl;

	BOOST_CHECK_EQUAL(mgr.existsService("Service1"), true);
	BOOST_CHECK_EQUAL(mgr.existsService("Service2"), true);
	BOOST_CHECK_EQUAL(mgr.existsService("UnknownService"), false);

	// call from same thread as handler
	RPCFuture<int> res1 = mgr.call<int>("Service1", "method1", 1.234f, 1, std::string("Test"));
	BOOST_CHECK_EQUAL(res1.get(), 11);

	// handler is within different thread
	RPCFuture<int> res2 = mgr.call<int>("Service2", "foo", 1);
	BOOST_CHECK_EQUAL(res2.get(), 31);

	BOOST_CHECK_THROW(mgr.call<int>("Service2", "doesnotexist", 1).get(), XRPC);
	BOOST_CHECK_THROW(mgr.call<int>("UnknownService", "doesnotexist", 1).get(), XRPC);

	BOOST_CHECK_THROW(mgr.call<int>("Service3", "doesnotexist", 1).get(), XRPC);

	std::cout << "<<<<<<<<<<<<<< Remote tests <<<<<<<<<<<<<<<<<<<<<<<" << std::endl;
	// simulate a remote RPCManager and a remote call
	Service3 service3;
	RPCManager remotemgr;
	remotemgr.addLocalService("Service3", service3, handler2);

	RPCServer::Service svcinfo = remotemgr.getLocalService("Service3");
	// copy to remove service structure
	RPCManager::RemoteService rsvcinfo(svcinfo.name);
	rsvcinfo.interfaces = svcinfo.interfaces;
	foreach(const RPCServer::MethodInfo& i, svcinfo.methods)
		rsvcinfo.methods.insert(i);

	RPCManager::RemoteRequestHandlerPtr handler3(new RemoteRequestHandler3(remotemgr,mgr));
	mgr.addRemoteService(rsvcinfo, handler3);

	std::cout << "Local services: " << std::endl;
	const std::set<std::string>& sl2 = mgr.getLocalServices();
	foreach(const std::string& v, sl2)
		std::cout << "  " << v << std::endl;

	std::cout << "Remote services: " << std::endl;
	const std::set<std::string>& sr2 = mgr.getRemoteServices();
	foreach(const std::string& v, sr2)
		std::cout << "  " << v << std::endl;

	std::list<std::string> svclist = mgr.queryServicesForInterface("IInterface1");
	std::cout << "Services with interface 'IInterface1': " << std::endl;
	foreach(const std::string& v, svclist)
		std::cout << "  " << v << std::endl;
	BOOST_CHECK_EQUAL(svclist.size(), 2);

	RPCFuture<int> res3 = mgr.call<int>("Service3", "method2", 1.234f, 1, std::string("Test"));
	BOOST_CHECK_EQUAL(res3.get(), 51);


	std::cout << "<<<<<<<<<<<<<< JSON tests <<<<<<<<<<<<<<<<<<<<<<<" << std::endl;
	// test json calls
	RPCFuture<JSONRPCResponse> jres1 = mgr.callJSON("Service1", "method1", std::string("1.234, 2, \"test\""));
	BOOST_CHECK_EQUAL(json::write(jres1.get()), "12");

	RPCFuture<JSONRPCResponse> jres2 = mgr.callJSON("Service2", "foo", std::string("2"));
	BOOST_CHECK_EQUAL(json::write(jres2.get()), "32");

	BOOST_CHECK_THROW(mgr.callJSON("Service2", "doesnotexist", std::string("1") ).get(), XRPC);
	BOOST_CHECK_THROW(mgr.callJSON("UnknownService", "doesnotexist", std::string("1") ).get(), XRPC);

	RPCFuture<JSONRPCResponse> jres3 = mgr.callJSON("Service3", "method2", std::string("1.234, 2, \"Test\""));
	BOOST_CHECK_EQUAL(json::write(jres3.get()), "52");

	BOOST_CHECK_THROW(mgr.callJSON("Service3", "doesnotexist", std::string("1") ).get(), XRPC);

	std::cout << "<<<<<<<<<<<<<< Testing with same ID <<<<<<<<<<<<<<<<<<<<<<<" << std::endl;
	{
	RPCManager lmgr;

	RPCServer::Service svcinfo = mgr.getLocalService("Service2");
	RPCManager::RemoteService rsvcinfo(svcinfo.name);
	foreach(const RPCServer::MethodInfo& i, svcinfo.methods)
		rsvcinfo.methods.insert(i);

	RPCManager::RemoteRequestHandlerPtr handler3(new RemoteRequestHandler3(mgr,lmgr));
	lmgr.addRemoteService(rsvcinfo, handler3);

	// test json calls
	json::Value request1 = json::Object();
	request1.get_obj()["jsonrpc"] = json::Value("2.0");
	request1.get_obj()["id"] = json::Value("sameforall");
	request1.get_obj()["method"] = json::Value("Service2.foo");
	json::read("[1]", request1.get_obj()["params"]);

	// call from DIFFERENT clients with same call id should work fine
	RPCFuture<JSONRPCResponse> jres1 = lmgr.callJSON(request1);
	json::read("[2]", request1.get_obj()["params"]);
	RPCFuture<JSONRPCResponse> jres2 = mgr.callJSON(request1);

	BOOST_CHECK_EQUAL(json::write(jres1.get()), "31");
	BOOST_CHECK_EQUAL(json::write(jres2.get()), "32");


	// call with same ID from SAME client results in broken promise
	json::read("[3]", request1.get_obj()["params"]);
	RPCFuture<JSONRPCResponse> jres3 = mgr.callJSON(request1);
	json::read("[4]", request1.get_obj()["params"]);
	RPCFuture<JSONRPCResponse> jres4 = mgr.callJSON(request1);

	BOOST_CHECK_THROW(jres3.get(), std::logic_error);
	std::string res4s = json::write(jres4.get());
	BOOST_CHECK(res4s=="33" || res4s=="34"); // result could be one of the two

	boost::this_thread::sleep(Duration::milliseconds(1000));
	
	json::read("[5]", request1.get_obj()["params"]);
	RPCFuture<JSONRPCResponse> jres5 = lmgr.callJSON(request1);
	
	json::read("[6]", request1.get_obj()["params"]);
	RPCFuture<JSONRPCResponse> jres6 = lmgr.callJSON(request1);
	
	BOOST_CHECK_THROW(jres5.get(), std::logic_error);
	std::string res6s = json::write(jres6.get());
	BOOST_CHECK(res6s=="36" || res6s=="35"); // result could be one of the two

	// wait to make sure that all pending calls came back, before we destroy the RPCManager
	boost::this_thread::sleep(Duration::milliseconds(2000));
	}

	std::cout << "<<<<<<<<<<<<<< Remove service tests <<<<<<<<<<<<<<<<<<<<<<<" << std::endl;
	mgr.removeLocalService("Service1");
	BOOST_CHECK_EQUAL(mgr.existsService("Service1"), false);

	std::cout << "Local services: " << std::endl;
	const std::set<std::string>& slb = mgr.getLocalServices();
	foreach(const std::string& v, slb)
		std::cout << "  " << v << std::endl;

	// call from same thread as handler
	BOOST_CHECK_THROW(mgr.call<int>("Service1", "method1", 1.234f, 1, std::string("Test")).get(), XRPC);


	boost::this_thread::sleep(Duration::milliseconds(1000));
}

