/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ImgSerializationTest.C
 *    Test cases for serialization of Img class.
 *
 * @author Erik Einhorn, Michael Volkhardt, Christof Schröter
 */

#include <boost/algorithm/string/replace.hpp>

#include <sstream>
#include <geometry/Point.h>
#include <image/Img.h>
#include <image/Color.h>
#include <serialization/MetaSerializer.h>
#include <opencv2/imgproc/imgproc.hpp>

#include <boost/test/unit_test.hpp>

using namespace mira;

///////////////////////////////////////////////////////////////////////////////

/**
 *    *---------------*
 *    | *----*        |
 *    |_|____|___ line|
 *    | |    |        |
 *    | *----*img2    |
 *    *---------------* img 100x100
 */

BOOST_AUTO_TEST_CASE( ImgSerializationTest ) {
	Img8U3 img(100,100);

	Buffer<uint8> buffer;
	BinaryBufferSerializer bos(&buffer);
	//draw line
	cv::line((cv::Mat&)img, Point2i(10,50),Point2i(90,50), Color::Red);
	bos.serialize(img);
	BinaryBufferDeserializer bis(&buffer);
	Img8U3 imgd;
	bis.deserialize(imgd);
	BOOST_CHECK(img == imgd);

	//roi
	cv::Rect roi =cv::Rect(10, 10, 50, 90);
	Img<> img2 = img(roi);
	bos.serialize(img2);
	Img<> img2d;
	bis.deserialize(img2d);
	BOOST_CHECK(img2 == img2d);

	//check against roied img
	BOOST_CHECK(img2d == img(roi));
}

BOOST_AUTO_TEST_CASE( MetaSerializerTest )
{
	MetaTypeDatabase db;
	MetaSerializer ms(db);

	Img8U3 img(100,100);
	ms.addMeta(img);

	// print out database for manual debugging purposes
	foreach(auto p, db)
	{
		std::cout << p.first <<  p.second->toString() << std::endl;
	}
}

template <typename ImgT = Img<>>
void checkJSON(const ImgT& img, int depth, char type, int channels, int width, int height)
{
	JSONSerializer serializer;
	json::Value v = serializer.serialize(img);

	std::string serialized = json::write(v, false);

	std::stringstream expected;

	expected << "{ \"Format\" : { \"Channels\" : " << channels << ", "
	                          << "\"Depth\" : " << depth << ", "
                              << "\"Type\" : \"" << type << "\" }, "
             <<   "\"Size\" : { \"Height\" : " << height << ", "
                              << "\"Width\" : " << width << " } }";

	BOOST_CHECK_EQUAL(boost::replace_all_copy(boost::replace_all_copy(serialized, "\n", ""), " ", ""),
	                  boost::replace_all_copy(boost::replace_all_copy(expected.str(), "\n", ""), " ", ""));
}

template<typename ImgT = Img<>>
void testJSON(int depth, char type, int channels, int width = 0, int height = 0)
{
	ImgT img(width, height);
	checkJSON(img, depth, type, channels, width, height);
}

template<>
void testJSON<Img<>>(int depth, char type, int channels, int width, int height)
{
	Img<> img;
	checkJSON(img, depth, type, channels, width, height);
}

BOOST_AUTO_TEST_CASE( ImgJSONSerializationTest ) {

	testJSON<>(8, 'U', 1);

	testJSON<Img8U1>(8, 'U', 1, 123, 456);
	testJSON<Img8U2>(8, 'U', 2, 234, 567);
	testJSON<Img8U3>(8, 'U', 3, 345, 678);
	testJSON<Img8U4>(8, 'U', 4, 456, 789);

	testJSON<Img16U1>(16, 'U', 1, 123, 456);
	testJSON<Img16U2>(16, 'U', 2, 234, 567);
	testJSON<Img16U3>(16, 'U', 3, 345, 678);
	testJSON<Img16U4>(16, 'U', 4, 456, 789);

	testJSON<Img16S1>(16, 'S', 1, 123, 456);
	testJSON<Img16S2>(16, 'S', 2, 234, 567);
	testJSON<Img16S3>(16, 'S', 3, 345, 678);
	testJSON<Img16S4>(16, 'S', 4, 456, 789);

	testJSON<Img32F1>(32, 'F', 1, 123, 456);
	testJSON<Img32F2>(32, 'F', 2, 234, 567);
	testJSON<Img32F3>(32, 'F', 3, 345, 678);
	testJSON<Img32F4>(32, 'F', 4, 456, 789);

	testJSON<Img64F1>(64, 'F', 1, 123, 456);
	testJSON<Img64F2>(64, 'F', 2, 234, 567);
	testJSON<Img64F3>(64, 'F', 3, 345, 678);
	testJSON<Img64F4>(64, 'F', 4, 456, 789);
}

