/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PathFinder.C
 *    Implementation of PathFinder.h.
 *
 * @author Tim Langner, Christian Martin
 * @date   2011/02/21
 */

#include <utils/PathFinder.h>

#include <boost/regex.hpp>

#include <utils/StringAlgorithms.h>
#include <platform/Environment.h>

#include <serialization/adapters/std/vector>
#include <serialization/Serialization.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

#define MIRA_PATH_VAR "MIRA_PATH"
#define MIRA_PATH "$" MIRA_PATH_VAR
#define MIRA_MAX_RECURSION_DEPTH 50

///////////////////////////////////////////////////////////////////////////////

void checkForMIRAPath()
{
	try
	{
		resolveEnvironmentVariable(MIRA_PATH_VAR);
	}
	catch(Exception&)
	{
		MIRA_THROW(XInvalidConfig, "The 'MIRA_PATH' environment variable is not "
		           "defined on your system");
	}
}

PathVector getMIRAPaths()
{
	checkForMIRAPath();
	return extractPaths(MIRA_PATH);
}

/**
 * Resolves all wildcards in path and returns all paths that match the wildcards
 * and that exist.
 */
static void resolvePathWildcards(const Path& path, PathVector& oPaths)
{
	// if there are no wildcards, simply add the path to oPaths.
	if(path.string().find('*')==std::string::npos &&
	   path.string().find('?')==std::string::npos) {
		if(boost::filesystem::is_directory(path))
			oPaths.push_back(path);
		return;
	}

	// go through path component by component and find first path component
	// that contains a wildcard
	Path base;
	Path p;
	auto it=path.begin();
	for(; it!=path.end(); ++it)
	{
		p = *it;
		if(p.string().find('*')==std::string::npos &&
		   p.string().find('?')==std::string::npos) {
			base /= p;
			continue;
		}
		break;
	}

	// p contains the pattern with wildcards
	std::string pattern = p.string();
	boost::replace_all(pattern, ".", "\\.");
	boost::replace_all(pattern, "*", ".*");
	boost::replace_all(pattern, "?", ".");

	// find rest of path (that also may contain wildcards, but they will be
	// handled in another iteration)
	Path rest;
	for(++it; it!=path.end(); ++it)
		rest /= *it;

	// now find all subdirs in base, that match the pattern in p
	// for each call resolvePathWildcards recursively
	// iterate over all files in this directory and check if the
	// filename/pattern matches
	boost::filesystem::directory_iterator endIt;
	try {
		for(boost::filesystem::directory_iterator dirIt(base); dirIt != endIt; ++dirIt)
		{
			if(!boost::filesystem::is_directory(dirIt->status()))
				continue;

			boost::smatch what;
			// Skip if no match
			if(!boost::regex_match(dirIt->path().filename().string(),
			                       what, boost::regex(pattern)))
				continue;

			Path fullPath = dirIt->path() / rest;
			resolvePathWildcards(fullPath,oPaths);
		}
	} catch(...) {} // ignore exceptions in directory_iterator constructor
}

static void resolvePathsWildcards(const PathVector& paths, PathVector& oPaths)
{
	foreach(const Path& p, paths)
		resolvePathWildcards(p, oPaths);
}

///////////////////////////////////////////////////////////////////////////////

std::string findProjectPath(const std::string& path)
{
	Path resolvedFind;
	// first try to find a single file
	try
	{
		resolvedFind = findProjectFile(path);
	}
	catch(XFileNotFound&)
	{
		// No file found try as single path
		try
		{
			resolvedFind = findProjectDirectory(path);
		}
		catch(XFileNotFound&)
		{
			// Not found as single file or directory try finding it recursively
			PathVector values;
			try
			{
				values = findProjectFiles(path, true);
				if (values.size() == 0)
					MIRA_THROW(XFileNotFound, "No files found");
			}
			catch(XFileNotFound&)
			{
				values = findProjectDirectories(path, true);
				if (values.size() == 0)
					MIRA_THROW(XFileNotFound, "FIND could not find any file or "
					           "directory that matches '" << path << "'");
			}
			if (values.size() > 1)
				MIRA_THROW(XFileNotFound, "FIND returned multiple matches."
				           " Please be more precise: " << print(values));
			resolvedFind = values[0];
		}
	}
	return resolvedFind.string();
}

///////////////////////////////////////////////////////////////////////////////

Path findFile(const PathVector& paths, const Path& file)
{
	PathVector rpaths;
	resolvePathsWildcards(paths, rpaths);

	foreach(const auto& p, rpaths)
	{
		Path searchPath = resolvePath(p / file);

		if (boost::filesystem::is_regular_file(searchPath))
			return searchPath;
	}
	MIRA_THROW(XFileNotFound, "Path was not found ('" << file.string() << "')");
	return Path();
}

Path findProjectFile(const Path& file)
{
	checkForMIRAPath();
	return findFile(extractPaths(MIRA_PATH), file);
}

///////////////////////////////////////////////////////////////////////////////

static void findFiles_intern(const Path& path, const Path& subPath,
                             const boost::regex& pattern,
                             bool recursive, PathVector& result,
                             int recursionDepth)
{
	// stop at max recursion depth because due to symlinks it is
	// possible that paths contain endless loops
	if (recursionDepth > MIRA_MAX_RECURSION_DEPTH)
		return;
	boost::filesystem::directory_iterator endIt;
	try
	{
		// iterate over all directories in the path
		for(boost::filesystem::directory_iterator it(path); it != endIt; ++it)
		{
			// skip if its no directory
			if(boost::filesystem::is_directory(it->status()))
			{
				// if we want to search recursively enter this path again to
				// search its subdirectories
				if (recursive)
					findFiles_intern(it->path(), subPath, pattern, recursive,
					                 result, recursionDepth+1);
				// combine path with sub path to searched file
				Path searchPath = it->path() / subPath;
				// check if this directory exists
				if (boost::filesystem::is_directory(searchPath))
				{
					// iterate over all files in this directory and check
					// if the filename/pattern matches
					for(boost::filesystem::directory_iterator fileIt(searchPath); fileIt != endIt; ++fileIt)
					{
						// skip if no file
						if(!boost::filesystem::is_regular_file(fileIt->status()))
							continue;
						boost::smatch what;
						// Skip if no match
						if(!boost::regex_match(fileIt->path().filename().string(),
						                       what, pattern))
							continue;
						// found a file - store it
						result.push_back(fileIt->path().string());
					}
				}
			}
		}
	}
	catch(...)
	{
		return;
	}
}

///////////////////////////////////////////////////////////////////////////////

PathVector findFiles(const PathVector& paths, const Path& file, bool recursive)
{
	PathVector rpaths;
	resolvePathsWildcards(paths, rpaths);

	PathVector ret;
	std::string pattern = file.filename().string();
	boost::replace_all(pattern, ".", "\\.");
	boost::replace_all(pattern, "*", ".*");
	boost::replace_all(pattern, "?", ".");

	// get the sub path to the searched file
	Path subPath = file.parent_path();

	// search in all paths contained in paths
	foreach(const auto& p, rpaths)
	{
		// combine MIRA_PATH component with sub path to searched file
		Path searchPath = p / subPath;
		// check if this directory exists
		if (boost::filesystem::is_directory(searchPath))
		{
			// iterate over all files in this directory and check if the
			// filename/pattern matches
			boost::filesystem::directory_iterator endIt;
			for(boost::filesystem::directory_iterator fileIt(searchPath); fileIt != endIt; ++fileIt)
			{
				// skip if no file
				if(!boost::filesystem::is_regular_file(fileIt->status()))
					continue;
				boost::smatch what;
				// Skip if no match
				if(!boost::regex_match(fileIt->path().filename().string(),
				                       what, boost::regex(pattern)))
					continue;
				// found a file - store it
				ret.push_back(fileIt->path().string());
			}
		}
		// if we want to search recursively enter the combined MIRA_PATH / path
		// to searched file
		if (recursive)
			findFiles_intern(p, subPath, boost::regex(pattern), recursive, ret, 0);
	}

	// Remove duplicates
	std::sort(ret.begin(), ret.end());
    ret.erase(std::unique(ret.begin(), ret.end()), ret.end());

	return ret;
}

PathVector findFiles(const Path& path, const Path& file, bool recursive)
{
	PathVector paths;
	paths.push_back(path);
	return findFiles(paths, file, recursive);
}

PathVector findProjectFiles(const Path& file, bool recursive)
{
	checkForMIRAPath();
	return findFiles(extractPaths(MIRA_PATH), file, recursive);
}

///////////////////////////////////////////////////////////////////////////////

Path findDirectory(const PathVector& paths, const Path& path)
{
	PathVector rpaths;
	resolvePathsWildcards(paths, rpaths);

	foreach(const auto& p, rpaths)
	{
		Path searchPath = resolvePath(p / path);
		if (boost::filesystem::is_directory(searchPath))
			return searchPath;
	}
	MIRA_THROW(XFileNotFound, "Path was not found ('" << path.string() << "')");
	return Path();
}

Path findProjectDirectory(const Path& path)
{
	checkForMIRAPath();
	return findDirectory(extractPaths(MIRA_PATH), path);
}

///////////////////////////////////////////////////////////////////////////////

static void findDirectories_intern(const Path& path, const Path& subPath,
                                   bool recursive, PathVector& result,
                                   int recursionDepth)
{
	if (recursionDepth > MIRA_MAX_RECURSION_DEPTH)
		return;
	boost::filesystem::directory_iterator endIt;
	try
	{
		for(boost::filesystem::directory_iterator it(path); it != endIt; ++it)
		{
			if(boost::filesystem::is_directory(it->status()))
			{
				if (recursive)
					findDirectories_intern(it->path(), subPath, recursive,
					                 result, ++recursionDepth);
				Path searchPath = it->path() / subPath;
				if (boost::filesystem::is_directory(searchPath))
					result.push_back(searchPath.string());
			}
		}
	}
	catch(...)
	{
		return;
	}
}

///////////////////////////////////////////////////////////////////////////////

PathVector findDirectories(const PathVector& paths, const Path& path, bool recursive)
{
	PathVector rpaths;
	resolvePathsWildcards(paths, rpaths);

	PathVector ret;
	foreach(const auto& p, rpaths)
	{
		if (boost::filesystem::is_directory(p / path))
			ret.push_back((p / path).string());
		if (recursive)
			findDirectories_intern(p, path, recursive, ret, 0);
	}
	return ret;
}

PathVector findDirectories(const Path& basePath, const Path& path, bool recursive)
{
	PathVector paths;
	paths.push_back(basePath);
	return findDirectories(paths, path, recursive);
}

PathVector findProjectDirectories(const Path& path, bool recursive)
{
	checkForMIRAPath();
	return findDirectories(extractPaths(MIRA_PATH), path, recursive);
}

///////////////////////////////////////////////////////////////////////////////

}
