/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ThreadMonitor.C
 *    Implementation of ThreadMonitor.h
 *
 * @author Christian Martin
 * @date   2010/09/20
 */

#include <thread/ThreadMonitor.h>

#include <fstream>

#include <boost/algorithm/string.hpp>
#include <boost/filesystem.hpp>
#include <boost/system/error_code.hpp>

#include <utils/Foreach.h>
#include <utils/ToString.h>

#include <error/LoggingCore.h>
#include <error/SystemError.h>

#ifdef MIRA_LINUX
#  include <sys/sysinfo.h>
#endif
#ifdef MIRA_WINDOWS
#  include <tlhelp32.h>
#endif

namespace mira {

///////////////////////////////////////////////////////////////////////////////

ThreadMonitor::ThreadMonitor()
{
}

ThreadMonitor::~ThreadMonitor()
{
}

///////////////////////////////////////////////////////////////////////////////

void ThreadMonitor::addThisThread(const std::string& pName)
{
	PrivateData tData;
	tData.id   = getCurrentThreadID();
	tData.bid  = boost::this_thread::get_id();
	tData.name = pName;

	boost::mutex::scoped_lock lock(mThreadsMutex);
	mThreads[tData.id] = tData;
}

void ThreadMonitor::removeThisThread()
{
	removeThread(getCurrentThreadID());
}

void ThreadMonitor::removeThread(const boost::thread& pThread)
{
	removeThread(pThread.get_id());
}

void ThreadMonitor::removeThread(const boost::thread::id& pThreadID)
{
	boost::mutex::scoped_lock lock(mThreadsMutex);

	ThreadMap::iterator tIter = mThreads.begin();
	while(tIter != mThreads.end()) {
		if (tIter->second.bid == pThreadID) {
			ThreadMap::iterator tErase = tIter;
			++tIter;
			mThreads.erase(tErase);
		} else
			++tIter;
	}
}

void ThreadMonitor::removeThread(ThreadID pThreadID)
{
	boost::mutex::scoped_lock lock(mThreadsMutex);

	ThreadMap::iterator tIter = mThreads.find(pThreadID);
	if (tIter != mThreads.end())
		mThreads.erase(tIter);
}

///////////////////////////////////////////////////////////////////////////////

ThreadMonitor::ThreadInfoVector ThreadMonitor::getThreadInformation()
{
	ThreadInfoVector tInfos;
	std::map<ThreadID, bool> tUsedThreadIDs;

	///////////////////////////////////////////////////////////////////////////
#ifdef MIRA_LINUX
	///////////////////////////////////////////////////////////////////////////

	long tClkTck = sysconf(_SC_CLK_TCK); // number of ticks per second

	long tUpTime = 0;
	struct sysinfo tSysInfo;
	if (sysinfo(&tSysInfo) == 0)
		tUpTime = tSysInfo.uptime;

	Time tNow = Time::now();
	// get only second precision
	tNow = Time(tNow.date(), Duration(tNow.time_of_day().hours(),
	            tNow.time_of_day().minutes(), tNow.time_of_day().seconds()));

	// Iterate through the /proc/self/task directory
	boost::filesystem::directory_iterator tEnd; // default construction is end
	for(boost::filesystem::directory_iterator tIter("/proc/self/task");
		tIter != tEnd; ++tIter)
	{
		if (!boost::filesystem::is_directory(tIter->status()))
			continue;

		// Get the thread ID from filename
		ThreadID tProcID = -1;
		try {
			tProcID = fromString<ThreadID>(tIter->path().filename().string());
		} catch (Exception& tEx) {
			// On Linux, each sub directory in /proc/self/task should have a number
			continue;
		}

		// Collect the thread information
		ThreadInfo tInfo;
		try {
			// Open the file /proc/self/task/%tid$/stat
			std::string tStatName = tIter->path().string()+"/stat";
			std::fstream tFStream;
			tFStream.open(tStatName.c_str(), std::fstream::in);
			std::string tLine;
			getline(tFStream, tLine);

			/**
			 * Content of the stat file (see proc(5) manual page for more details):
			 *   0  pid          %d   The process ID.
			 *   1  comm         %s   The filename of the executable in parentheses.
			 *   2  state        %c   The state (R|S|D|Z|T).
			 *   3  ppid         %d   The PID of the parent.
			 *   4  pgrp         %d   The process group ID of the process.
			 *   5  session      %d   The session ID of the process.
			 *   6  tty          %d   The controlling terminal of the process.
			 *   7  tpgid        %d   The ID of the foreground process group of the terminal.
			 *   8  flags        %u   The kernel flag of the process.
			 *   9  minflt       %u   The number of minor faults the process made.
			 *  10  cminflt      %u   The number of minor faults the process's children have made.
			 *  11  majflt       %u   The number of major faults the process made.
			 *  12  cmajflt      %u   The number of major faults the process's children have made.
			 *  13  utime        %ld  Amount of time in user mode, measured in clock ticks.
			 *  14  stime        %ld  Amount of time in kernel mode, measured in clock ticks.
			 *  15  cutime       %ld  Amount of time in user mode of child processes, measured in clock ticks.
			 *  16  cstime       %ld  Amount of time in kernel mode of child processes, measured in clock ticks.
			 *  17  priority     %ld  The process priority.
			 *  18  nice         %ld  The nice value.
			 *  19  num_threads  %ld  Number of threads in this process.
			 *  20  itrealvalue  %u   The time in jiffies before the next SIGALRM.
			 *  21  starttime    %llu The time in jiffies the process started after system boot.
			 *  22  vsize        %lu  Virtual memory size in bytes.
			 *  23  rss          %ld  Resident Set Size
			 *  24  rsslime      %u   Current soft limit in bytes on the rss of the process.
			 *  25  startcode    %lu  The address above which program text can run.
			 *  26  endcode      %lu  The address below which program text can run.
			 *  27  startstack   %lu  The address of the start (i.e., bottom) of the stack.
			 *  28  kstkesp      %lu  The current value of ESP (stack pointer).
			 *  29  kstkeip      %lu  The current EIP (instruction pointer).
			 *  30  signal       %d   The bitmap of pending signals, displayed as decimal numbers.
			 *  31  blocked      %d   The bitmap of blocked signals, displayed as a decimal number.
			 *  32  sigignore    %d   The bitmap of ignored signals, displayed as a decimal number.
			 *  33  sigcatch     %d   The bitmap of caught signals, displayed as a decimal number.
			 *  34  wchan        %u   This is the "channel" in which the process is waiting.
			 *  35  nswap        %lu  Number of pages swapped (not maintained).
			 *  36  cnswap       %lu  Cumulative nswap for child processes (not maintained).
			 *  37  exit_signal  %d   Signal to be sent to parent when we die.
			 *  38  processor    %d   CPU number last executed on.
			 *  39  rt_priority  %u   Real-time scheduling priority.
			 *  40  policy       %u   Scheduling policy.
			 *  41  delayacct_blkio_ticks %llu Aggregated block I/O delays, measured in clock ticks.
			 *  42  guest_time   %lu  Guest time of the process.
			 *  43  cguest_time  %ld  Guest time of the process's children, measured in clock ticks.
			 */

			std::vector<std::string> tParts;
			boost::algorithm::split(tParts, tLine, boost::algorithm::is_space());

			int64 tUTime = tParts.size() > 13?fromString<int64>(tParts[13]):0;
			int64 tSTime = tParts.size() > 14?fromString<int64>(tParts[14]):0;
			int64 tStart = tParts.size() > 21?fromString<int64>(tParts[21]):0;

			tInfo.create_time = tNow - Duration::seconds(tUpTime) +
				Duration::milliseconds(1000.0f*tStart/tClkTck);
			tInfo.user_time   = Duration::milliseconds(1000.0f*tUTime/tClkTck);
			tInfo.kernel_time = Duration::milliseconds(1000.0f*tSTime/tClkTck);
		} catch (Exception& tEx) {
		}


		tInfo.id = tProcID;

		// Determine the thread name and boost thread id
		mThreadsMutex.lock();
		ThreadMap::iterator tIter2 = mThreads.find(tProcID);
		if (tIter2 == mThreads.end()) {
			tInfo.known = false;
			tInfo.name = std::string("Unknown thread ID=") + toString(tProcID);
		} else {
			tInfo.known = true;
			tInfo.name = tIter2->second.name;
			tInfo.bid = tIter2->second.bid;
		}
		mThreadsMutex.unlock();

		// Append status information to list
		tInfos.push_back(tInfo);
		tUsedThreadIDs[tProcID] = true;
	}

	///////////////////////////////////////////////////////////////////////////
#elif defined(MIRA_WINDOWS)
	///////////////////////////////////////////////////////////////////////////

	// Take a snapshot of all running thread
	HANDLE tThreadSnap = CreateToolhelp32Snapshot(TH32CS_SNAPTHREAD, 0);
	if (tThreadSnap == INVALID_HANDLE_VALUE) {
		boost::system::error_code tErr = getLastSystemErrorCode();
		MIRA_THROW(XSystemCall, "CreateToolhelp32Snapshot failed: " <<
		           tErr.message());
	}

	// Initiailze interator structure before using it.
	THREADENTRY32 tEntry;
	tEntry.dwSize = sizeof(THREADENTRY32);

	// Retrieve the first thread.
	if (!Thread32First(tThreadSnap, &tEntry)) {
		boost::system::error_code tErr = getLastSystemErrorCode();
		MIRA_THROW(XSystemCall, "Thread32First failed: " <<
		           tErr.message());
	}

	// Go through the thread list and collect the information
	do {
		// Only use threads from our own process
		if (tEntry.th32OwnerProcessID == GetCurrentProcessId()) {
			ThreadInfo tInfo;
			tInfo.name = toString(tEntry.th32ThreadID);

			// Open the thread
			HANDLE tThread = OpenThread(THREAD_QUERY_INFORMATION, TRUE, tEntry.th32ThreadID);
			if (tThread == NULL) {
				boost::system::error_code tErr = getLastSystemErrorCode();
				MIRA_THROW(XSystemCall, "OpenThread failed: " << tErr.message());
			}

			// Get the thread times
			FILETIME tCreateTime, tExitTime, tKernelTime, tUserTime;
			if (GetThreadTimes(tThread, &tCreateTime, &tExitTime, &tKernelTime, &tUserTime)) {
				// Create and ExitTime are absolute times since January 1, 1601, GMT
				// Kernel and User time are amounts of time in 100-nanoseconds

				ULARGE_INTEGER tLargeInt;

				tLargeInt.LowPart  = tCreateTime.dwLowDateTime;
				tLargeInt.HighPart = tCreateTime.dwHighDateTime;

				// tLargeInt.QuadPart is in [100ns] !
				Time t1601 = Time(Date(1601,boost::date_time::Jan,1), Duration(0,0,0));
				tInfo.create_time = t1601 + Duration::microseconds(tLargeInt.QuadPart/10);

				tLargeInt.LowPart  = tUserTime.dwLowDateTime;
				tLargeInt.HighPart = tUserTime.dwHighDateTime;
				tInfo.user_time = Duration::microseconds(tLargeInt.QuadPart / 10);

				tLargeInt.LowPart  = tKernelTime.dwLowDateTime;
				tLargeInt.HighPart = tKernelTime.dwHighDateTime;
				tInfo.kernel_time = Duration::microseconds(tLargeInt.QuadPart / 10);
			} else {
				boost::system::error_code tErr = getLastSystemErrorCode();
				MIRA_THROW(XSystemCall, "GetThreadTimes failed: " << tErr.message());
			}

			tInfo.id = tEntry.th32ThreadID;

			// Determine the thread name
			mThreadsMutex.lock();
			ThreadMap::iterator tIter = mThreads.find(tEntry.th32ThreadID);
			if (tIter == mThreads.end()) {
				tInfo.known = false;
				tInfo.name = std::string("Unknown thread ID=") + toString(tEntry.th32ThreadID);
			} else {
				tInfo.known = true;
				tInfo.name = tIter->second.name;
				tInfo.bid = tIter->second.bid;
			}
			mThreadsMutex.unlock();

			// Cleanup
			CloseHandle(tThread);

			// Append status information to list
			tUsedThreadIDs[tEntry.th32ThreadID] = true;
			tInfos.push_back(tInfo);
		}
	} while(Thread32Next(tThreadSnap, &tEntry));

	// Close the handle
	CloseHandle(tThreadSnap);

	///////////////////////////////////////////////////////////////////////////
#endif
	///////////////////////////////////////////////////////////////////////////

	// Remove threads, which are not longer exist
	mThreadsMutex.lock();
	ThreadMap::iterator tIter = mThreads.begin();
	while (tIter != mThreads.end()) {
		ThreadID tID = tIter->first;
		if (tUsedThreadIDs.find(tID) == tUsedThreadIDs.end()) {
			MIRA_LOG(NOTICE) << "Removed thread with ID=" << tID << " (name='"
			                 << tIter->second.name << "') from list, "
			                 << "since it does not longer exist.";
			ThreadMap::iterator tErase = tIter;
			++tIter;
			mThreads.erase(tErase);
		} else
			++tIter;
	}
	mThreadsMutex.unlock();

	return(tInfos);
}

///////////////////////////////////////////////////////////////////////////////

}
