/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file LoggingCore.C
 *    Implementation of LoggingCore.h
 *
 * @author Tim Langner
 * @date   2010/07/08
 */

#include <boost/bind.hpp>
#include <boost/function.hpp>
#include <boost/algorithm/string.hpp>

#include <error/LoggingCore.h>
#include <error/Exceptions.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

void LogSink::aboutToConsume(const LogRecord& record)
{
	if ( mFilter != NULL )
	{
		if ( mFilter->filter(record) )
			consume(record);
	}
	else
		consume(record);
}

///////////////////////////////////////////////////////////////////////////////

LogCore::LogCore()
{
	mStartTime = Time::now();
	mSeverityLevel = DEBUG;
	mCallbackCounter = 0;
}

LogRecordFunctionInfo LogCore::parseFunction(const LogRecord& record)
{
	LogRecordFunctionInfo info;
	const char* p = record.function.c_str();
	const char* scope = NULL;
	const char* nsscope = NULL;
	const char* ns = NULL;
	const char* cl = NULL;

	// skip return value (search for first space)
	while (*p!=0 && !isspace(*p))
		++p;
	info.functionName = record.function;
	while (*p!=0 && *p!='(')
	{
		if (isspace(*p))
		{
			while(isspace(*p)) // skip spaces
				if(*(++p) == 0)
					return info;
			scope = nsscope = ns = p;
			info.functionName = p;
		}
		if(*p==':' && *(p+1)==':')
		{
			info.className = std::string(scope, p-scope);
			cl = nsscope;
			nsscope = p;
			p=p+2;
			info.functionName = p;
			scope = p;
		}
		++p;
	}
	if ( cl != NULL && cl != ns )
		info.nameSpace = std::string(ns, cl-ns);

	return info;
}

///////////////////////////////////////////////////////////////////////////////

SeverityLevel stringToSeverityLevel(const std::string& levelString)
{
	std::string levelStringTrimmed = boost::trim_copy(levelString);

	for(int level=CRITICAL; level<=TRACE; ++level)
	{
		std::string s = boost::trim_copy(severityLevelStr[level]);
		if(boost::iequals(s,levelStringTrimmed))
			return (SeverityLevel)level;
	}

	MIRA_THROW(XInvalidParameter, "The severity level '" << levelStringTrimmed << "' does not exist.");
	return CRITICAL;
}

///////////////////////////////////////////////////////////////////////////////

}
