/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Time.h
 *    Time and Duration wrapper class.
 *
 * @author Tim Langner
 * @date   2010/09/06
 */

#ifndef _MIRA_TIME_H_
#define _MIRA_TIME_H_

#ifndef Q_MOC_RUN
#include <boost/date_time/posix_time/posix_time.hpp>
#include <boost/date_time/local_time/local_time.hpp>
#include <boost/date_time/c_local_time_adjustor.hpp>
#endif

#include <platform/Types.h>
#include <serialization/SplitReflect.h>
#include <serialization/IsTransparentSerializable.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup TimeModule
 * Typedef for the boost Gregorian calendar date.
 */
typedef boost::gregorian::date Date;
typedef boost::gregorian::date_duration DateDuration;

using boost::date_time::Jan;
using boost::date_time::Feb;
using boost::date_time::Mar;
using boost::date_time::Apr;
using boost::date_time::May;
using boost::date_time::Jun;
using boost::date_time::Jul;
using boost::date_time::Aug;
using boost::date_time::Sep;
using boost::date_time::Oct;
using boost::date_time::Nov;
using boost::date_time::Dec;

using boost::date_time::Monday;
using boost::date_time::Tuesday;
using boost::date_time::Wednesday;
using boost::date_time::Thursday;
using boost::date_time::Friday;
using boost::date_time::Saturday;
using boost::date_time::Sunday;

/**
 * Use this class to represent time durations.
 * Time durations can be added to each other and to Time
 * objects. For specifying time durations you can use
 * \ref microseconds, \ref milliseconds, \ref seconds,
 * \ref minutes and \ref hours.
 * The underlying class is boost::posix_time::time_duration.
 * For documentation see
 * http://boost.org/doc/libs/1_44_0/doc/html/date_time/posix_time.html .
 *
 * @ingroup TimeModule
 */
class Duration : public boost::posix_time::time_duration
{
protected:

	typedef boost::posix_time::time_duration Base;

public:

	/**
	 * Can be used to construct a Duration object that
	 * is specified in nanoseconds.
	 * @see seconds
	 */
	static Duration nanoseconds(int64 v) {
		Duration d;
#if defined(BOOST_DATE_TIME_HAS_NANOSECONDS)
		d.ticks_ = v;
#else
		d.ticks_ = v / 1000;
#endif
		return d;
	}

	/**
	 * Can be used to construct a Duration object that
	 * is specified in microseconds.
	 * @see seconds
	 */
	static Duration microseconds(int64 v) {
		Duration d;
#if defined(BOOST_DATE_TIME_HAS_NANOSECONDS)
		d.ticks_ = v * 1000;
#else
		d.ticks_ = v;
#endif
		return d;
	}

	/**
	 * Can be used to construct a Duration object that
	 * is specified in milliseconds.
	 * @see seconds
	 */
	static Duration milliseconds(int64 v) {
		Duration d;
#if defined(BOOST_DATE_TIME_HAS_NANOSECONDS)
		d.ticks_ = v * 1000000;
#else
		d.ticks_ = v * 1000;
#endif
		return d;
	}

	/**
	 * Can be used to construct a Duration object that
	 * is specified in seconds.
	 * e.g:
	 * \code
	 * Duration duration = Duration::seconds(20);
	 * \endcode
	 *
	 * Durations can also be added:
	 * \code
	 * Duration duration = Duration::seconds(20) + Duration::milliseconds(100);
	 * \endcode
	 */
	static Duration seconds(int32 v) { return boost::posix_time::seconds(v); }

	/**
	 * Can be used to construct a Duration object that
	 * is specified in minutes.
	 * @see seconds
	 */
	static Duration minutes(int32 v) { return boost::posix_time::minutes(v); }

	/**
	 * Can be used to construct a Duration object that
	 * is specified in hours.
	 * @see seconds
	 */
	static Duration hours(int32 v) { return boost::posix_time::hours(v); }

public:
	Duration(hour_type hour, min_type min, sec_type sec, 
	         fractional_seconds_type fs=0) :
		Base(hour, min, sec, fs)
	{}

	/// Construct from duration
	Duration(const Duration& d) :
		Base(d)
	{}

	/// Construct from base class.
	Duration(Base d) :
		Base(d)
	{}

	/// Construct from special value.
	Duration(const boost::posix_time::special_values sv) :
		Base(sv)
	{}

#if !defined(DATE_TIME_NO_DEFAULT_CONSTRUCTOR)
	/// Default constructor constructs to not_a_date_time.
	Duration()
	{}
#endif // DATE_TIME_NO_DEFAULT_CONSTRUCTOR

public:

	MIRA_SPLIT_REFLECT_MEMBER

	/// Reflect
	template<typename Reflector>
	void reflectRead(Reflector& r)
	{
		int64 t = totalMilliseconds();
		r.delegate(t);
	}

	template<typename Reflector>
	void reflectWrite(Reflector& r)
	{
		int64 t;
		r.delegate(t);
		*this = Duration::milliseconds(t);
	}

public:

	/**
	 * Returns a special duration time representing negative infinity
	 */
	static Duration negativeInfinity()
	{
		return Duration(boost::posix_time::neg_infin);
	}
	/**
	 * Returns a special duration time representing positive infinity
	 */
	static Duration infinity()
	{
		return Duration(boost::posix_time::pos_infin);
	}
	/**
	 * Returns an invalid duration
	 */
	static Duration invalid()
	{
		return Duration(boost::posix_time::not_a_date_time);
	}

	/**
	 * Checks if this duration is invalid
	 */
	bool isValid() const
	{
		return !this->is_not_a_date_time();
	}
	/**
	 * Checks if this duration is infinity
	 */
	bool isInfinity() const
	{
		return this->is_pos_infinity() || this->is_neg_infinity();
	}

	/// Returns number of hours in the duration
	hour_type hours() const
	{
		return Base::hours();
	}
	/// Returns normalized number of minutes (0..59)
	min_type minutes() const
	{
		return Base::minutes();
	}
	/// Returns normalized number of seconds (0..59)
	sec_type seconds() const
	{
		return Base::seconds();
	}
	/// Returns normalized number of milliseconds (0..999)
	tick_type milliseconds() const
	{
		return totalMilliseconds() % 1000;
	}
	/// Returns normalized number of microseconds (0..999)
	tick_type microseconds() const
	{
		return totalMicroseconds() % 1000;
	}
	/// Returns normalized number of nanoseconds (0..999)
	tick_type nanoseconds() const
	{
		return totalNanoseconds() % 1000;
	}
	/// Returns total number of seconds truncating any fractional seconds
	sec_type totalSeconds() const
	{
		return Base::total_seconds();
	}
	/// Returns total number of milliseconds truncating any fractional milliseconds
	tick_type totalMilliseconds() const
	{
		return Base::total_milliseconds();
	}
	/// Returns total number of microseconds truncating any fractional microseconds
	tick_type totalMicroseconds() const
	{
		return Base::total_microseconds();
	}
	/// Returns total number of nanoseconds truncating any fractional nanoseconds
	tick_type totalNanoseconds() const
	{
		return Base::total_nanoseconds();
	}

public:

	/** @name Operators */
	//@{
	Duration operator-()const
	{
		return Base::operator -();
	}
	Duration operator-(const Duration& d) const
	{
		return Base::operator-(d);
	}
	Duration operator+(const Duration& d) const
	{
		return Base::operator+(d);
	}
	Duration operator/(int divisor) const
	{
		assert(divisor != 0);
		return Base::operator/(divisor);
	}
	Duration operator/(float divisor) const
	{
		assert(divisor != 0.0f);
		Duration d;
		d.ticks_ = (int64)(ticks_.as_number() / divisor);
		return d;
	}
	float operator/(Duration other) const
	{
		return (float)totalNanoseconds() / other.totalNanoseconds();
	}
	Duration operator-=(const Duration& d)
	{
		return Base::operator-=(d);
	}
	Duration operator+=(const Duration& d)
	{
		return Base::operator+=(d);
	}
	//! Division operations on a duration with an integer.
	Duration operator/=(int divisor)
	{
		assert(divisor != 0);
		return Base::operator/=(divisor);
	}
	Duration operator/=(float divisor)
	{
		assert(divisor != 0.0f);
		ticks_ = (int64)(ticks_.as_number() / divisor);
		return *this;
	}
	//! Multiplication operations an a duration with an integer
	Duration operator*(int rhs) const
	{
		return Base::operator*(rhs);
	}
	Duration operator*(float rhs) const
	{
		Duration d;
		d.ticks_ = (int64)(ticks_.as_number() * rhs);
		return d;
	}
	Duration operator*=(int rhs)
	{
		return Base::operator*=(rhs);
	}
	Duration operator*=(float rhs)
	{
		ticks_ = (int64)(ticks_.as_number() * rhs);
		return *this;
	}
	//@}
};

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup TimeModule
 * Get the absolute duration from a duration
 */
inline Duration abs(const Duration& duration)
{
	if ( duration.is_negative() )
		return duration.invert_sign();
	return duration;
}

/**
 * @ingroup TimeModule
 * Wrapper class for boost::posix_time::ptime for adding more functionality to it.
 * If not stated otherwise the time represented by this class is in UTC time.
 *
 * For more information on the underlying boost::posix_time classes refer to
 * http://boost.org/doc/libs/1_44_0/doc/html/date_time/posix_time.html .
 *
 * The underlying datatype in boost::posix_time::ptime is int64.
 */
class MIRA_BASE_EXPORT Time : public boost::posix_time::ptime
{
public:

	typedef boost::posix_time::ptime Base;

public:

	/// Construct from date and timespan.
	Time(Date d, Duration td) :
		Base(d, td)
	{}

	/// Construct from base class.
	Time(Base p) :
		Base(p)
	{}
	/// Construct a time at start of the given day (midnight)
	explicit Time(Date d) :
		Base(d, Duration(0, 0, 0))
	{}

	/// Construct from time representation
	Time(const Base::time_rep_type& rhs) :
		Base(rhs)
	{}
	/// Construct from special value.
	Time(const boost::posix_time::special_values sv) :
		Base(sv)
	{}

#if !defined(DATE_TIME_NO_DEFAULT_CONSTRUCTOR)
	/// Default constructor constructs to not_a_date_time.
	Time()
	{}
#endif // DATE_TIME_NO_DEFAULT_CONSTRUCTOR

public:

	MIRA_SPLIT_REFLECT_MEMBER

	template<typename Reflector>
	void reflectRead(Reflector& r)
	{
		int64 t = toUnixNS();
		r.delegate(t);
	}

	template<typename Reflector>
	void reflectWrite(Reflector& r)
	{
		int64 t;
		r.delegate(t);
		*this = Time::fromUnixNS(t);
	}


public:

	/**
	 * Returns the current utc based time
	 * @return The current time.
	 */
	static Time now()
#ifdef MIRA_LINUX
	{
		return boost::posix_time::microsec_clock::universal_time();
	}
#endif
#ifdef MIRA_WINDOWS
	; // see Time.C for implementation
#endif

	/**
	 * Returns 9999/12/31 23:59:59.999999
	 */
	static Time eternity()
	{
		return Time(boost::posix_time::max_date_time);
	}

	/**
	 * Returns an invalid time
	 */
	static Time invalid()
	{
		return Time(boost::posix_time::not_a_date_time);
	}

	/**
	 * Returns the unix epoch 1.1.1970 0:0:0.000
	 * @return Unix epoch time.
	 */
	static Time unixEpoch()
	{
		static Time t1970 = Time(Date(1970,Jan,1), Duration(0,0,0));
		return t1970;
	}

	/**
	 * Converts the current time to a unix timestamp in seconds
	 * @return timestamp in seconds.
	 */
	uint32 toUnixTimestamp() const
	{
		return (uint32)(*this - Time::unixEpoch()).totalSeconds();
	}

	/**
	 * Creates a time representation out of a unix timestamp
	 * @param[in] seconds The unix timestamp in seconds.
	 */
	static Time fromUnixTimestamp(uint32 seconds)
	{
		return Time::unixEpoch() + Duration::seconds(seconds);
	}

	/**
	 * Converts the current time to a unix timestamp in nanoseconds
	 * @return timestamp in nanoseconds.
	 */
	uint64 toUnixNS() const
	{
		return (uint64)(*this - Time::unixEpoch()).totalNanoseconds();
	}

	/**
	 * Creates a time representation out of a unix timestamp
	 * @param[in] nanoseconds The unix timestamp in nanoseconds.
	 */
	static Time fromUnixNS(uint64 nanoseconds)
	{
		return Time::unixEpoch() + Duration::nanoseconds(nanoseconds);
	}

	/**
	 * Converts to local time zone based on the settings of the machine.
	 * @return The time in local time zone.
	 */
	Time toLocal() const
	{
		typedef boost::date_time::c_local_adjustor<Base> l;
		return l::utc_to_local(*this);
	}

	/**
	 * Converts to given time zone.
	 * @param[in] zone The time zone to convert to e.g. "CET+1"
	 * (= name CET, 1 hour ahead of GMT).
	 * @return The time in the requested time zone.
	 */
	Time toTimeZone(const std::string& zone)
	{
		boost::local_time::time_zone_ptr tz(new boost::local_time::posix_time_zone(zone));
		boost::local_time::local_date_time dt(*this, tz);
		return dt.local_time();
	}

	/**
	 * Returns true if this contains a valid time
	 * @return true if this is a valid time, false otherwise
	 */
	bool isValid() const
	{
		return !this->is_not_a_date_time();
	}

public:

	/** @name Operators */
	//@{
	Duration operator-(const Time& t) const
	{
		return Base::operator-(t);
	}
	Time operator+(const DateDuration& d) const
	{
		return Base::operator+(d);
	}
	Time operator+=(const DateDuration& d)
	{
		return Base::operator+=(d);
	}
	Time operator-(const DateDuration& d) const
	{
		return Base::operator-(d);
	}
	Time operator-=(const DateDuration& d)
	{
		return Base::operator-=(d);
	}
	Time operator+(const Duration& d) const
	{
		return Base::operator+(d);
	}
	Time operator+=(const Duration& d)
	{
		return Base::operator+=(d);
	}
	Time operator-(const Duration& d) const
	{
		return Base::operator-(d);
	}
	Time operator-=(const Duration& d)
	{
		return Base::operator-=(d);
	}
	//@}
};

template<typename SerializerTag>
class IsTransparentSerializable<Time,SerializerTag> : public std::true_type {};

template<typename SerializerTag>
class IsTransparentSerializable<Duration,SerializerTag> : public std::true_type {};

///////////////////////////////////////////////////////////////////////////////

// non intrusive reflect for Date

template<typename Reflector>
void reflectRead(Reflector& r, Date& date) {
	Time t = Time(date);
	r.delegate(t);
}

template<typename Reflector>
void reflectWrite(Reflector& r, Date& date) {
	 Time t;
	 r.delegate(t);
	 date = t.date();
}
MIRA_SPLIT_REFLECT(Date)
template<typename SerializerTag>
class IsTransparentSerializable<Date,SerializerTag> : public std::true_type {};

///////////////////////////////////////////////////////////////////////////////

}

#endif
