/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Stamped.h
 *    Mix in for adding a timestamp to data types.
 *
 * @author Tim Langner, Erik Einhorn
 * @date   2010/08/18
 */

#ifndef _MIRA_STAMPED_H_
#define _MIRA_STAMPED_H_

#include <serialization/adapters/boost/shared_ptr.hpp>

#include <utils/Time.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * The common header for all stamped data. It contains the time stamp, the
 * frame ID and the sequence ID.
 */
class StampedHeader
{
public:
	/**
	 *  Default constructor, that does not initialize time or
	 *  frame id. Sets sequence id to 0.
	 */
	StampedHeader() : sequenceID(0) {}

	/**
	 * Constructs the header from its time stamp and an
	 * optional sequence id.
	 */
	StampedHeader(const Time& iTimestamp, uint32 iSequenceID = 0) :
		timestamp(iTimestamp),
		sequenceID(iSequenceID) {}

	/**
	 * Constructs the header from its time stamp, the frame id and an
	 * optional sequence id.
	 */
	StampedHeader(const Time& iTimestamp,
	              const std::string& iFrameID, uint32 iSequenceID = 0) :
		timestamp(iTimestamp),
		frameID(iFrameID),
		sequenceID(iSequenceID) {}

public:

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("Timestamp",  timestamp, "The timestamp", Time::now());
		r.member("FrameID",    frameID,
		         "The id of the transformation frame, this data belongs to",
		         "");
		r.member("SequenceID", sequenceID, "A user defined sequence ID", 0);
	}

public:

	/// The time stamp when the data was obtained
	Time timestamp;

	/// The unique id of the transform frame that this data is assigned to
	/// (if any, otherwise empty)
	std::string frameID;

	/// A user defined sequence ID.
	uint32 sequenceID;
};

/**
 * Mix in for adding a time stamp, an optional frame id and an optional
 * sequence id to data types like Pose, RangeScan, etc.
 *
 * The data is stamped in a temporal and spatial way. The time stamp indicates
 * the moment in time the data was created, measured or computed. The frame id
 * specifies the transformation frame, the data belongs to. The sequence id can
 * be used to track the origin or creator of the data.
 *
 * \code
 * class MyClass
 * {
 * public:
 *     void foo();
 * }
 *
 * Stamped<MyClass> myStampedObject;
 *
 * myStampedObject.foo();
 *
 * \endcode
 *
 * This class also is specialized for primitive data types. If Stamped
 * is used on primitive types like int, float, etc. it behaves like
 * the @ref StampedPrimitive class.
 *
 * \code
 * Stamped<int> myStampedInt;
 *
 * myStampedInt = 123;
 * cout << myStampedInt.value;
 * cout << (int)myStampedInt;
 * \endcode
 */
template <typename T>
class Stamped : public StampedHeader, public T
{
public:

	typedef T value_type ;

public:

	/**
	 *  Default constructor, that does not initialize time or
	 *  frame id. Sets sequence id to 0.
	 */
	Stamped() {}

	/**
	 * Constructs the @ref Stamped data from its time stamp and the
	 * actual data. The frame id will be empty, hence the data is not
	 * assigned to any transformation frame.
	 */
	Stamped(const T& iData, const Time& iTimestamp, uint32 iSequenceID = 0) :
		StampedHeader(iTimestamp, iSequenceID), T(iData) {}

	/**
	 * Constructs the @ref Stamped data from its time stamp, the frame id it
	 * belongs to and the actual data.
	 */
	Stamped(const T& iData, const Time& iTimestamp,
	        const std::string& iFrameID, uint32 iSequenceID = 0) :
		StampedHeader(iTimestamp,  iFrameID, iSequenceID),
		T(iData) {}

public:

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		StampedHeader::reflect(r);
		r.member("Value", (T&)*this, "The value");
	}

public:

	/**
	 * Returns a read-write reference to the underlying data.
	 */
	T& value() { return *this; }

	/**
	 * Returns a read-only reference to the underlying data.
	 */
	const T& value() const { return *this; }

	/**
	 * Returns a read-write reference to the underlying data representation.
	 */
	T& internalValueRep() { return *this; }

	/**
	 * Returns a read-only reference to the underlying data representation.
	 */
	const T& internalValueRep() const { return *this; }

};

///////////////////////////////////////////////////////////////////////////////

class Object; // forward declaration of mira::Object in factory/Object.h

/**
 * Stamped class specialization for polymorphic pointers.
 * This specialization can only be used for pointers of polymorphic classes that
 * are derived from mira::Object. The reason is, that this class internally
 * stores a pointer to mira::Object only and performs a dynamic_cast when the
 * concrete pointer T* is requested. This is necessary to be able to cast
 * a Stamped<Object*> class to any other Stamped<T*> class.
 *
 * Internally, this class uses a boost::shared_ptr to store the pointer,
 * hence after assigning a pointer to Stamped<T*> it will take ownership of
 * it and handles its destruction.
 */
template <typename T>
class Stamped<T*> : public StampedHeader
{
	static_assert(std::is_base_of<mira::Object,T>::value, 
	              "Pointers that are used with Stamped<T> must be of polymorphic"
	              " classes that are derived from mira::Object. Pointers to other"
	              " classes cannot be used with Stamped<T>.");
public:

	typedef T* value_type ;

public:

	/**
	 *  Default constructor, no time is set and data pointer
	 *  is null.
	 */
	Stamped() {}

	/**
	 * Constructs the @ref Stamped data from its time stamp and the
	 * actual data.
	 */
	Stamped(T* iData, const Time& iTimestamp, uint32 iSequenceID = 0) :
		StampedHeader(iTimestamp, iSequenceID), mPointer(iData) {}

	/**
	 * Constructs the @ref Stamped data from its time stamp, the
	 * frame id and the actual data.
	 */
	Stamped(T* iData, const Time& iTimestamp, const std::string& iFrameID,
	        uint32 iSequenceID = 0) :
	        StampedHeader(iTimestamp, iFrameID, iSequenceID), mPointer(iData) {}

public:

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		StampedHeader::reflect(r);
		r.member("Value", mPointer, "The value");
	}

public:

	/**
	 * Returns the underlying pointer.
	 */
	T* value() { return dynamic_cast<T*>(mPointer.get()); }

	/**
	 * Returns the underlying pointer as constant pointer
	 */
	const T* value() const { return dynamic_cast<T*>(mPointer.get()); }

	/// Cast operator that casts this into T
	operator const T*() const { return value(); }

	/// Cast operator that casts this into T
	operator T*() { return value(); }

	/// Assignment operator that can assign T to this
	T* operator=(T* iValue) {mPointer.reset(iValue);  return iValue; }

	/**
	 * Returns a read-write reference to the underlying data representation.
	 */
	boost::shared_ptr<Object>& internalValueRep() { return mPointer; }

	/**
	 * Returns a read-only reference to the underlying data representation.
	 */
	const boost::shared_ptr<Object>& internalValueRep() const { return mPointer; }

public:

	// Note: the following casts look dangerous, however, they're safe since
	// all Stamped<T*> store a pointer to Object* and hence all of them are
	// equal. Moreover, our accessor methods above take care of type safety
	// using the dynamic_casts when accessing the underlying data.

	/// Cast operator that can cast any Stamped<T*> into another Stamped<U*>
	template <typename U>
	operator const Stamped<U*>& () const {
		static_assert(sizeof(Stamped<U*>)==sizeof(Stamped<T*>), 
		              "The following is safe as long as all Stamped<T*> have the same size");
		return reinterpret_cast<const Stamped<U*>&>(*this);
	}

	/// Cast operator that can cast any Stamped<T*> into another Stamped<U*>
	template <typename U>
	operator Stamped<U*>& () {
		static_assert(sizeof(Stamped<U*>)==sizeof(Stamped<T*>), 
		              "The following is safe as long as all Stamped<T*> have the same size");
		return reinterpret_cast<Stamped<U*>&>(*this);
	}

private:

	/// The pointer
	boost::shared_ptr<Object> mPointer;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Stamped class for primitive types like int, float, etc.
 * You should never use this class directly. Use @ref Stamped instead
 * and specify the primitive type as template parameter, e.g.
 * \code
 * Stamped<int> myStampedInt;
 *
 * myStampedInt = 123;
 * cout << myStampedInt.value;
 * cout << (int)myStampedInt;
 * \endcode
 *
 * @see Stamped
 */
template <typename T>
class StampedPrimitive : public StampedHeader
{
public:

	typedef T value_type ;

public:

	/// Default constructor, that does not initializes anything.
	StampedPrimitive() {}

	/**
	 * Constructs the @ref Stamped data from its time stamp and the
	 * actual data.
	 */
	StampedPrimitive(const T& iData, const Time& iTimestamp,
	                 uint32 iSequenceID = 0) :
	    StampedHeader(iTimestamp, iSequenceID), mValue(iData) {}

	/**
	 * Constructs the @ref Stamped data from its time stamp, the
	 * frame id and the actual data.
	 */
	StampedPrimitive(const T& iData, const Time& iTimestamp,
	                 const std::string& iFrameID, uint32 iSequenceID = 0) :
	    StampedHeader(iTimestamp, iFrameID, iSequenceID), mValue(iData) {}

public:

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		StampedHeader::reflect(r);
		r.member("Value", mValue, "The value");
	}

public:

	/**
	 * Returns a read-write reference to the underlying data.
	 */
	T& value() { return mValue; }

	/**
	 * Returns a read-only reference to the underlying data.
	 */
	const T& value() const { return mValue; }

	/// Cast operator that casts this into T
	operator const T&() const { return mValue; }

	/// Cast operator that casts this into T
	operator T&() { return mValue; }

	/// Assignment operator that can assign T to this
	const T& operator=(const T& iValue) { return mValue = iValue; }

	/**
	 * Returns a read-write reference to the underlying data representation.
	 */
	T& internalValueRep() { return mValue; }

	/**
	 * Returns a read-only reference to the underlying data representation.
	 */
	const T& internalValueRep() const { return mValue; }

private:

	/// The primitive data
	T mValue;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Macro for specializing template class Stamped for primitive types 
 */
#define MIRA_DECLARE_STAMPED_PRIMITIVE_SPECIALIZATION(T)                      \
template <>                                                                  \
class Stamped<T> : public StampedPrimitive<T>                                \
{                                                                            \
public:                                                                      \
	Stamped() {}                                                             \
	Stamped(const T& iData, const Time& iTimestamp, uint32 iSequenceID=0) :  \
	    StampedPrimitive<T>(iData, iTimestamp,iSequenceID) {}                \
	Stamped(const T& iData, const Time& iTimestamp,                          \
	        const std::string& iFrameID, uint32 iSequenceID=0) :             \
	        StampedPrimitive<T>(iData, iTimestamp, iFrameID, iSequenceID) {} \
};

/**
 * Declare stamped classes for all standard data types including std::string
 */
MIRA_DECLARE_STAMPED_PRIMITIVE_SPECIALIZATION(bool)
MIRA_DECLARE_STAMPED_PRIMITIVE_SPECIALIZATION(uint8)
MIRA_DECLARE_STAMPED_PRIMITIVE_SPECIALIZATION(int8)
MIRA_DECLARE_STAMPED_PRIMITIVE_SPECIALIZATION(uint16)
MIRA_DECLARE_STAMPED_PRIMITIVE_SPECIALIZATION(int16)
MIRA_DECLARE_STAMPED_PRIMITIVE_SPECIALIZATION(uint32)
MIRA_DECLARE_STAMPED_PRIMITIVE_SPECIALIZATION(int32)
MIRA_DECLARE_STAMPED_PRIMITIVE_SPECIALIZATION(uint64)
MIRA_DECLARE_STAMPED_PRIMITIVE_SPECIALIZATION(int64)
MIRA_DECLARE_STAMPED_PRIMITIVE_SPECIALIZATION(char)
MIRA_DECLARE_STAMPED_PRIMITIVE_SPECIALIZATION(float)
MIRA_DECLARE_STAMPED_PRIMITIVE_SPECIALIZATION(double)
MIRA_DECLARE_STAMPED_PRIMITIVE_SPECIALIZATION(std::string)

///////////////////////////////////////////////////////////////////////////////

/**
 * Helper function to create stamped data.
 */
template <typename T>
Stamped<T> makeStamped(const T& value, const Time& timestamp = Time::now(),
                       const std::string& nodeId = std::string(),
                       uint32 sequenceID = 0)
{
	return Stamped<T>(value, timestamp, nodeId, sequenceID);
}

/**
 * Helper to extract the header from stamped data.
 */
template <typename T>
StampedHeader& stampedHeader(Stamped<T>& data) {
	return data;
}

template <typename T>
const StampedHeader& stampedHeader(const Stamped<T>& data) {
	return data;
}


///////////////////////////////////////////////////////////////////////////////

}

#endif
