/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file StlCollections.h
 *    Helpers for serialization and deserialization of STL containers
 *
 * @author Erik Einhorn
 * @date   2010/07/16
 */

#ifndef _MIRA_STLCOLLECTIONS_H_
#define _MIRA_STLCOLLECTIONS_H_

#include <utils/Foreach.h>

#include <serialization/ReflectCollectionCount.h>

#include <serialization/IsPointerOrSharedPointer.h>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

namespace serialization { // our private namespace

//////////////////////////////////////////////////////////////////////////////

// SEQUENCES: list, vector, deque

/**
 * Reflects generic sequential containers like vectors, list, deque
 * (Read Only / Serialization)
 *
 * The resulting format is like:
 * \code
 *   item ...
 *   item ...
 *   item ...
 *   ...
 * \endcode
 */
template<typename Reflector, typename Container>
struct ReflectReadSeq
{
	typedef typename Container::iterator iterator;
	typedef typename Container::value_type type;

	static void reflect(Reflector& r, Container& c)
	{
		// store the size first
		uint32 count = c.size();
		ReflectCollectionCount<Reflector,Container>::reflect(r, count);

		// store each item
		int id=0;
		foreach(type& v, c)
		{
			// specifies the id string of the element, since the
			// name "item" is the same for all elements
			MIRA_PROPERTY_WITH_ID(r, "item", "item["+toString(id)+"]", v, "");
			++id;
		}
	}
};

/**
 * Reflects generic sequencial containers like vectors, list, deque
 * (Write Only / Deserialization)
 *
 * The resulting format is like:
 * \code
 *   item ...
 *   item ...
 *   item ...
 *   ...
 * \endcode
 *
 */
template<typename Reflector, typename Container>
struct ReflectWriteSeq
{
	typedef typename Container::iterator iterator;
	typedef typename Container::value_type type;

	static void reflect(Reflector& r, Container& c)
	{
		c.clear(); // clear previous content

		// restore the size first
		uint32 count;
		ReflectCollectionCount<Reflector,Container>::reflect(r, count);


		// reserve the space and create the elements
		c.resize(count);

		// restore each item
		int id=0;
		foreach(type& v, c)
		{
			// specifies the id string of the element, since the
			// name "item" is the same for all elements
			MIRA_PROPERTY_WITH_ID(r, "item", "item["+toString(id)+"]", v, "");
			++id;
		}
	}
};

// SET: set, multiset
/**
 * reflects generic set containers like set, multiset
 *
 * The resulting format is like:
 * \code
 *   item ...
 *   item ...
 *   item ...
 *   ...
 * \endcode

 */
template<typename Reflector, typename Container>
struct ReflectReadSet
{
	typedef typename Container::iterator iterator;
	typedef typename Container::value_type type;

	static void reflect(Reflector& r, Container& c)
	{
		// store the size first
		uint32 count = c.size();
		ReflectCollectionCount<Reflector,Container>::reflect(r, count);

		// store each item
		int id=0;
		foreach(const type& cv, c)
		{
			type& v = const_cast<type&>(cv);

			if(IsPointerOrSharedPointer<type>::value) {
				MIRA_MEMBER_WITH_ID(r, "item", "item["+toString(id)+"]", v, "");
			} else {
				// deserializing references to non-pointer-type keys will not work !!!
				MIRA_MEMBER_WITH_ID(r, "item", "item["+toString(id)+"]", v, "", REFLECT_CTRLFLAG_TEMP_TRACKING);
				// MIRA_MEMBER_WITH_ID(r, "item", "item["+toString(id)+"]", v, "", REFLECT_CTRLFLAG_NO_TRACKING);
			}
			++id;
		}
	}
};

/**
 * Derializes generic set containers like set, multiset
 *
 * The resulting format is like:
 * \code
 *   item ...
 *   item ...
 *   item ...
 *   ...
 * \endcode
 */
template<typename Reflector, typename Container>
struct ReflectWriteSet
{
	typedef typename Container::iterator iterator;
	typedef typename Container::value_type type;

	static void reflect(Reflector& r, Container& c)
	{
		c.clear(); // clear previous content

		// restore the size first
		uint32 count;
		ReflectCollectionCount<Reflector,Container>::reflect(r, count);

		iterator hint = c.begin();

		// restore each item and insert into the set
		for(uint32 id=0; id<count; ++id)
		{
			// reflect the value
			type v;

			if(IsPointerOrSharedPointer<type>::value) {
				MIRA_MEMBER_WITH_ID(r, "item", "item["+toString(id)+"]", v, "");
			} else {
				// we have to deserialize the key to a temp object, cannot restore references to it !!!
				MIRA_MEMBER_WITH_ID(r, "item", "item["+toString(id)+"]", v, "", REFLECT_CTRLFLAG_TEMP_TRACKING);
				// however, if we stored it during serialization, we have to keep it in the tracking store
				// MIRA_MEMBER_WITH_ID(r, "item", "item["+toString(id)+"]", v, "");
			}
			// add the value to the container
			hint = c.insert(hint, v);

			// TODO: if we want object tracking for sets, we must relocate the object and its
			// subobjects here
		}
	}
};

// MAP: map, multimap, hash_map, etc
/**
 * reflects generic set containers like set, multiset
 *
 * The resulting format is like:
 * \code
 *   key
 *   item ...
 *   key
 *   item ...
 *   key
 *   item ...
 *   ...
 * \endcode
 */
template<typename Reflector, typename Container>
struct ReflectReadMap
{
	typedef typename Container::iterator    iterator;
	typedef typename Container::value_type  value_type;
	typedef typename Container::key_type    key_type;

	static void reflect(Reflector& r, Container& c)
	{
		// store the size first
		uint32 count = c.size();
		ReflectCollectionCount<Reflector,Container>::reflect(r, count);


		// store each item
		int id=0;
		foreach(value_type& p, c)
		{
			if(IsPointerOrSharedPointer<key_type>::value) {
				MIRA_MEMBER_WITH_ID(r, "key", "key["+toString(id)+"]", p.first, "");
			} else {
				// deserializing references to non-pointer-type keys will not work !!!
				MIRA_MEMBER_WITH_ID(r, "key", "key["+toString(id)+"]", const_cast<key_type&>(p.first),
				                    "", REFLECT_CTRLFLAG_TEMP_TRACKING);
				// MIRA_MEMBER_WITH_ID(r, "key", "key["+toString(id)+"]", const_cast<key_type&>(p.first),
				//                     "", REFLECT_CTRLFLAG_NO_TRACKING);
			}

			// the values can be reflected/deserialized with tracking
			MIRA_PROPERTY_WITH_ID(r, "item", "item["+toString(id)+"]", p.second, "");

			++id;
		}
	}
};

/**
 * reflects generic map containers like map, multimap
 *
 * The resulting format is like:
 * \code
 *   key
 *   item ...
 *   key
 *   item ...
 *   key
 *   item ...
 *   ...
 * \endcode
 */
template<typename Reflector, typename Container>
struct ReflectWriteMap
{
	typedef typename Container::iterator iterator;
	typedef typename Container::value_type   value_type;
	typedef typename Container::key_type     key_type;
	typedef typename Container::mapped_type  mapped_type;

	static void reflect(Reflector& r, Container& c)
	{
		c.clear(); // clear previous content

		// restore the size first
		uint32 count;
		ReflectCollectionCount<Reflector,Container>::reflect(r, count);

		iterator hint = c.begin();

		// restore each item and insert into the map
		for(uint32 id=0; id<count; ++id)
		{
			// reflect key
			key_type key;

			if(IsPointerOrSharedPointer<key_type>::value) {
				MIRA_MEMBER_WITH_ID(r, "key", "key["+toString(id)+"]", key, "");
			} else {
				// we have to deserialize the key to a temp object, cannot restore references to it !!!
				MIRA_MEMBER_WITH_ID(r, "key", "key["+toString(id)+"]", key, "", REFLECT_CTRLFLAG_TEMP_TRACKING);
				// however, if we stored it during serialization, we have to keep it in the tracking store
				// MIRA_MEMBER_WITH_ID(r, "key", "key["+toString(id)+"]", key, "");
			}


			// insert a dummy value with the key
			iterator it = c.insert(hint, value_type(key, mapped_type()));
			hint = it; // the iterator will be insertion hint for next element

			// reflect the value directly into the map item

			// the values can be reflected/deserialized with tracking
			MIRA_PROPERTY_WITH_ID(r, "item", "item["+toString(id)+"]", it->second, "");
		}
	}
};

//////////////////////////////////////////////////////////////////////////////

} //namespace

//////////////////////////////////////////////////////////////////////////////

} //namespace

#endif
