/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file JSONSerializer.h
 *    Serializer and Deserializer for JSON format.
 *
 * @author Tim Langner
 * @date   2010/11/01
 */

#ifndef _MIRA_JSONSERIALIZER_H_
#define _MIRA_JSONSERIALIZER_H_

#include <iostream>
#include <list>
#include <algorithm>
#include <stack>

#include <serialization/Serializer.h>
#include <serialization/Deserializer.h>

#include <json/JSON.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup SerializationModule
 *
 * Serializer for serializing objects in JSON format.
 *
 * Example usage:
 * \code
 * #include <serialization/adapters/std/vector>
 * // the data to be serialized
 * std::vector<int> myValue;
 * ...
 *
 * // create a JSONSerializer that writes into a JSON value
 * JSONSerializer serializer;
 *
 * // serialize the data and return the JSON value containing the data
 * json::Value v = serializer.serialize(myValue);
 *
 * // write the JSON value to the console
 * json::write(v, std::cout, true);
 * \endcode
 *
 * For the deserialization of JSON content see
 * @ref mira::JSONDeserializer "JSONDeserializer".
 *
 * @see @ref SerializationPage, JSONDeserializer
 */
class JSONSerializer : public Serializer<JSONSerializer>
{
	typedef Serializer<JSONSerializer> Base;

public:
	/**
	 * Construct a serializer.
	 * @param readOnly If true it also visits ro properties and creates
	 *                 JSON output that should NOT be used to deserialize an object from.
	 */
	JSONSerializer(bool readOnly = false) :
		mReadOnly(readOnly) {}

	/// serialize value and return its json representation
	template <typename T>
	json::Value serialize(const T& value)
	{
		mFirstComplex = true;
		mValue = json::Value();
		Base::serialize("Value", value);
		return mValue;
	}

	json::Value serialize(const json::Value& value)
	{
		return value;
	}

	int version(int version) {
		// serialize as atomic
		//atomic(version);
		mValue.get_obj()["@version"] = json::Value(json::cast(version));
		return version;
	}

	template<typename T>
	void atomic(T& member)
	{
		// member of a currently serialized object -> add member
		if ( mValue.type() == json_spirit::obj_type )
			mValue.get_obj()[getCurrentMemberMeta().id] = json::Value(json::cast(member));
		// item of a currently serialized array -> add item
		else if ( mValue.type() == json_spirit::array_type )
			mValue.get_array().push_back(json::Value(json::cast(member)));
		// no currently serialized value -> set member as new value
		else
			mValue = json::Value(json::cast(member));
	}

	void atomic(json::Value& member)
	{
		// member of a currently serialized object -> add member
		if ( mValue.type() == json_spirit::obj_type )
			mValue.get_obj()[getCurrentMemberMeta().id] = member;
		// item of a currently serialized array -> add item
		else if ( mValue.type() == json_spirit::array_type )
			mValue.get_array().push_back(member);
		// no currently serialized value -> set member as new value
		else
			mValue = member;
	}


	template<typename T>
	void object(T& member)
	{
		// in json the outer most object (the root object) has no name
		if ( mFirstComplex )
		{
			mValue = json::Object();
			mFirstComplex = false;
			if(!mPointerClass.empty()) {
				mValue.get_obj()["@class"] = json::Value(mPointerClass);
				mPointerClass.clear();
			}
			Base::object(member);
		}
		else
		{
			json::Value oldValue = mValue;
			mValue = json::Object();
			if(!mPointerClass.empty()) {
				mValue.get_obj()["@class"] = json::Value(mPointerClass);
				mPointerClass.clear();
			}
			Base::object(member);
			if ( oldValue.type() == json_spirit::obj_type )
				oldValue.get_obj()[getCurrentMemberMeta().id] = mValue;
			else if ( oldValue.type() == json_spirit::array_type )
					oldValue.get_array().push_back(mValue);
			mValue = oldValue;
		}
	}

	template<typename T>
	void collection(T& member)
	{
		if ( mFirstComplex )
		{
			mValue = json::Array();
			mFirstComplex = false;
			Base::object(member);
		}
		else
		{
			json::Value oldValue = mValue;
			mValue = json::Array();
			Base::object(member);
			if ( oldValue.type() == json_spirit::obj_type )
				oldValue.get_obj()[getCurrentMemberMeta().id] = mValue;
			else
				if ( oldValue.type() == json_spirit::array_type )
					oldValue.get_array().push_back(mValue);
			mValue = oldValue;
		}
	}

	void pointerReference(int referencedObjectID) {
		json::Value val = json::Object();
		val.get_obj()["@ref"] = json::Value(this->getHumanReadableFullID(referencedObjectID));
		if (mValue.type() == json_spirit::obj_type )
			mValue.get_obj()[getCurrentMemberMeta().id] = val;
		else if (mValue.type() == json_spirit::array_type )
			mValue.get_array().push_back(val);
	}

	void pointerWithClassType(const std::string& type) {
		mPointerClass = type;
	}

	void pointerNull() {
		if ( mValue.type() == json_spirit::obj_type )
			mValue.get_obj()[getCurrentMemberMeta().id] = json::Value();
		else if ( mValue.type() == json_spirit::array_type )
			mValue.get_array().push_back(json::Value());
		else
			mValue = json::Value();
	}

	template<typename T>
	void roproperty(const char* name, const T& member, const char* comment,
	                PropertyHint&& hint = PropertyHint(),
	                ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		if (!mReadOnly)
			return;
		// cast away constness, this is okay, since read only properties we
		// will never write to the data
		property(name, const_cast<T&>(member),
		         comment, std::move(hint), flags);
	}

	template<typename T>
	void roproperty(const char* name, const std::string& id, const T& member,
	                const char* comment, PropertyHint&& hint = PropertyHint(),
	                ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		if (!mReadOnly)
			return;
		// cast away constness, this is okay, since read only properties we
		// will never write to the data
		property(name, id, const_cast<T&>(member),
		         comment, std::move(hint), flags);
	}

	template<typename T>
	void roproperty(const char* name, Getter<T> getter, const char* comment,
	                PropertyHint&& hint = PropertyHint(),
	                ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		if (!mReadOnly)
			return;
		auto a = makeAccessor(getter, NullSetter<T>());
		property(name, a, comment, std::move(hint), flags);
	}

private:
	bool mFirstComplex;
	bool mReadOnly;
	json::Value mValue;
	std::string mPointerClass;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup SerializationModule
 *
 * Deserializer for serializing objects from JSON format.
 *
 * Example usage:
 * \code
 * // load the json content
 * json::Value v;
 * json::read(..., v);
 *
 * // create the JSON deserializer
 * JSONDeserializer deserializer(v);
 *
 * // the object that will be filled with the content
 * std::vector<int> myValue;
 *
 * // deserialize the value
 * deserializer.deserialize(myValue);
 *
 * // myValue is now filled with the content that was stored in the JSON value
 * \endcode
 *
 * For the serialization of JSON content see
 * @ref mira::JSONSerializer "JSONSerializer".
 *
 * @see @ref SerializationPage, JSONSerializer
 */
class JSONDeserializer : public Deserializer<JSONDeserializer>
{
	typedef Deserializer<JSONDeserializer> Base;
public:

	JSONDeserializer(const json::Value& value) :
		mValue(&value)
	{
	}

	template <typename T>
	void deserialize(T& value)
	{
		mIndex = 0;
		mFirstComplex = true;
		Base::deserialize("Value",value);
	}

	void deserialize(json::Value& value)
	{
		value = *mValue;
	}

	const json::Value* getMember(const json::Value* value, const std::string& name)
	{
		if (value->get_obj().count(name) == 0)
			MIRA_THROW(XMemberNotFound, "Member '" << name << "' is missing"); // hence we have to abort with an exception
		return &value->get_obj().at(name);
	}
	
	const json::Value* getItem(const json::Value* value, std::size_t index)
	{
		if (value->get_array().size() <= index)
			MIRA_THROW(XMemberNotFound, "Item[" << index << "] is missing"); // hence we have to abort with an exception
		return &value->get_array()[index];
	}

	int version(int version) {
		if (mValue->get_obj().count("@version") == 0)
			return 0;
		return mValue->get_obj().at("@version").get_value<json::TypeTrait<int>::type >();
	}

	template<typename T>
	void atomic(T& member)
	{
		if ( mValue->type() == json_spirit::obj_type )
		{
			try
			{
				member = (T)getMember(mValue, mCurrentMemberID)->get_value<typename json::TypeTrait<T>::type >();
			}
			catch(std::exception& ex)
			{
				MIRA_THROW(XMemberNotFound, "Error deserializing member '" << mCurrentMemberID
				           << "' (" << getCurrentMemberFullID()
				           << ") from object='" << json::write(*mValue)
				           << "' as " << typeName<T>() << ": " << ex.what());
			}
		}
		else if ( mValue->type() == json_spirit::array_type )
		{
			try
			{
				member = (T)getItem(mValue, mIndex++)->get_value<typename json::TypeTrait<T>::type >();
			}
			catch(std::exception& ex)
			{
				MIRA_THROW(XMemberNotFound, "Error deserializing item[" << mIndex-1
				           << "] (" << getCurrentMemberFullID()
				           << ") from collection='" << json::write(*mValue)
				           << "' as " << typeName<T>() << ": " << ex.what());
			}
		}
		else
		{
			try
			{
				member = (T)mValue->get_value<typename json::TypeTrait<T>::type>();
			}
			catch(std::exception& ex)
			{
				MIRA_THROW(XMemberNotFound, "Error deserializing atomic (" << getCurrentMemberFullID()
				           << ") with content='" << json::write(*mValue) << "' as " << typeName<T>() << ": " << ex.what());
			}
		}
	}

	void atomic(json::Value& member)
	{
		if ( mValue->type() == json_spirit::obj_type )
			member = *getMember(mValue, mCurrentMemberID);
		else if ( mValue->type() == json_spirit::array_type )
			member = *getItem(mValue, mIndex++);
		else
			member = *mValue;
	}

	template<typename T>
	void object(T& member)
	{
		if ( mFirstComplex )
		{
			mFirstComplex = false;
			Base::object(member);
			return;
		}
		const json::Value* oldValue = mValue;
		std::size_t oldIndex = mIndex;
		if ( mValue->type() == json_spirit::obj_type )
		{
			mValue = getMember(mValue, mCurrentMemberID);
		}
		else if ( mValue->type() == json_spirit::array_type )
		{
			mValue = getItem(mValue, oldIndex);
			mIndex = 0;
		}

		Base::object(member);

		mValue = oldValue;
		mIndex = oldIndex+1;
	}

	template<typename T>
	void collection(T& member)
	{
		if ( mFirstComplex )
		{
			mFirstComplex = false;
			Base::object(member);
			return;
		}
		const json::Value* oldValue = mValue;
		std::size_t oldIndex = mIndex;
		if ( mValue->type() == json_spirit::obj_type )
		{
			mValue = getMember(mValue, mCurrentMemberID);
			mIndex = 0;
		}
		else if ( mValue->type() == json_spirit::array_type )
		{
			mValue = getItem(mValue, oldIndex);
			mIndex = 0;
		}

		Base::object(member);

		mValue = oldValue;
		mIndex = oldIndex+1;
	}

	const json::Array* getCollection()
	{
		const json::Array* array;
		try
		{
			array = &mValue->get_array();
		}
		catch(std::exception& ex)
		{
			MIRA_THROW(XInvalidConfig, "Error deserializing collection (" << getCurrentMemberFullID()
			           << ") from json::Value='" << json::write(*mValue) << "': " << ex.what());
		}
		return array;
	}

	template<typename T>
	void pointer(T* &pointer)
	{
		const json::Value* value;
		if ( mFirstComplex )
			value = mValue;
		else
		{
			if ( mValue->type() == json_spirit::obj_type )
				value = getMember(mValue, mCurrentMemberID);
			else if ( mValue->type() == json_spirit::array_type )
				value = getItem(mValue, mIndex);
		}

		// do we have a null pointer?
		if(value->is_null()) {
			pointer = NULL;
			if (mValue->type() == json_spirit::array_type)
				mIndex++;
		}
		// do we have a reference?
		else {
			assert(value->type() == json_spirit::obj_type);
			auto it = value->get_obj().find("@ref");
			if(it!=value->get_obj().end()) {
				// we have a reference, so resolve it
				pointer = resolveReference<T>(it->second.get_value<std::string>());
			} else {
				// we have a "normal" pointer, so deserialize it
				Base::pointer(pointer);
			}
		}
	}

	std::string pointerClassType()
	{
		const json::Value* value;
		if ( mFirstComplex )
			value = mValue;
		else
		{
			if ( mValue->type() == json_spirit::obj_type )
			{
				try
				{
					value = getMember(mValue, mCurrentMemberID);
				}
				catch(XMemberNotFound& ex)
				{
					MIRA_RETHROW(ex, "Error getting pointer class type of member '"
					             << mCurrentMemberID << "' (" << getCurrentMemberFullID()
					             << ") from object='" << json::write(*mValue)<< "'");
				}
			}
			else if ( mValue->type() == json_spirit::array_type )
			{
				try
				{
					value = getItem(mValue, mIndex);
				}
				catch(XMemberNotFound& ex)
				{
					MIRA_RETHROW(ex, "Error getting pointer class type of item[" << mIndex
					             << "] (" << getCurrentMemberFullID()
					             << ") from collection='" << json::write(*mValue) << "'");
				}
			}
		}
		if(value->type() == json_spirit::obj_type)
		{
			auto it = value->get_obj().find("@class");
			if(it!=value->get_obj().end())
				return it->second.get_value<std::string>();
		}
		return "Unknown Class";
	}

	// hijack the invokeOverwrite method for adding a starting and closing tag
	// around the serialized data
	template<typename T>
	void invokeMemberOverwrite(T& member, const ReflectMemberMeta& meta)
	{
		std::string oldID = mCurrentMemberID;
		if(meta.id!=NULL)
			mCurrentMemberID = meta.id;
		try
		{
			Base::invokeMemberOverwrite(member, meta);
		}
		catch(...)
		{
			if(meta.id!=NULL)
				mCurrentMemberID = oldID;
			throw;
		}
		if(meta.id!=NULL)
			mCurrentMemberID = oldID;
	}

private:

private:
	bool mFirstComplex;
	std::size_t mIndex;
	const json::Value* mValue;
	std::string mCurrentMemberID;
};

namespace serialization { // our private namespace

///////////////////////////////////////////////////////////////////////////////

/**
 * Specialization for JSON  Serializer which does not write the
 * item count explicitly. The deserializer will count the
 * item nodes in the parent node to recover the item count, which
 * is much more user friendly, since the user does not need
 * to provide the count himself.
 */
template<typename Collection>
struct ReflectCollectionCount<JSONSerializer,Collection>
{
	static void reflect(JSONSerializer& r, uint32& ioCount)
	{
		// do nothing
	}
};
/**
 * Specialization for JSON  Deserializer which counts the
 * item nodes in the parent node to recover the item count, which
 * is much more user friendly, since the user does not need
 * to provide the count himself.
 */
template<typename Collection>
struct ReflectCollectionCount<JSONDeserializer,Collection>
{
	static void reflect(JSONDeserializer& r, uint32& ioCount)
	{
		const json::Array* a = r.getCollection();
		ioCount = (uint32)a->size();
	}
};

// specialization for maps
template<typename K, typename T, typename C, typename A>
struct ReflectCollectionCount<JSONDeserializer,std::map<K,T,C,A>>
{
	static void reflect(JSONDeserializer& r, uint32& ioCount)
	{
		const json::Array* a = r.getCollection();
		ioCount = (uint32)a->size() / 2; // in maps, we have key/value pairs per item
	}
};

template<typename K, typename T, typename C, typename A>
struct ReflectCollectionCount<JSONDeserializer,std::multimap<K,T,C,A>>
{
	static void reflect(JSONDeserializer& r, uint32& ioCount)
	{
		const json::Array* a = r.getCollection();
		ioCount = (uint32)a->size() / 2; // in maps, we have key/value pairs per item
	}
};

///////////////////////////////////////////////////////////////////////////////

}

//////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
