/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file EigenFormat.h
 *    Wrappers around eigen in/output to support streaming operators.
 *
 * @author Erik Einhorn
 * @date   2010/08/02
 */

#ifndef _MIRA_EIGENFORMAT_H_
#define _MIRA_EIGENFORMAT_H_

#ifndef Q_MOC_RUN
#include <boost/algorithm/string/trim.hpp>
#endif

#include <platform/Platform.h>

#if defined(MIRA_GNUC_VERSION)
# if MIRA_GNUC_VERSION >= 40600
#  pragma GCC diagnostic push
#  pragma GCC diagnostic ignored "-Wdeprecated-declarations"
# endif
#endif

#include <Eigen/Eigen>
#include <Eigen/StdVector>

#if defined(MIRA_GNUC_VERSION)
# if MIRA_GNUC_VERSION >= 40600
#  pragma GCC diagnostic pop
# endif
#endif

#include <stream/NumericalStream.h>
#include <error/Exceptions.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Base class for formatting eigen matrices. Defines also some basic formats.
 */
class EigenFormat
{
public:
	EigenFormat(Eigen::IOFormat format) : mFormat(format) {}

public:
	// some predefined formats.
	static Eigen::IOFormat matlab(int precision=4) {
		return Eigen::IOFormat(precision, 0, ", ", ";\n", "", "", "[", "]");
	}
	static Eigen::IOFormat clean(int precision=4) {
		return Eigen::IOFormat(precision, 0, ", ", "\n", "[", "]");
	}
	static Eigen::IOFormat eigen(int precision=-1) {
		return Eigen::IOFormat();
	}
	static Eigen::IOFormat python(int precision=4) {
		return Eigen::IOFormat(precision, 0, ", ", ",\n", "[", "]", "[", "]");
	}

protected:
	Eigen::IOFormat mFormat;
};

///////////////////////////////////////////////////////////////////////////////

///@cond INTERNAL

// general implementation for any Eigen::MatrixBase<Derived>
template <typename MatrixType>
struct EigenFormatSizeHelper
{
	static bool checkOrResize(MatrixType& matrix, int rows, int cols)
	{
		// in generic case just check if the size matches
		return (matrix.rows() == rows && matrix.cols() == cols);
	}
};

// specialization for Eigen::Matrix that resizes the matrix if it is dynamic
template<typename Scalar, int Rows, int Cols, int Options, int MaxRows, int MaxCols>
struct EigenFormatSizeHelper<Eigen::Matrix<Scalar, Rows, Cols, Options, MaxRows, MaxCols>>
{
	typedef Eigen::Matrix<Scalar, Rows, Cols, Options, MaxRows, MaxCols> Derived;
	typedef Eigen::MatrixBase<Derived> MatrixType;
	static bool checkOrResize(MatrixType& matrix, int rows, int cols)
	{
		Derived& m = static_cast<Derived&>(matrix);
		if (Derived::RowsAtCompileTime == Eigen::Dynamic ||
		    Derived::ColsAtCompileTime == Eigen::Dynamic )
			m.resize(rows, cols);
		return (matrix.rows() == rows && matrix.cols() == cols);
	}
};

///@endcond

///////////////////////////////////////////////////////////////////////////////

/**
 * Template class wrapping the serializing functionality of eigen
 * to support stream operators using a given format.
 * This variant is for read-only matrices (<< operator supported only)
 */
template <typename Derived>
class TEigenFormatReadOnly : public EigenFormat
{
private:

public:
	TEigenFormatReadOnly(Eigen::MatrixBase<Derived>& matrix, Eigen::IOFormat format) :
		EigenFormat(format), mMatrix(matrix) {}

	/**
	 * Output stream operator for writing a matrix to stream in a given format.
	 * @param s The output stream
	 * @param format The format class
	 * @return The stream
	 */
	friend std::ostream& operator<< (std::ostream & s,
	                                 const TEigenFormatReadOnly<Derived>& format)
	{
		format.print(s);
		return s;
	}

	/// not supported for read-only matrices
	friend std::istream& operator>> (std::istream & is,
	                                 TEigenFormatReadOnly<Derived> format)
	{
		static_assert(sizeof(Derived)==0, "operator>> is not supported for read-only matrices");
		return is;
	}

private:

	void print(std::ostream& os) const
	{
		if (mMatrix.size() == 0) {
			os << mFormat.matPrefix << mFormat.matSuffix;
			return;
		}

		const typename Derived::Nested m = mMatrix;
		typedef typename Derived::Scalar Scalar;
		typedef typename Derived::Index Index;

		Index width = 0;

		std::streamsize precision;
		if (mFormat.precision == Eigen::StreamPrecision)
			precision = 0;
		else if (mFormat.precision == Eigen::FullPrecision)
			precision = 10; // TODO: obtain full precision
		else
			precision = mFormat.precision;

		bool alignCols = !(mFormat.flags & Eigen::DontAlignCols);
		if (alignCols) {
			// compute the largest width
			for (Index j = 1; j < m.cols(); ++j)
				for (Index i = 0; i < m.rows(); ++i) {
					std::stringstream sstr;
					NumericalOstream nos(sstr);
					if(precision)
						sstr.precision(precision);
					nos << m.coeff(i, j);
					width = std::max<Index>(width, Index(sstr.str().length()));
				}
		}

		NumericalOstream nos(os);
		std::streamsize oldprecision = 0;
		if(precision)
			oldprecision = os.precision(precision);
		os << mFormat.matPrefix;
		for(Index i = 0; i < m.rows(); ++i) {
			if(i)
				os << mFormat.rowSpacer;
			os << mFormat.rowPrefix;
			if(width)
				os.width(width);
			nos << m.coeff(i, 0);
			for(Index j = 1; j < m.cols(); ++j) {
				os << mFormat.coeffSeparator;
				if(width)
					os.width(width);
				nos << m.coeff(i, j);
			}
			os << mFormat.rowSuffix;
			if(i < m.rows() - 1)
				os << mFormat.rowSeparator;
		}
		os << mFormat.matSuffix;
		if(precision)
			os.precision(oldprecision);

		return;
	}


protected:
	/// The matrix that gets formatted
	Eigen::MatrixBase<Derived>& mMatrix;
};


/**
 * Template class wrapping the serializing functionality of eigen
 * to support stream operators using a given format.
 * This variant supports reading and writing, hence the >>operator is available.
 */
template <typename Derived>
class TEigenFormat : public TEigenFormatReadOnly<Derived>
{
	typedef TEigenFormatReadOnly<Derived> Base;
public:
	TEigenFormat(Eigen::MatrixBase<Derived>& matrix, Eigen::IOFormat format) :
		Base(matrix, format) {}

	/**
	 * Input stream operator for reading a matrix from stream in a given format.
	 * @param is The input stream
	 * @param format The format class
	 * @return The stream
	 */
	friend std::istream& operator>> (std::istream & is,
	                                 TEigenFormat<Derived> format)
	{
		format.parse(is);
		return is;
	}

private:

	void parse(std::istream& is)
	{
		typedef typename Eigen::internal::traits< Derived >::Scalar Scalar;

		NumericalIstream nis(is);

		Scalar val;
		std::vector<Scalar> values;

		std::string matPrefix = boost::trim_copy(this->mFormat.matPrefix);
		std::string matSuffix = boost::trim_copy(this->mFormat.matSuffix);
		std::string rowPrefix = boost::trim_copy(this->mFormat.rowPrefix);
		std::string rowSuffix = boost::trim_copy(this->mFormat.rowSuffix);
		std::string rowSeparator = boost::trim_copy(this->mFormat.rowSeparator);
		std::string coeffSeparator = boost::trim_copy(this->mFormat.coeffSeparator);

		// if we have no rowSuffix, then keep the new line as row separator
		// if there is one (we need it to detect new rows)
		if(rowSuffix.empty() && this->mFormat.rowSeparator=="\n")
			rowSeparator = "\n";

		// if both rowSuffix and coeffSeparator are empty (and therefore
		// equal, we have a problem, so try to swap the rowSuffix and the
		// rowSeparator (this is valid for parsing and might solve the
		// problem)
		if(rowSuffix.empty() && coeffSeparator.empty()) {
			std::swap(rowSuffix, rowSeparator);
		}

		assert(rowSuffix != coeffSeparator);

		int rows = 0;
		int cols = 0;
		int col=0;

		// we must start with the prefix of the matrix
		match(is, matPrefix);

		// start parsing the matrix now
		while(true)
		{
			if(col==0) {
				// next could be the rowPrefix, or the suffix of the matrix
				// or we could reach eof. In the latter cases we have to break
				if(match(is, rowPrefix, matSuffix)==2 || is.eof())
					break;
			}

			// read in next value from our numerical stream
			nis >> val;
			if(!is.fail()) {
				++col;
				values.push_back(val);
			} else
				is.clear(); // if parsing of coeff failed, ignore that error and continue
				            // will detect syntax error afterwards if any

			// next one will be coeff separator or row suffix
			if(match(is, coeffSeparator, rowSuffix)==2 || is.eof()) {

				// found end of row

				if(rows==0) {
					cols = col; // save the number of columns if this was the first row ...
				} else {
					// ... otherwise, make sure that the number of columns is the same as in the first row
					if(col!=cols)
						MIRA_THROW(XIO, "Invalid number of columns, expected "
						          << cols << " but got " << col);
				}
				++rows;
				col = 0; // start new row

				// next could be the rowSeparator or the suffix of the matrix
				if(match(is, rowSeparator, matSuffix)==2 || is.eof())
				{
					// found suffix of matrix, so we are done
					break;
				}

				// otherwise we expect another row:
				//match(is, rowPrefix);

			} else {
				// we found a coeffSeparator so continue with parsing the
				// row in the next iteration
			}
		}
		// phew, we finished parsing the matrix successfully, now
		// resize the matrix to the found size (if it is a dynamic one) or
		// throw an exception that the sizes do not match.

		if(!EigenFormatSizeHelper<Derived>::checkOrResize(this->mMatrix,rows,cols))
			MIRA_THROW(XIO, "The size of the given matix ("
			           << this->mMatrix.rows() << "x" << this->mMatrix.cols() << ") "
			              "does not match the size of the read matrix ("
			           << rows << "x" << cols << ")");

		// now put the values into the matrix
		std::size_t idx = 0;
		for(int i=0; i<rows; ++i)
			for(int j=0; j<cols; ++j, ++idx)
				this->mMatrix(i,j) = values[idx];
	}

public:
	// returns the next token from the stream
	// - reads at most maxlength characters
	// - keeps the keepwhitespace character (other whitespaces are skipped)
	static std::string getToken(std::istream& is, std::size_t maxlength=255,
	                     char keepwhitespace=0)
	{
		assert(maxlength>0);

		std::string token;

		// skip white spaces except 'keepwhitespace'
		while(!is.eof())
		{
			char ch = is.peek();
			if(!isspace(ch) || ch==keepwhitespace)
				break;

			ch = is.get(); // read white space
		}

		if(is.eof())
			return token;

		// add the next char to the token
		token.push_back(is.get());

		// read token until maxlength, eof or a whitespace is reached
		for(std::size_t i=1; i<maxlength && !is.eof() && !isspace(is.peek()); ++i)
			token.push_back(is.get()); // read chars of token

		return token;
	}

	static void putback(std::istream& is, int num)
	{
		for(int i=0; i<num; ++i)
			is.unget();
	}

	// tries to match str1 OR str2 in the stream, throws an exception if no match
	// returns 1 if str1 matches and 2 if str2 matches
	static int match(std::istream& is, const std::string str1, const std::string str2)
	{
		std::size_t len = std::max(str1.size(), str2.size());

		if(len==0)
			return 1; // if both strings are empty, then return immediately

		// check if first of second match string is a whitespace

		char whitespace1=0;
		if(str1.size()==1 && isspace(str1[0]))
			whitespace1=str1[0];

		char whitespace2=0;
		if(str2.size()==1 && isspace(str2[0]))
			whitespace2=str2[0];

		// we can not handle the case where both strings are white spaces
		assert(whitespace1==0 || whitespace2==0);

		// checkout if we should match a whitespace
		char whitespace = 0;
		if(whitespace1!=0)
			whitespace=whitespace1;
		else if(whitespace2!=0)
			whitespace=whitespace2;

		std::string token = getToken(is, len, whitespace);

		bool matches1st = (token.substr(0,str1.size()) == str1);
		bool matches2nd = (token.substr(0,str2.size()) == str2);

		// if both match, return longest match
		if(matches1st && matches2nd) {
			if(str1.size()>=str2.size())
				return 1;
			else
				return 2;
		}

		if(matches1st && !matches2nd) {
			// if the matching string is shorter than the amount of characters
			// we have read, then we must put back those characters
			if(str1.size() < token.size())
				putback(is, token.size()-str1.size());
			return 1;
		}

		if(matches2nd && !matches1st) {
			// put back the characters if we have read too much
			if(str2.size() < token.size())
				putback(is, token.size()-str2.size());
			return 2;
		}

		// no match at all
		MIRA_THROW(XIO, "Expected either '" << str1 << "' or '" << str2
		           << "' in the input stream");
		return 0;
	}

	// tries to match str in the stream, throws an exception if no match
	static void match(std::istream& is, const std::string str)
	{
		if(str.empty())
			return;

		char whitespace=0;
		if(str.size()==1 && isspace(str[0]))
			whitespace=str[0];

		std::string token = getToken(is, str.size(), whitespace);

		if(token==str)
			return;

		// no match
		MIRA_THROW(XIO, "Expected '" << str <<  "' in the input stream");
	}

};

/**
 * Function for formatting an Eigen matrix using a special format.
 * This function can be used for writing and even reading a matrix
 * to or from a stream in the given format.
 *
 * Predefined formats are:
 * - EigenFormat::matlab()
 * - EigenFormat::clean()
 * - EigenFormat::eigen()
 * - EigenFormat::python()
 *
 * Usage:
 *
 * \code
 *   Matrix... m
 *
 *   // read a matrix from the stream in Matlab format:
 *   cin >> format(m,EigenFormat::matlab());
 *
 *   // write a matrix to the given stream in Matlab format using
 *   // a precision of 9 significant digits.
 *   cout << format(m,EigenFormat::matlab(9));
 * \endcode
 *
 * @param matrix The matrix that gets serialized.
 * @param format The format to use.
 * @ingroup MathModule
 */
template <typename Derived>
inline TEigenFormat<Derived> format(Eigen::MatrixBase<Derived>& matrix,
                                    Eigen::IOFormat format=EigenFormat::matlab())
{
	return TEigenFormat<Derived>(matrix, format);
}

/**
 * Same as above, provided for const correctness.
 */
template <typename Derived>
inline TEigenFormatReadOnly<Derived> format(const Eigen::MatrixBase<Derived>& matrix,
                                    Eigen::IOFormat format=EigenFormat::matlab())
{
	return TEigenFormatReadOnly<Derived>(const_cast<Eigen::MatrixBase<Derived>&>(matrix), format);
}

//////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
