/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Rect.h
 *    Provides multidimensional rectangle templates.
 *
 * @author Tim Langner, Erik Einhorn, Jens Kessler
 * @date   2010/10/24
 */

#ifndef _MIRA_RECT_H_
#define _MIRA_RECT_H_

#include <geometry/Line.h>
#include <geometry/Size.h>

namespace mira
{

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup GeometryModule
 *
 * The base class for rectangles. Rectangles are also template objects and
 * could be created with different data types (like floats, integer) and
 * different dimensions. In 2D it is the classical rectangle while in 1D it is
 * essentially a line and in 3D it is a box. Again Boost::geometry is used to
 * perform geometrical tasks like intersection, union or enclosure tests. The
 * base class implements a lot of constructors, operators and type converters.
 */
template<typename T, int D, typename Derived>
class RectBase
{
public:
	typedef Point<T,D> PointType;
	typedef Size<T,D> SizeType;
	typedef boost::geometry::model::box<PointType> BoxType;

public:
	/** @name Constructors */
	//@{

	/// The default constructor.
	RectBase() {}

	/// A constructor for the 3D case.
	RectBase(T x, T y, T z, T width, T height, T depth) :
		minCorner(x, y, z), maxCorner(x+width, y+height, z+depth) {}

	/// A constructor for the 2D case.
	RectBase(T x, T y, T width, T height) :
		minCorner(x, y), maxCorner(x+width, y+height) {}

	/// A constructor for the 2D case.
	explicit RectBase(const PointType& pos, T width, T height) :
		minCorner(pos), maxCorner(pos.x()+width, pos.y()+height) {}

	/// A constructor for the 3D case.
	explicit RectBase(const PointType& pos, T width, T height, T depth) :
		minCorner(pos), maxCorner(pos.x()+width, pos.y()+height, pos.z()+depth)	{}

	/** a constructor to construct the rect, cube or whatever by defining the
	  * lower corner and specifying an n-dimensional size.*/
	explicit RectBase(const PointType& pos, const SizeType& s) :
		minCorner(pos), maxCorner(pos+s) {}

	/// Creating an n-dimensional object from boost::model::box.
	explicit RectBase(const BoxType& box) :
		minCorner(box.min_corner()), maxCorner(box.max_corner()) {}

	/**
	 * Create an n-dimensional object from specifying two corners.
	 *
	 * Note: If the p1[i] > p2[i], the specific dimension will be switched.
	 *
	 * @param[in] p1: The lower corner.
	 * @param[in] p2: The upper corner.
	 */
	explicit RectBase(const PointType& p1, const PointType& p2)
	{
		for(int d=0; d<D; ++d)
		{
			if(p1[d]<p2[d]) {
				minCorner[d]=p1[d];
				maxCorner[d]=p2[d];
			} else {
				minCorner[d]=p2[d];
				maxCorner[d]=p1[d];
			}
		}
	}

	//@}

public:
	/// the method to serialize this object
	template<typename Reflector>
	void reflect(Reflector& reflector)
	{
		reflector.property("MinCorner", minCorner, "The minimum corner");
		reflector.property("MaxCorner", maxCorner, "The maximum corner");
	}

public:
	/** @name Implicit conversion operators */
	//@{

	/// converts to boost::geometry::box
	operator BoxType() const {
		return BoxType(minCorner, maxCorner);
	}

	//@}

public:
	/// return the size of the multidimensional rect
	SizeType size() const {
		return SizeType(maxCorner-minCorner);
	}

public:
	/** @name Comparison Operators */
	//@{

	template <typename OtherDerived>
	bool operator==(const RectBase<T,D, OtherDerived>& other) const {
		return minCorner == other.minCorner && maxCorner == other.maxCorner;
	}

	template <typename OtherDerived>
	bool operator!=(const RectBase<T,D, OtherDerived>& other) const {
		return !this->operator==(other);
	}

	//@}

public:
	/** @name Intersection tests */
	//@{

	/// this method checks if a given geometry is enclosed by this box
	template <typename OtherGeometry>
	bool contains(const OtherGeometry& otherGeometry) const {
		return boost::geometry::within(otherGeometry, *this);
	}

	/// this method checks for intersections with other geometry
	template <typename OtherGeometry>
	bool intersects(const OtherGeometry& otherGeometry) const {
		return boost::geometry::intersects(otherGeometry, *this);
	}

	/// this method checks for non-intersections with other geometry
	template <typename OtherGeometry>
	bool disjoint(const OtherGeometry& otherGeometry) const {
		return boost::geometry::disjoint(otherGeometry, *this);
	}

	//@}

public:
	/** @name Invalid and null Rects */
	//@{

	/**
	 * Returns true if this is an valid Rect where the maxCorner's components
	 * are larger than the minCorner, i.e. the width and height is not negative.
	 */
	bool isValid() const {
		for(int d=0; d<D; ++d)
			if(maxCorner[d]<minCorner[d])
				return false;
		return true;
	}

	/**
	 * Returns an invalid rect with negative extends.
	 */
	static Derived invalid() {
		Derived r;
		for(int d=0; d<D; ++d)	{
			r.minCorner[d] = 0;
			r.maxCorner[d] = -1;
		}
		return r;
	}

	/**
	 * Returns the normalized Rect, where the maxCorner and minCorner components
	 * are swapped if necessary to produce a valid Rect, where all maxCorner's
	 * components are larger than the minCorner.
	 */
	Derived normalized() const
	{
		Derived r;
		for(int d=0; d<D; ++d)
		{
			if(maxCorner[d]<minCorner[d]) {
				r.maxCorner[d] = minCorner[d];
				r.minCorner[d] = maxCorner[d];
			} else {
				r.maxCorner[d] = maxCorner[d];
				r.minCorner[d] = minCorner[d];
			}
		}
		return r;
	}

	/**
	 * Returns true, if all extends (i.e. width, height, etc) of this Rect
	 * are null.
	 */
	bool isNull() const {
		for(int d=0; d<D; ++d)
			if(maxCorner[d]!=minCorner[d])
				return false;
		return true;
	}

	/**
	 * Returns a null-Rect which extends are null (minCorner and maxCorner)
	 * are set to 0.
	 */
	static Derived null() {
		Derived r;
		r.minCorner = Eigen::Matrix<T,D,1>::Zero(D,1);
		r.maxCorner = Eigen::Matrix<T,D,1>::Zero(D,1);
		return r;
	}

	/**
	 * Same as null(). Provided for compatibility to Ogre.
	 */
	static Derived zero() {
		return null();
	}
	//@}

public:
	/** @name Geometric Operators */
	//@{

	/**
	 * Returns the bounding rectangle of this rectangle and the given rectangle.
	 * <pre>
	 * Moreover:
	 *   a | invalid = a
	 *   invalid | b = b
	 *   invalid | invalid = invalid
	 * </pre>
	 */
	Derived operator|(const Derived& other) const
	{
		if(!this->isValid())
			return other;
		if(!other.isValid())
			return *((Derived*)this); // we are of type Derived

		Derived r;
		for(int d=0; d<D; ++d) {
			r.minCorner[d] = std::min(minCorner[d], other.minCorner[d]);
			r.maxCorner[d] = std::max(maxCorner[d], other.maxCorner[d]);
		}
		return r;
	}

	/**
	 * Unites this rectangle with the given rectangle.
	 * <pre>
	 * Moreover:
	 *   a |= invalid, leaves a unchanged
	 *   invalid |= b, previous null becomes b
	 * </pre>
	 */
	const Derived& operator|=(const Derived& other)
	{
		if(!isValid()) {
			*this = other;
		} else if(other.isValid()) {
			for(int d=0; d<D; ++d) {
				if(other.minCorner[d]<minCorner[d])
					minCorner[d] = other.minCorner[d];
				if(other.maxCorner[d]>maxCorner[d])
					maxCorner[d] = other.maxCorner[d];
			}
		}
		return *((Derived*)this); // we are of type Derived
	}

	/**
	 * Unites this rectangle with the given point. Using this operator bounding
	 * boxes around single points can be created:
	 * \code
	 * Rect2i r = Rect2i::invalid();
	 * r |= point1;
	 * r |= point2;
	 * r |= point3;
	 * ...
	 * \endcode
	 * <pre>
	 * Moreover:
	 *   invalid |= point, creates an empty rect as bounding box around the single point
	 * </pre>
	 */
	const Derived& operator|=(const PointType& p)
	{
		if(!isValid()) {
			minCorner = p;
			maxCorner = p;
		} else {
			for(int d=0; d<D; ++d) {
				if(p[d]<minCorner[d])
					minCorner[d] = p[d];
				if(p[d]>maxCorner[d])
					maxCorner[d] = p[d];
			}
		}
		return *((Derived*)this); // we are of type Derived
	}

	/**
	 * Returns the intersection of this rectangle and the given rectangle.
	 * Returns an invalid rectangle if there is no intersection.
	 * <pre>
	 * Moreover:
	 *   a & invalid = invalid
	 *   invalid & b = invalid
	 *   invalid & invalid = invalid
	 *   a & b = invalid, only if a and b do not intersect at all.
	 * </pre>
	 */
	Derived operator&(const Derived& other) const
	{
		if(!isValid())
			return invalid();
		if(!other.isValid())
			return invalid();

		Derived r;
		for(int d=0; d<D; ++d) {
			r.minCorner[d] = std::max(minCorner[d], other.minCorner[d]);
			r.maxCorner[d] = std::min(maxCorner[d], other.maxCorner[d]);
		}
		return r;
	}

	/**
	 * Intersects the rectangle with the given rectangle.
	 * Returns an invalid rectangle if there is no intersection.
	 */
	const Derived& operator&=(const Derived& other)
	{
		if(!isValid())
			return *((Derived*)this);

		if(!other.isValid()) {
			*this = other;
		} else {
			for(int d=0; d<D; ++d) {
				minCorner[d] = std::max(minCorner[d], other.minCorner[d]);
				maxCorner[d] = std::min(maxCorner[d], other.maxCorner[d]);
			}
		}
		return *((Derived*)this);
	}
	//@}

	/// Converts from a Rect of a different type.
	template<typename U, typename OtherDerived>
	static Derived convertFrom(const RectBase<U,D, OtherDerived>& other) {
		return Derived(PointType(other.minCorner.template cast<T>()),
		               PointType(other.maxCorner.template cast<T>()));
	}

	/**
	 * Moves the rect (the minCorner and the maxCorner) by the specified displacement.
	 */
	const Derived& operator+=(const PointType& displacement)
	{
		minCorner+=displacement;
		maxCorner+=displacement;
		return *((Derived*)this);
	}

	/**
	 * Returns a rect that is moved by the specified displacement.
	 */
	Derived operator+(const PointType& displacement) const
	{
		return Derived(minCorner+displacement, maxCorner+displacement);
	}

public:
	PointType minCorner;
	PointType maxCorner;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup GeometryModule
 *
 * Rect class for defining rectangles
 */
template<typename T, int D>
class Rect : public RectBase<T,D, Rect<T,D> >
{
public:
	typedef RectBase<T,D, Rect<T,D> > Base;
	typedef typename Base::PointType PointType;
	typedef typename Base::SizeType SizeType;
	typedef typename Base::BoxType BoxType;

public:
	/// The default constructor.
	Rect() {}

	/// A constructor to initialize from 2 points.
	Rect(const PointType& p1, const PointType& p2) :
		Base(p1, p2) {}
};

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup GeometryModule
 *
 * Specialization for 2D
 */
template<typename T>
class Rect<T,2> : public RectBase<T,2, Rect<T,2> >
{
public:
	typedef RectBase<T,2, Rect<T,2> > Base;
	typedef typename Base::PointType PointType;
	typedef typename Base::SizeType SizeType;
	typedef typename Base::BoxType BoxType;

public:
	/** @name Constructors */
	//@{

	/// The default constructor.
	Rect() {}

	/**
	 * @brief A constructor to create a 2D rect.
	 * @param[in] x:      the min corner x-coordinate
	 * @param[in] y:      the min corner y-coordinate
	 * @param[in] width:  the rect width
	 * @param[in] height: the rect height
	 */
	Rect(T x, T y, T width, T height) :
		Base(x, y, width, height) {}

	/**
	 * @brief A constructor to create a 2D rect.
	 * @param[in] pos:    the min corner of the rect
	 * @param[in] width:  the rect width
	 * @param[in] height: the rect height
	 */
	Rect(const PointType& pos, T width, T height) :
		Base(pos, width, height) {}

	/**
	 * @brief A constructor to create a 2D rect.
	 * @param[in] pos: the min corner of the rect
	 * @param[in] s:   the size of the rect
	 */
	Rect(const PointType& pos, const SizeType& s) :
		Base(pos, s) {}

	/// copy constructor from boost::BoxType
	explicit Rect(const BoxType& box) :
		Base(box) {}

	/// copy constructor from OpenCV rect
	explicit Rect(const cv::Rect_<T>& rect) :
		Base(rect.x, rect.y, rect.width, rect.height) {}

	/**
	 * @brief A constructor to create a 2D rect.
	 * @param[in] p1: the lower corner of the rect
	 * @param[in] p2: the upper corner of the rect
	 * */
	Rect(const PointType& p1, const PointType& p2) :
		Base(p1, p2) {}

	//@}

public:
	/** @name Implicit conversion operators */
	//@{

	/// converts to open cv rect
	operator cv::Rect_<T>() const
	{
		return cv::Rect_<T>(Base::minCorner, Base::size());
	}

	//@}

public:
	/** @name Access to corner elements and size*/
	//@{

	/// Returns the x-coordinate of the lower corner.
	T& x0() { return this->minCorner[0]; }
	/// Returns the x-coordinate of the lower corner.
	T  x0() const { return this->minCorner[0]; }

	/// Returns the y-coordinate of the lower corner.
	T& y0() { return this->minCorner[1]; }
	/// Returns the y-coordinate of the lower corner.
	T  y0() const { return this->minCorner[1]; }

	/// Returns the x-coordinate of the upper corner.
	T& x1() { return this->maxCorner[0]; }
	/// Returns the x-coordinate of the upper corner.
	T  x1() const { return this->maxCorner[0]; }

	/// Returns the y-coordinate of the upper corner.
	T& y1() { return this->maxCorner[1]; }
	/// Returns the y-coordinate of the upper corner.
	T  y1() const { return this->maxCorner[1]; }

	/// Returns the width.
	T width()  const { return this->maxCorner[0] - this->minCorner[0]; }
	/// Returns the height.
	T height() const { return this->maxCorner[1] - this->minCorner[1]; }

	//@}
};

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup GeometryModule
 *
 * Specialization for 3D
 */
template<typename T>
class Rect<T,3> : public RectBase<T,3, Rect<T,3> >
{
public:
	typedef RectBase<T,3, Rect<T,3> > Base;
	typedef typename Base::PointType PointType;
	typedef typename Base::SizeType SizeType;
	typedef typename Base::BoxType BoxType;

public:
	/** @name Constructors */
	//@{

	Rect() {} ///< the default constructor

	/** a constructor to create a 3D rect
	 * @param x: the min corner x-coordinate
	 * @param y: the min corner y-coordinate
	 * @param z: the min corner z-coordinate
	 * @param width: the rect width
	 * @param height: the rect height
	 * @param depth: the rect depth
	 */
	Rect(T x, T y, T z, T width, T height, T depth) :
		Base(x, y, z, width, height, depth)	{}

	/** a constructor to create a 3D rect
	 * @param pos: the min corner
	 * @param width: the rect width
	 * @param height: the rect height
	 * @param depth: the rect depth
	 */
	Rect(const PointType& pos, T width, T height, T depth) :
		Base(pos, width, height, depth) {}

	/** a constructor to create a 3D rect
	 * @param pos: the min corner
	 * @param s: the rect size
	 */
	Rect(const PointType& pos, const SizeType& s) :
		Base(pos, s) {}

	/** copy constructor from boost::BoxType*/
	explicit Rect(const BoxType& box) :
		Base(box) {}

	/** a constructor to create a 3D rect
	 * @param[in] p1: the lower corner of the rect
	 * @param[in] p2: the upper corner of the rect*/
	Rect(const PointType& p1, const PointType& p2) :
		Base(p1, p2) {}

	//@}

public:
	/** @name Access to corner elements and size*/
	//@{

	/// Returns the x-coordinate of the lower corner.
	T& x0() { return this->minCorner[0]; }
	/// Returns the x-coordinate of the lower corner.
	T  x0() const { return this->minCorner[0]; }

	/// Returns the y-coordinate of the lower corner.
	T& y0() { return this->minCorner[1]; }
	/// Returns the y-coordinate of the lower corner.
	T  y0() const { return this->minCorner[1]; }

	/// Returns the z-coordinate of the lower corner.
	T& z0() { return this->minCorner[2]; }
	/// Returns the z-coordinate of the lower corner.
	T  z0() const { return this->minCorner[2]; }

	/// Returns the x-coordinate of the upper corner.
	T& x1() { return this->maxCorner[0]; }
	/// Returns the x-coordinate of the upper corner.
	T  x1() const { return this->maxCorner[0]; }

	/// Returns the y-coordinate of the upper corner.
	T& y1() { return this->maxCorner[1]; }
	/// Returns the y-coordinate of the upper corner.
	T  y1() const { return this->maxCorner[1]; }

	/// Returns the z-coordinate of the upper corner.
	T& z1() { return this->maxCorner[2]; }
	/// Returns the z-coordinate of the upper corner.
	T  z1() const { return this->maxCorner[2]; }

	/// Returns the width.
	T width()  const { return this->maxCorner[0] - this->minCorner[0]; }
	/// Returns the height.
	T height() const { return this->maxCorner[1] - this->minCorner[1]; }
	/// Returns the depth.
	T depth()  const { return this->maxCorner[2] - this->minCorner[2]; }
	//@}
};

///////////////////////////////////////////////////////////////////////////////

/// A 2D rect with integer precision.
typedef Rect<int, 2> Rect2i;

/// A 2D rect with floating point precision.
typedef Rect<float, 2> Rect2f;

/// A 2D rect with 64 bit floating point precision.
typedef Rect<double, 2> Rect2d;

/// A 3D box with integer precision.
typedef Rect<int, 3> Box3i;

/// A 3D box with floating point precision.
typedef Rect<float, 3> Box3f;

/// A 3D box with 64 bit floating point precision.
typedef Rect<double, 3> Box3d;

///////////////////////////////////////////////////////////////////////////////
}

///@cond INTERNAL
// BOOST specialization on Rect to accept them as geometry entity
namespace boost { namespace geometry { namespace traits {

///////////////////////////////////////////////////////////////////////////////

template <typename T, int D, typename Derived>
struct tag<mira::RectBase<T,D, Derived> > { typedef box_tag type; };

template <typename T, int D, typename Derived>
struct point_type<mira::RectBase<T,D, Derived> > { typedef mira::Point<T,D> type; };

template <typename T, int D, std::size_t Dimension, typename Derived>
struct indexed_access<mira::RectBase<T,D, Derived>, min_corner, Dimension>
{
	typedef typename geometry::coordinate_type<mira::Point<T,D> >::type coordinate_type;
	static inline coordinate_type get(mira::RectBase<T,D,Derived> const& b) {
		return geometry::get<Dimension>(b.minCorner);
	}
	static inline void set(mira::RectBase<T,D,Derived>& b, coordinate_type const& value) {
		geometry::set<Dimension>(b.minCorner, value);
	}
};

template <typename T, int D, std::size_t Dimension, typename Derived>
struct indexed_access<mira::RectBase<T,D, Derived>, max_corner, Dimension>
{
	typedef typename geometry::coordinate_type<mira::Point<T,D>>::type coordinate_type;
	static inline coordinate_type get(mira::RectBase<T,D, Derived> const& b) {
		return geometry::get<Dimension>(b.maxCorner);
	}
	static inline void set(mira::RectBase<T,D, Derived>& b, coordinate_type const& value) {
		geometry::set<Dimension>(b.maxCorner, value);
	}
};

template <typename T, int D>
struct tag<mira::Rect<T,D> > : public tag<mira::RectBase<T,D, mira::Rect<T,D>>> {};

template <typename T, int D>
struct point_type<mira::Rect<T,D> > : public point_type<mira::RectBase<T,D,mira::Rect<T,D>>> {};

template <typename T, int D, std::size_t Dimension>
struct indexed_access<mira::Rect<T,D>, min_corner, Dimension> :
	public indexed_access<mira::RectBase<T,D,mira::Rect<T,D>>, min_corner, Dimension> {};

template <typename T, int D, std::size_t Dimension>
struct indexed_access<mira::Rect<T,D>, max_corner, Dimension> :
	public indexed_access<mira::RectBase<T,D,mira::Rect<T,D>>, max_corner, Dimension> {};

///////////////////////////////////////////////////////////////////////////////

}}}
///@endcond

///////////////////////////////////////////////////////////////////////////////

#endif
