/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SyncTimedRead.h
 *    Methods for synchronously reading data using boost asio with timeout.
 *
 * @author Tim Langner
 * @date   2011/06/29
 */

#ifndef _MIRA_SYNCTIMEDREAD_H_
#define _MIRA_SYNCTIMEDREAD_H_

#ifndef Q_MOC_RUN
#include <boost/asio/deadline_timer.hpp>
#include <boost/optional.hpp>
#include <boost/bind.hpp>
#endif

#include <utils/Time.h>
#include <error/Exceptions.h>

namespace boost { namespace asio {

///////////////////////////////////////////////////////////////////////////////

///@cond INTERNAL

namespace Private {

inline void setTimerResult(optional<system::error_code>* codeStorage,
                           system::error_code code)
{
	// If we got a boost::asio::error::operation_aborted, ignore it.
	if (code == error::operation_aborted)
		return;

	codeStorage->reset(code);
}

inline void setResult(optional<system::error_code>* codeStorage,
                      std::size_t* bytesReceivedStorage,
                      system::error_code code, std::size_t bytesReceived)
{
	// If we got a boost::asio::error::operation_aborted, ignore it.
	if (code == error::operation_aborted)
		return;

	codeStorage->reset(code);
	*bytesReceivedStorage = bytesReceived;
}

}
///@endcond

///////////////////////////////////////////////////////////////////////////////

/**
 * Read some data from the specified stream 's' into the given buffer 'buffers'.
 * The method returns the number of bytes that were read. Additionally, a
 * timeout can be specified. The method will abort if the timeout has passed
 * before all data was read.
 * If an error occurs, an exception is thrown. If you do not want an exception
 * to be thrown, use the method below.
 * @ingroup CommunicationModule
 */
template<typename SyncReadStream, typename MutableBufferSequence>
std::size_t read_some(SyncReadStream& s, const MutableBufferSequence& buffers,
                      mira::Duration timeout)
{
	system::error_code ec;
	std::size_t bytesRead = read_some(s, buffers, timeout, ec);
	detail::throw_error(ec);
	return bytesRead;
}

/**
 * Read some data from the specified stream 's' into the given buffer 'buffers'.
 * Same as above, but additionally, the caller can pass an system::error_code
 * variable to retrieve the error code if an error occurs.
 * @ingroup CommunicationModule
 */
template<typename SyncReadStream, typename MutableBufferSequence>
std::size_t read_some(SyncReadStream& s, const MutableBufferSequence& buffers,
                      mira::Duration timeout, system::error_code& ec)
{
	ec = boost::system::error_code();
	optional<system::error_code> timerResult;
	optional<system::error_code> readResult;
	std::size_t bytesRead = 0;

	deadline_timer timer(s.get_io_service());

	timer.expires_from_now(timeout);
	timer.async_wait(bind(&Private::setTimerResult, &timerResult, _1));

	s.async_read_some(buffers, bind(&Private::setResult,
	                                &readResult, &bytesRead, _1, _2));

	s.get_io_service().reset();

	// For more details on the following code, please look here:
	//  http://stackoverflow.com/questions/10858719/cancel-async-read-due-to-timeout

	// It is important to use a loop here, to ensure, that all handlers are
	// remove from the io_service, before leaving this method. Otherwise a
	// subsequent call of read_some will return immediately caused by still
	// outstanding handlers.
	// The io_service::reset() method only allows the io_service to resume
	// running from a stopped state, it does not remove any handlers already
	// queued into the io_service.
	while (s.get_io_service().run_one())
	{
		if (readResult)
		{
			// Data received, so cancel the timer.
			// This will result in a call of the handler Private::setTimerResult
			// with boost::asio::error::operation_aborted as the error.
			timer.cancel();
			ec = readResult.get();
		} else
		if (timerResult)
		{
			// Timeout occurred, so cancel the read operation. This will result
			// in a call of the handler Private::setResult with the error code
			// boost::asio::error::operation_aborted.
			s.cancel();
			ec = system::error_code(system::errc::timed_out,
			                        system::get_generic_category());
			//return bytesRead;
		}
	}

	// Reset service, guaranteeing it is in a good state for subsequential runs.
	s.get_io_service().reset();

	return bytesRead;
}

///////////////////////////////////////////////////////////////////////////////

}}

#endif 
