/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TapeChannelView.C
 *    Implementation of TapeChannelView.h.
 *
 * @author Tim Langner
 * @date   2012/01/01
 */

#include <TapeChannelView.h>

#include <QAbstractListModel>
#include <QHeaderView>
#include <QInputDialog>
#include <QLabel>
#include <QMenu>
#include <QMouseEvent>
#include <QPushButton>

#include <widgets/SelectionListDialog.h>
#include <widgets/MultiInputDialog.h>

#include <TapeDataRenderer.h>
#include <TapeEditor.h>
#include <TapeCommand.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class TapeChannelListHeaderModel : public QAbstractListModel
{
public:

	enum ColumnIndex
	{
		Name = 0,
		Type = 1,
		Items = 2
	};

	TapeChannelListHeaderModel(QObject* parent = NULL) :
		QAbstractListModel(parent)
	{
		mHeaderText
			<< tr("Name")
			<< tr("Type")
			<< tr("Items");
	}

	QVariant headerData(int section, Qt::Orientation orient, int role) const
	{
		if (orient == Qt::Horizontal)
		{
			switch (role) {
			case Qt::DisplayRole:
				return mHeaderText.at(section);
			case Qt::TextAlignmentRole:
				return int(Qt::AlignLeft | Qt::AlignVCenter);
			}
		}
		return QVariant();
	}

	int rowCount(const QModelIndex&) const { return 0; }

	int columnCount(const QModelIndex&) const { return mHeaderText.count(); }

	QVariant data(const QModelIndex&, int) const { return QVariant(); }

private:

	QStringList mHeaderText;
};

//////////////////////////////////////////////////////////////////////////////

TapeChannelPropertyDialog::TapeChannelPropertyDialog(TapeChannelInfo* channel,
                                                     TapeEditor* editor, QWidget* parent) :
	QDialog(parent),
	mChannel(channel),
	mEditor(editor)
{
	setWindowTitle(QString("Properties of ") + mChannel->name.c_str());
	QVBoxLayout* layout = new QVBoxLayout(this);
	layout->setSpacing(0);
	layout->setContentsMargins(0, 0, 0, 0);

	QHBoxLayout* hLayout = new QHBoxLayout();

	mRenderer = new QComboBox(this);
	mRenderer->addItem("DefaultRenderer", TapeDataRenderer::CLASS().getIdentifier().c_str());

	int index = 1;
	// query all codec classes from the class factory:
	typedef std::map<std::string, ClassProxy > ClassMap;
	ClassMap rendererClasses = TapeDataRenderer::CLASS().getDerivedClasses();
	foreach(ClassMap::value_type i, rendererClasses)
	{
		if(i.second.isAbstract())
			continue;

		TapeDataRendererPtr r(i.second.newInstance<TapeDataRenderer>());
		if (r->getDatatype() != mChannel->type)
			continue;

		std::string name = i.second.getMetaInfo("Name");
		if(name.empty())
			name = i.first; // use class name, if no meta name was specified

		mRenderer->addItem(name.c_str(), i.first.c_str());
		if (mChannel->renderer->getClass().getIdentifier() == i.first)
			mRenderer->setCurrentIndex(index);
		++index;
	}
	connect(mRenderer, SIGNAL(currentIndexChanged(int)), this, SLOT(changeRenderer(int)));
	hLayout->addWidget(new QLabel("Renderer", this));
	hLayout->addWidget(mRenderer);
	layout->addLayout(hLayout);
	mPropertyEditor = new PropertyEditor(this);
	mPropertyEditor->setHideSingleRootNode(true);

	PropertySerializer s;
	PropertyNode* p = s.reflectProperties("Channel", mChannel);
	mPropertyEditor->addProperty(p);
	mPropertyNode.reset(p);

	layout->addWidget(mPropertyEditor);
	mButtonBox = new QDialogButtonBox(QDialogButtonBox::Ok | QDialogButtonBox::Cancel);

	connect(mButtonBox,  SIGNAL(accepted()), this, SLOT(accept()));
	connect(mButtonBox,  SIGNAL(rejected()), this, SLOT(reject()));
	layout->addWidget(mButtonBox);
}

void TapeChannelPropertyDialog::changeRenderer(int index)
{
	std::string identifier = mRenderer->itemData(index).toString().toStdString();
	TapeDataRendererPtr r(TapeDataRenderer::CLASS().newInstance(identifier));

	mChannel->renderer = r;
	mChannel->renderer->setEditor(mEditor);

	mPropertyEditor->clear();
	PropertySerializer s;
	PropertyNode* p = s.reflectProperties("Channel", mChannel);
	mPropertyEditor->addProperty(p);
	mPropertyNode.reset(p);
	mEditor->updateContents(false);
}

//////////////////////////////////////////////////////////////////////////////

TapeChannelView::TapeChannelView(QWidget* parent, TapeEditor* editor) :
	ScrollView(parent)
{
	mEditor = editor;
	mHeader = new QHeaderView(Qt::Horizontal, viewport());
	mHeader->setModel(new TapeChannelListHeaderModel(this));
	mHeader->setHighlightSections(false);
	mHeader->setStretchLastSection(true);
	mHeader->setSortIndicatorShown(false);
	mHeader->resizeSection(TapeChannelListHeaderModel::Name, 120);
	mHeader->resizeSection(TapeChannelListHeaderModel::Type, 150);
	mHeader->resizeSection(TapeChannelListHeaderModel::Items, 20);
	
	connect(mHeader, SIGNAL(sectionResized(int, int, int)), SLOT(updateHeader()));

	setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
	viewport()->setFocusPolicy(Qt::ClickFocus);
	setMouseTracking(true);
	const QFont& font = this->font();
	setFont(QFont(font.family(), font.pointSize() - 1));

	mInUpdate = false;
	mResizeDragRow = -1;
}

void TapeChannelView::resizeEvent(QResizeEvent* resizeEvent)
{
	ScrollView::resizeEvent(resizeEvent);
	updateHeader();
}

void TapeChannelView::contextMenuEvent(QContextMenuEvent* contextMenuEvent)
{
	QPoint pos = viewportToContents(contextMenuEvent->pos());
	TapeChannelInfo* info = getChannelAtPos(pos);

	QMenu menu;
	QAction* deleteAction = NULL;
	if (info != NULL)
		deleteAction = menu.addAction(tr("Delete channel"));
	QAction* renameAction = NULL;
	if (info != NULL)
		renameAction = menu.addAction(tr("Rename channel"));
	QAction* addAction = menu.addAction(tr("Create channel"));

	QAction* changeRendererAction;
	if (info != NULL) {
		menu.addSeparator();
		changeRendererAction = menu.addAction(tr("Change Renderer"));
	}

	QAction* action = menu.exec(contextMenuEvent->globalPos());
	if(action==NULL)
		return;

	if (action == deleteAction)
	{
		mEditor->executeCommand(new DeleteChannelCommand(info->name, mEditor));
	}
	else if (action == renameAction)
	{
		bool ok;
		QString text = QInputDialog::getText(this,
											 tr("Renaming channel"),
											 tr("Enter channel name:"),
											 QLineEdit::Normal,
											 info->name.c_str(), &ok);
		if (ok && !text.isEmpty())
		{
			std::string oldName = info->name;
			std::string newName = text.toStdString();
			if (oldName == newName)
				return;
			if (mEditor->getChannels().count(newName) > 0)
				MIRA_THROW(XInvalidConfig, "Channel with name '" << newName << "' already exists");
			mEditor->executeCommand(new RenameChannelCommand(oldName, newName, mEditor));
		}
	}
	else if (action == addAction)
	{
		MultiInputDialog dialog("Creating channel", this);
		dialog.addInputField("name", "Channel name", "");
		dialog.addInputField("type", "Channel type", "");
		if (dialog.exec() == 0)
			return;
		std::string name = dialog.getInput("name").toStdString();
		std::string type = dialog.getInput("type").toStdString();
		mEditor->executeCommand(new CreateChannelCommand(name, type, mEditor));
	}
	else if(action == changeRendererAction) {
		changeRendererDialog(info);
	}
}

void TapeChannelView::mousePressEvent(QMouseEvent* mouseEvent)
{
	QPoint pos = viewportToContents(mouseEvent->pos());

	// check for resizing rows
	if (mouseEvent->button() == Qt::LeftButton)
	{
		mResizeDragRow = -1;
		for(size_t i=0; i<mRowSeparatorPositions.size(); ++i)
		{
			int top    = mRowSeparatorPositions[i]-2;
			int bottom = mRowSeparatorPositions[i]+2;
			if(pos.y()>=top && pos.y()<=bottom) {
				mResizeDragRow = (int)i;
				mResizeDragStartPos = pos.y();
				mResizeDragStartHeight = getChannel(i)->renderer->getTotalHeight();
				setCursor(QCursor(Qt::SizeVerCursor));
				return;
			}
		}
	}

	// check selection of rows
	if (mouseEvent->button() == Qt::LeftButton)
	{
		bool modifier = (mouseEvent->modifiers() & (Qt::ShiftModifier | Qt::ControlModifier));
		TapeChannelInfo* info = getChannelAtPos(pos);
		bool wasSelected = false;
		if (info != NULL)
			wasSelected = info->selected;
		if (!modifier)
			mEditor->clearSelectedChannels();
		if (info != NULL)
			info->selected = wasSelected?false:true;
		mEditor->getChannelView()->updateContents();
	}
	ScrollView::mousePressEvent(mouseEvent);
}

void TapeChannelView::mouseMoveEvent(QMouseEvent* mouseEvent)
{
	QPoint pos = viewportToContents(mouseEvent->pos());
	if(mResizeDragRow<0) {
		for(size_t i=0; i<mRowSeparatorPositions.size(); ++i)
		{
			int top    = mRowSeparatorPositions[i]-2;
			int bottom = mRowSeparatorPositions[i]+2;
			if(pos.y()>=top && pos.y()<=bottom) {
				setCursor(QCursor(Qt::SizeVerCursor));
				return;
			}
		}
		unsetCursor();
	} else {
		int delta = pos.y()-mResizeDragStartPos;

		int newHeight = std::max(mResizeDragStartHeight+delta,25);
		getChannel(mResizeDragRow)->renderer->setTotalHeight(newHeight);
		updateContentsHeight();
		updateContents();
		// Somehow, the following are updated by the above calles
		//mEditor->getDataView()->updateContentsHeight();
		//mEditor->updateContents(false);
	}


}

void TapeChannelView::mouseReleaseEvent(QMouseEvent* mouseEvent)
{
	mResizeDragRow = -1;
	unsetCursor();
}


void TapeChannelView::mouseDoubleClickEvent(QMouseEvent *mouseEvent)
{
	QPoint pos = viewportToContents(mouseEvent->pos());
	TapeChannelInfo* info = getChannelAtPos(pos);
	changeRendererDialog(info);
}

void TapeChannelView::changeRendererDialog(TapeChannelInfo* info)
{
	if (info != NULL)
	{
		TapeChannelPropertyDialog dialog(info, mEditor, this);
		if (dialog.exec() == 0)
			return;
		mEditor->getDataView()->updateContentsHeight();
		mEditor->updateContents(false);
	}
}

void TapeChannelView::drawCell(QPainter* painter, const TapeChannelInfoMap::value_type& data, int col, const QRect& rect)
{
	const QPalette& pal = palette();
	QColor bg, fg;
	if (col == TapeChannelListHeaderModel::Name)
	{
		bg = data.second.color.lighter();
		fg = pal.windowText().color();
	}
	else if (data.second.selected)
	{
		bg = pal.midlight().color().darker(150);
		fg = pal.midlight().color().lighter();
	}
	else
	{
		bg = pal.window().color();
		fg = pal.windowText().color();
	}

	QRect rectText(rect.topLeft() + QPoint(4, 4), rect.size() - QSize(8, 8));
	QLinearGradient grad(0, rect.top(), 0, rect.bottom());
	grad.setColorAt(0.4, bg);
	grad.setColorAt(1.0, bg.darker(120));
	painter->fillRect(rect, grad);
	painter->setPen(fg);
	if (col == TapeChannelListHeaderModel::Name)
	{
		painter->drawText(rectText, Qt::AlignLeft | Qt::AlignTop,
		                  data.first.c_str());
	}
	if (col == TapeChannelListHeaderModel::Type)
	{
		painter->drawText(rectText, Qt::AlignLeft | Qt::AlignTop,
		                  data.second.type.c_str());
	}
	if (col == TapeChannelListHeaderModel::Items)
	{
		painter->drawText(rectText, Qt::AlignLeft | Qt::AlignTop,
		                  toString(data.second.data.size()).c_str());
	}
	painter->setPen(bg.lighter(150));
	painter->drawLine(rect.left(), rect.top(), rect.left(), rect.bottom());
	painter->drawLine(rect.left(), rect.top(), rect.right(), rect.top());
	painter->setPen(bg.darker(150));
	painter->drawLine(rect.right(), rect.top(), rect.right(), rect.bottom());
	painter->drawLine(rect.left(), rect.bottom(), rect.right(), rect.bottom());
}

void TapeChannelView::updatePixmap(int cx, int cy)
{
	QWidget* viewport = this->viewport();
	int w = viewport->width();
	int h = viewport->height();

	if (w < 1 || h < 1)
		return;

	const QPalette& pal = palette();

	mPixmap = QPixmap(w, h);
	mPixmap.fill(pal.window().color());

	QPainter painter(&mPixmap);
	painter.initFrom(viewport);

	mHeader->setOffset(cx);

	int colCount = mHeader->count();
	int hh = mHeader->sizeHint().height();
	int ch = contentsHeight() - (25 << 1);
	int x, y1, y2, h1;
	y1 = y2 = 0;

	const TapeChannelInfoMap& channels = mEditor->getChannels();
	foreach(const auto& channel, channels)
	{
		h1  = channel.second.renderer->getTotalHeight();
		y1  = y2;
		y2 += h1;
		if (y2 > cy && y1 < cy + h)
		{
			QRect rect(0, y1 - cy + hh, w, h1);
			x = 0;
			for (int col = 0; col < colCount; ++col) {
				int dx = mHeader->sectionSize(col);
				if (x + dx > cx && x < cx + w) {
					rect.setX(x - cx);
					rect.setWidth(dx);
					drawCell(&painter, channel, col, rect);
				}
				x += dx;
			}
		}
	}

	if (cy + h > ch)
	{
		painter.setPen(pal.mid().color());
		painter.drawLine(0, ch - cy, w, ch - cy);
		painter.fillRect(0, ch - cy + 1, w, cy + h - ch, pal.dark().color());
	}
}

void TapeChannelView::updateContentsHeight()
{
	int contentsHeight = mHeader->sizeHint().height();
	mRowSeparatorPositions.clear();

	const TapeChannelInfoMap& channels = mEditor->getChannels();
	foreach(const TapeChannelInfoMap::value_type& channel, channels)
	{
		contentsHeight += channel.second.renderer->getTotalHeight();
		mRowSeparatorPositions.push_back(contentsHeight);
	}

	contentsHeight+=(25<<1);

	resizeContents(contentsWidth(), contentsHeight);

	mEditor->getDataView()->updateContentsHeight();
	mEditor->getDataView()->updateContents();

	updateContents();
}

void TapeChannelView::updateContents(const QRect& rect)
{
	if (mInUpdate)
		return;
	mInUpdate = true;
	updatePixmap(contentsX(), contentsY());
	ScrollView::updateContents(rect);
	mInUpdate = false;
}

void TapeChannelView::updateContents()
{
	if (mInUpdate)
		return;
	mInUpdate = true;
	updatePixmap(contentsX(), contentsY());
	ScrollView::updateContents();
	mInUpdate = false;
}

TapeChannelInfo* TapeChannelView::getChannelAtPos(const QPoint& pos)
{
	int cy = contentsY();
	int  y = pos.y() - mHeader->sizeHint().height();
	int  h = viewport()->height();

	int y1, y2;
	y1 = y2 = 0;
	TapeChannelInfoMap& channels = mEditor->getChannels();
	TapeChannelInfoMap::iterator channel = channels.begin();
	foreach(TapeChannelInfoMap::value_type& channel, channels)
	{
		if (y2 >= cy + h)
			break;
		y1  = y2;
		y2 += channel.second.renderer->getTotalHeight();
		if (y >= y1 && y < y2)
			return &channel.second;
	}
	return NULL;
}

TapeChannelInfo* TapeChannelView::getChannel(int idx)
{
	TapeChannelInfoMap::iterator it = mEditor->getChannels().begin();
	std::advance(it, idx);
	if(it==mEditor->getChannels().end())
		return NULL;

	return &it->second;
}

void TapeChannelView::onContentsMoving( int cx, int cy)
{
	if (contentsY() != cy)
		setContentsPos(contentsX(), cy);
}

void TapeChannelView::drawContents(QPainter* painter, const QRect& rect)
{
	painter->drawPixmap(rect, mPixmap, rect);
}

void TapeChannelView::updateHeader()
{
	int colCount = mHeader->count() - 1;
	int contentsWidth = (viewport()->width() >> 1);
	for (int col = 0; col < colCount; ++col)
		contentsWidth += mHeader->sectionSize(col);
	mHeader->setFixedWidth(contentsWidth);

	resizeContents(contentsWidth, contentsHeight());
	updateContents();
}

///////////////////////////////////////////////////////////////////////////////

}
