/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file MIRATape.C
 *    A command line tool for recording, playing back, copying, merging and editing tape files.
 *
 * @author Tim Langner
 * @date   2010/12/28
 */

#include <boost/algorithm/string.hpp>

#include <opencv2/highgui/highgui.hpp>

#include <image/Img.h>
#include <utils/Path.h>
#include <serialization/BinarySerializer.h>

#include <utils/PathFinder.h>
#include <utils/PrintProgressBar.h>
#include <factory/ManifestAgent.h>

#include <fw/Tape.h>
#include <fw/Framework.h>
#include <fw/FrameworkDefines.h>
#include <fw/TapeRecorder.h>
#include <fw/TapePlayer.h>

#include <video/AbstractVideoOutput.h>

using namespace mira;
using namespace mira::video;

void listVideoCodecs(const std::string & indent)
{
	typedef std::map<std::string, mira::ClassProxy > ClassMap;
	ClassMap codecClasses = AbstractVideoOutput::CLASS().getDerivedClasses( );

	int numCodecs = 0;
	foreach(ClassMap::value_type i, codecClasses)
	{
		if(i.second.isAbstract())
			continue;

		std::string codec = i.second.getMetaInfo("Codec");
		if (!codec.empty())
		{
			std::cout << std::endl << indent << codec << std::endl;

			AbstractVideoOutputPtr c(i.second.newInstance<AbstractVideoOutput>());
			ProgramOptions::Description desc;
			c->addProgramOptions(desc);
			foreach (auto o, desc.options())
				std::cout << indent << "  " << o->format_name() << " : " << o->description() << std::endl;

			++numCodecs;
		}
	}

	if (!numCodecs)
	{
		std::cout << indent << "NO CODECS FOUND" << std::endl;
		std::cout << indent << "you probably have no packages installed that provide video codecs" << std::endl;
	}
}

void usage()
{
	std::cout << "Usage:" << std::endl;
	std::cout << "    miratape <command> <parameters> [<options>]:" << std::endl;
	std::cout << std::endl;
	std::cout << "Available commands with their parameters and options are:" << std::endl << std::endl;
	std::cout << "    info <tapefile> : displays information about the tape" << std::endl<< std::endl;
	std::cout << "    rec <tapefile> : starts a framework and records channels to a tape." << std::endl;
	std::cout << "       --channels \"channel1,channel2,...\" : channels to record" << std::endl;
	std::cout << "       --messageblocksize : the maximum size of a message block in kB of the new tape file (default is 64 MB)" << std::endl<< std::endl;
	std::cout << "    repair <inputtapefile> <outputtapefile> : try to repair a tape" << std::endl<< std::endl;
	std::cout << "    play <tapefile> : starts a framework and plays a tape." << std::endl;
	std::cout << "       --close_when_finished : closes miratape after play back finished" << std::endl;
	std::cout << "       --use_original_timestamp : use the original recording time as base for play back - default=false" << std::endl;
	std::cout << "       --loop : play in a loop (disables close_when_finished)" << std::endl;
	std::cout << "       --channels \"channel1,channel2,...\" : limit play back to given channels" << std::endl;
	std::cout << "       --channels_not \"channel1,channel2,...\" : play all but given channels (ignored if '--channels' option is used)" << std::endl;
	std::cout << "       --first : specify relative time of first message to play in microseconds" << std::endl;
	std::cout << "       --last : specify relative time of last message to play in microseconds" << std::endl;
	std::cout << "       --time_scale : The scale factor for play back (1 = normal play back) - default=1)" << std::endl<< std::endl;
	std::cout << "    copy <inputtapefile> <outputtapefile> : copy tape file to a new one while optionally filtering and cutting the content." << std::endl;
	std::cout << "       --channels \"channel1,channel2,...\" : write the given channels only" << std::endl;
	std::cout << "       --channels_not \"channel1,channel2,...\" : write all but the given channels (ignored if '--channels' option is used)" << std::endl;
	std::cout << "       --compress [\"channel1,channel2,...\"] : specify the channels that should be stored compressed,\n"
			     "                                              if no channels are specified, all channels will be compressed" << std::endl;
	std::cout << "       --retype from=to : replace all occurrences of 'from' by 'to' in the stored type names (removes meta data information)" << std::endl;
	std::cout << "       --first : specify relative time of first message to copy in microseconds" << std::endl;
	std::cout << "       --first_and_rectime : like --first, but also offset recording time" << std::endl;
	std::cout << "                             (overrides --first, --recordingtime, --shiftmessageoffset)" << std::endl;
	std::cout << "       --last : specify relative time of last message to copy in microseconds" << std::endl;
	std::cout << "       --recordingtime : Sets the recording time to the specified value given as timestamp in microseconds" << std::endl;
	std::cout << "       --shiftmessageoffset : adjusts the time offset relative to the recording time of each message (given in microseconds" << std::endl;
	std::cout << "       --messageblocksize : the maximum size of a message block in kB of the new tape file (default is 64 MB)" << std::endl;
	std::cout << "       --drop_same_timestamp: drop message if same timestamp appeared in same channel within last n messages" << std::endl << std::endl;
	std::cout << "    transcode <inputtapefile> <outputtapefile> : transcodes the specified channels using a different codec" << std::endl;
	std::cout << "       --channels \"channel1=codec1,channel2=codec2\" : the channels and the target codecs" << std::endl;
	std::cout << "                                                        if no codec is given, no compression is used." << std::endl << std::endl;
	std::cout << "    merge <inputtapefileA> <inputtapefileB> <outputtapefile> : merge two tape files to a new one." << std::endl << std::endl;
	std::cout << "    writeJSON <tapefile> <outputfile> : writes tapefile in JSON format." << std::endl;
	std::cout << "       --channels \"channel1,channel2,...\" : use the given channels only" << std::endl;
	std::cout << "       --channels_not \"channel1,channel2,...\" : write all but given channels (ignored if '--channels' option is used)" << std::endl;
	std::cout << "       --first : specify relative time of first message to copy in microseconds" << std::endl;
	std::cout << "       --last : specify relative time of last message to copy in microseconds" << std::endl;
	std::cout << "       --matlab : write timestamps, channel values and frame ids as distinct arrays." << std::endl;
	std::cout << "                  This can significantly speed up matlab import (e.g. using jsonlab toolbox)." << std::endl << std::endl;
	std::cout << "    extractimages <tapefile> <imagedirectory> : writes all images stored in the specified tape into the given image directory." << std::endl;
	std::cout << "       --channels \"channel1,channel2,...\" : use the given channels only" << std::endl;
	std::cout << "       --first : specify relative time of first message to copy in microseconds" << std::endl;
	std::cout << "       --last : specify relative time of last message to copy in microseconds" << std::endl;
	std::cout << "       --scale : scale factor 'alpha' for cv::Mat::convertTo()" << std::endl;
	std::cout << "       --format [jpg/png/png16...] : image format that is used (png16 is 16bit png)" << std::endl << std::endl;
	std::cout << "    extractvideo <tapefile> <videofilename> : writes all images stored in the specified tape into a video." << std::endl;
	std::cout << "       --channels \"channel1,channel2,...\" : use the given channels only" << std::endl;
	std::cout << "       --first : specify relative time of first message to copy in microseconds" << std::endl;
	std::cout << "       --last : specify relative time of last message to copy in microseconds" << std::endl;
	std::cout << "       --skip : specify number of images to skip at beginning of tape" << std::endl;
	std::cout << "       --codec : specify video codec - available codecs and their specific options are listed below" << std::endl;
	listVideoCodecs("         ");
	exit(-1);
}

int record(Framework* fw, const std::string& file)
{
	TapeRecorder r;
	MIRA_CMDLINE.getDescriptions().add_options()
			("channels", boost::program_options::value<std::string>(), "Channels to record file")
			("messageblocksize", boost::program_options::value<uint32>()->default_value(64*1024), "");

	ProgramOptions::VariableMap vmap = MIRA_CMDLINE.getOptions();

	if ( vmap.count("channels") )
	{
		std::string txt = vmap["channels"].as<std::string>();
		std::vector<std::string> channels;
		boost::split( channels, txt, boost::is_from_range(',',',') );
		foreach(const std::string& a, channels)
		{
			r.addChannel(a);
		}
	}
	uint32 blocksize = vmap["messageblocksize"].as<uint32>();
	r.setMaxMessageBlockSize(blocksize*1024);
	r.record(file);
	int res = fw->run();
	r.stop();
	return res;
}

int play(Framework* fw, const std::string& file)
{
	Tape tape;
	tape.open(file, Tape::READ);
	TapeVisitor visitor;
	TapePlayer p;

	MIRA_CMDLINE.getDescriptions().add_options()
			("channels", boost::program_options::value<std::string>(), "Channels to play")
			("channels_not", boost::program_options::value<std::string>(), "Channels not to play")
			("close_when_finished", "closes miratape after play back finished - default=false")
			("first", boost::program_options::value<int64>()->default_value(std::numeric_limits<int64>::min()/1000), "relative time of first message in microseconds")
			("last", boost::program_options::value<int64>()->default_value(std::numeric_limits<int64>::max()/1000), "relative time of last message in microseconds")
			("loop", "play in a loop (disables close_when_finished) - default=false")
			("time_scale", boost::program_options::value<float>()->default_value(1.0f), "The scale factor for play back (1 = normal play back) - default=1)")
			("use_original_timestamp", "use the original recording time as base for play back - default=false");

	ProgramOptions::VariableMap vmap = MIRA_CMDLINE.getOptions();

	Duration first = Duration::microseconds(vmap["first"].as<int64>());
	Duration last = Duration::microseconds(vmap["last"].as<int64>());
	bool closeWhenFinished = vmap.count("close_when_finished") > 0;
	bool useOriginalTimestamp = vmap.count("use_original_timestamp") > 0;
	if ( vmap.count("loop") )
	{
		closeWhenFinished = false;
		p.setLoop(true);
	}
	p.setTimeScaler(vmap["time_scale"].as<float>());

	if ( vmap.count("channels") )
	{
		std::string txt = vmap["channels"].as<std::string>();
		std::vector<std::string> channels;
		boost::split( channels, txt, boost::is_from_range(',',',') );
		visitor.visit(&tape, channels, first, last);
	}
	else if  ( vmap.count("channels_not") )
	{
		std::string txt = vmap["channels_not"].as<std::string>();
		std::vector<std::string> channels_not;
		boost::split( channels_not, txt, boost::is_from_range(',',',') );

		std::vector<std::string> channels;
		foreach(auto c, tape.getChannels()) {
			if (find(channels_not.begin(), channels_not.end(), c.first) == channels_not.end())
				channels.push_back(c.first);
		}

		visitor.visit(&tape, channels, first, last);
	}
	else
		visitor.visit(&tape, first, last);

	if (useOriginalTimestamp)
		p.load(&visitor, visitor.getStartTime());
	else
		p.load(&visitor);
	p.play();
	int res = 0;
	if (closeWhenFinished)
	{
		try
		{
			fw->load();
			fw->start();
			while(!boost::this_thread::interruption_requested())
			{
				if (!p.isPlaying())
					break;
				MIRA_SLEEP(100);
			}
		}
		catch(std::exception& ex)
		{
			MIRA_LOG_EXCEPTION(ERROR, ex) << "Exception:\n";
			res = -1;
		}
	}
	else
		res = fw->run();
	return res;
}

int copy(const std::string& infile, const std::string& outfile)
{
	Tape intape;
	Tape outtape;
	TapeVisitor visitor;

	intape.open(infile, Tape::READ);

	MIRA_CMDLINE.getDescriptions().add_options()
			("channels", boost::program_options::value<std::string>(), "Channels to copy")
			("channels_not", boost::program_options::value<std::string>(), "Channels not to copy")
			("compress", boost::program_options::value<std::string>()->implicit_value(""), "Should the data be compressed")
			("retype", boost::program_options::value<std::string>(), "Retype channel data type from=to")
			("first", boost::program_options::value<int64>()->default_value(std::numeric_limits<int64>::min()/1000), "relative time of first message in microseconds")
			("first_and_rectime", boost::program_options::value<int64>(), "relative time of first message in microseconds, also offset recording time")
			("last", boost::program_options::value<int64>()->default_value(std::numeric_limits<int64>::max()/1000), "relative time of last message in microseconds")
			("recordingtime", boost::program_options::value<int64>(), "use the specified time as new start of recording (unix timestamp in microseconds)")
			("shiftmessageoffset", boost::program_options::value<int64>(), "adjust the time offset relative to the recording time of each message (given in microseconds)")
			("messageblocksize", boost::program_options::value<uint32>()->default_value(64*1024), "")
			("drop_same_timestamp", boost::program_options::value<uint32>()->default_value(0), "drop message if same timestamp appeared in same channel within last n messages");

	ProgramOptions::VariableMap vmap = MIRA_CMDLINE.getOptions();

	Duration first = Duration::microseconds(vmap["first"].as<int64>());
	if ( vmap.count("first_and_rectime") )
		first = Duration::microseconds(vmap["first_and_rectime"].as<int64>());
	Duration last = Duration::microseconds(vmap["last"].as<int64>());
	uint32 blocksize = vmap["messageblocksize"].as<uint32>();
	uint32 dropSameTimestamp = vmap["drop_same_timestamp"].as<uint32>();

	if ( vmap.count("channels") )
	{
		std::string txt = vmap["channels"].as<std::string>();
		std::vector<std::string> channels;
		boost::split( channels, txt, boost::is_from_range(',',',') );
		visitor.visit(&intape, channels, first, last);
	}
	else if  ( vmap.count("channels_not") )
	{
		std::string txt = vmap["channels_not"].as<std::string>();
		std::vector<std::string> channels_not;
		boost::split( channels_not, txt, boost::is_from_range(',',',') );

		std::vector<std::string> channels;
		foreach(auto c, intape.getChannels()) {
			if (find(channels_not.begin(), channels_not.end(), c.first) == channels_not.end())
				channels.push_back(c.first);
		}

		visitor.visit(&intape, channels, first, last);
	}
	else
		visitor.visit(&intape, first, last);

	std::set<std::string> compress;
	bool compressall = false;
	if ( vmap.count("compress") )
	{
		std::string txt = vmap["compress"].as<std::string>();
		if(txt.empty())
			compressall=true;
		std::vector<std::string> channels;
		boost::split( channels, txt, boost::is_from_range(',',',') );
		foreach(const std::string& s, channels)
			compress.insert(s);
	}

	std::vector<std::string> retype;
	if ( vmap.count("retype") )
	{
		std::string txt = vmap["retype"].as<std::string>();
		boost::split(retype, txt, boost::is_from_range('=','=') );
		if(retype.size()!=2) {
			std::cout << "Invalid --retype parameter! Usage: --retype from=to" << std::endl;
			return -1;
		}
	}

	Time startTime = visitor.getStartTime();
	if ( vmap.count("first_and_rectime") )
		startTime += first;
	else if ( vmap.count("recordingtime") )
		startTime = Time::fromUnixNS(vmap["recordingtime"].as<int64>()*1000);

	Duration shiftOffset = Duration::seconds(0);
	if ( vmap.count("first_and_rectime") )
		shiftOffset = -first;
	else if ( vmap.count("shiftmessageoffset") )
		shiftOffset = Duration::nanoseconds(vmap["shiftmessageoffset"].as<int64>()*1000);

	Duration totalTime = visitor.getLastMessageTimeOffset()-visitor.getFirstMessageTimeOffset();

	// prepare the output tape
	outtape.open(outfile, Tape::WRITE);
	outtape.setMaxMessageBlockSize(blocksize*1024);
	outtape.setStartTime(startTime);

	int i=0;
	int messageCount = 0;
	int totalMessages = (int)visitor.getMessageCount();
	Time copyStart = Time::now();
	printProgressBarHeader();
	std::map<std::string, std::list<Time>> timestamps;
	for(TapeVisitor::iterator it = visitor.begin(); it!=visitor.end(); ++it,++messageCount)
	{
		printProgressBar(messageCount, totalMessages, copyStart, 100);

		const Tape::ChannelInfo* info = it.getChannelInfo();
		Time t = startTime + it.getTimeOffset() + shiftOffset;

		if (dropSameTimestamp &&
			std::find(timestamps[info->name].rbegin(),
			          timestamps[info->name].rend(), t) != timestamps[info->name].rend())
		{
			continue;
		}
		// keep previous compression state, compress all, or those that were specified
		bool compressChannel = it->compressed || compressall || compress.count(info->name)>0;

		if(!retype.empty())
			// change the channels type
			std::string msgtype = boost::replace_all_copy(info->type, retype[0], retype[1]);

		outtape.write(info->name, info->type, t, it->frameID, it->sequenceID,
		              it->data, compressChannel, info->meta, info->metaDB);

		if (dropSameTimestamp)
		{
			timestamps[info->name].push_back(t);
			if (timestamps[info->name].size() > dropSameTimestamp)
				timestamps[info->name].pop_front();
		}
	}
	printFinishedProgressBar();

	return 0;
}

int merge(const std::string& infileA, const std::string& infileB, const std::string& outfile)
{
	Tape intapeA,intapeB;
	Tape outtape;
	TapeVisitor visitor;

	// open tape files
	intapeA.open(infileA, Tape::READ);
	intapeB.open(infileB, Tape::READ);

	// set some default parameter
	Duration first = Duration::microseconds(std::numeric_limits<int64>::min()/1000);
	Duration last = Duration::microseconds(std::numeric_limits<int64>::max()/1000);
	uint32 blocksize = 64*1024;

	// visit both tape files
	visitor.visit(&intapeA, first, last);
	visitor.visit(&intapeB, first, last);

	Time startTime = visitor.getStartTime();
	Duration totalTime = visitor.getLastMessageTimeOffset()-visitor.getFirstMessageTimeOffset();

	// prepare the output tape
	outtape.open(outfile, Tape::WRITE);
	outtape.setMaxMessageBlockSize(blocksize*1024);
	outtape.setStartTime(startTime);

	int messageCount = 0;
	int totalMessages = (int)visitor.getMessageCount();
	Time copyStart = Time::now();
	printProgressBarHeader();
	for(TapeVisitor::iterator it = visitor.begin(); it!=visitor.end(); ++it,++messageCount)
	{
		printProgressBar(messageCount, totalMessages, copyStart, 100);

		const Tape::ChannelInfo* info = it.getChannelInfo();
		Time t = startTime + it.getTimeOffset();
		outtape.write(info->name,info->type, t, it->frameID, it->sequenceID,
		              it->data, it->compressed, info->meta, info->metaDB);
	}
	printFinishedProgressBar();

	return 0;
}

// for current tape versions
template<typename ImgType>
bool tryDeserializeImage(BinaryBufferDeserializer& bs, const std::string& type, Img<>& oImg)
{
	if(type==typeName<ImgType>()) {
		ImgType data;
		bs.deserialize(data, false);
		oImg  = data;
		return true;
	}
	return false;
}

template<typename T>
bool tryDeserializeImages(BinaryBufferDeserializer& bs, const std::string& type, Img<>& oImg)
{
	return
		tryDeserializeImage<Img<T,1>>(bs,type,oImg) ||
		tryDeserializeImage<Img<T,2>>(bs,type,oImg) ||
		tryDeserializeImage<Img<T,3>>(bs,type,oImg) ||
		tryDeserializeImage<Img<T,4>>(bs,type,oImg);
}

bool deserializeImage(const Buffer<uint8>& data, const std::string& type, Img<>& oImg)
{
	BinaryBufferDeserializer bs(const_cast<Buffer<uint8>*>(&data));
	return tryDeserializeImage<Img<>>  (bs, type, oImg) ||
	       tryDeserializeImages<float> (bs, type, oImg) ||
	       tryDeserializeImages<uint8> (bs, type, oImg) ||
	       tryDeserializeImages<uint16>(bs, type, oImg);
}

inline std::string replaceSlashes(const std::string& s)
{
	std::string tag = s;
	for(std::size_t i=0; i<tag.size(); ++i)
	{
		if(tag[i]=='/' && i!=0)
			tag[i]='_';
	}

	return tag;
}

int extractimages(const std::string& tapefile, const std::string& imagedir)
{
	Tape intape;
	TapeVisitor visitor;

	if (!boost::filesystem::is_directory(imagedir)) {
		std::cout << "ERROR: Directory " << imagedir << " does not exist. Abort.eog" << std::endl;
		return -1;
	}

	// load manifests to enable all available (image) codecs
	// TODO: "fix class identifier already in use" problem
	//loadManifests();

	intape.open(tapefile, Tape::READ);

	MIRA_CMDLINE.getDescriptions().add_options()
			("channels", boost::program_options::value<std::string>(), "")
			("first", boost::program_options::value<int64>()->default_value(std::numeric_limits<int64>::min()/1000), "")
			("last", boost::program_options::value<int64>()->default_value(std::numeric_limits<int64>::max()/1000), "")
			("scale", boost::program_options::value<float>()->default_value(1.0f), "")
			("format", boost::program_options::value<std::string>()->default_value("jpg"), "");

	ProgramOptions::VariableMap vmap = MIRA_CMDLINE.getOptions();

	Duration first = Duration::microseconds(vmap["first"].as<int64>());
	Duration last = Duration::microseconds(vmap["last"].as<int64>());
	float scale = vmap["scale"].as<float>();
	std::string format = vmap["format"].as<std::string>();
	std::string ext = "." + format; // image file extension

	if ( vmap.count("channels") )
	{
		std::string txt = vmap["channels"].as<std::string>();
		std::vector<std::string> channels;
		boost::split( channels, txt, boost::is_from_range(',',',') );
		visitor.visit(&intape, channels, first, last);
	}
	else
		visitor.visit(&intape, first, last);

	Time startTime = visitor.getStartTime();
	Duration totalTime = visitor.getLastMessageTimeOffset()-visitor.getFirstMessageTimeOffset();

	int i=0;
	int messageCount = 0;
	int totalMessages = (int)visitor.getMessageCount();
	Time copyStart = Time::now();
	printProgressBarHeader();
	for(TapeVisitor::iterator it = visitor.begin(); it!=visitor.end(); ++it,++messageCount)
	{
		printProgressBar(messageCount, totalMessages, copyStart, 100);

		const Tape::ChannelInfo* info = it.getChannelInfo();
		Time t = startTime + it.getTimeOffset();

		Img<> img;
		if(deserializeImage(it->data, info->type, img)) {
			Time timestamp = startTime + it.getTimeOffset();

			cv::Mat mat = img;
			cv::Mat dest;
			if(format=="png16") {
				mat.convertTo(dest,CV_16UC1,scale);
				ext = ".png";
			} else
				mat.convertTo(dest,CV_8U,scale);

			Path p = imagedir;

			Path filename = p / Path(replaceSlashes(info->name+toString(timestamp.toUnixNS())+ext));
			cv::imwrite(filename.string(),dest);
		} else {
			//std::cout << "No image type" << std::endl;
		}
	}
	printFinishedProgressBar();

	return 0;
}

int extractvideo(const std::string& tapefile, const std::string& videoname)
{
	Tape intape;
	TapeVisitor visitor;

	intape.open(tapefile, Tape::READ);

	MIRA_CMDLINE.getDescriptions().add_options()
			("channels", boost::program_options::value<std::string>(), "")
			("first", boost::program_options::value<int64>()->default_value(std::numeric_limits<int64>::min()/1000), "")
			("last", boost::program_options::value<int64>()->default_value(std::numeric_limits<int64>::max()/1000), "")
			("skip", boost::program_options::value<int32>()->default_value(0), "")
			("codec", boost::program_options::value<std::string>()->default_value("mpeg2video"), "");

	ProgramOptions::VariableMap vmap = MIRA_CMDLINE.getOptions();

	Duration first = Duration::microseconds(vmap["first"].as<int64>());
	Duration last = Duration::microseconds(vmap["last"].as<int64>());
	int skip = vmap["skip"].as<int32>();
	std::string codec = vmap["codec"].as<std::string>();

	// find AbstractVideoOutput with the specified codec

	// query all view classes from the class factory:
	auto classes = AbstractVideoOutput::CLASS().getClassByMeta("Codec", codec);
	if(classes.empty()) {
		MIRA_LOG(ERROR) << "Did not find a plugin for video codec '" << codec << "'.\nThe FFmpeg package provides plugins for video export, but not all codecs are avalailable as miratape plugins yet.";
		return -1;
	}

	// create and open the output plugin
	AbstractVideoOutputPtr plugin(classes.begin()->newInstance<AbstractVideoOutput>());
	MIRA_LOG(NOTICE) << "Writing file using video codec '" << plugin->getClass().getMetaInfo("Codec") << "'";
	plugin->addProgramOptions(MIRA_CMDLINE.getDescriptions());
	plugin->setProgramParameters(MIRA_CMDLINE.getOptions());
	plugin->open(videoname);

	if ( vmap.count("channels") )
	{
		std::string txt = vmap["channels"].as<std::string>();
		std::vector<std::string> channels;
		boost::split( channels, txt, boost::is_from_range(',',',') );
		visitor.visit(&intape, channels, first, last);
	}
	else
		visitor.visit(&intape, first, last);

	Time startTime = visitor.getStartTime().toLocal();
	Duration totalTime = visitor.getLastMessageTimeOffset()-visitor.getFirstMessageTimeOffset();

	int i=0;
	int messageCount = 0;
	int totalMessages = (int)visitor.getMessageCount();
	Time copyStart = Time::now();
	printProgressBarHeader();
	for(TapeVisitor::iterator it = visitor.begin(); it!=visitor.end(); ++it,++messageCount)
	{
		printProgressBar(messageCount, totalMessages, copyStart, 100);

		const Tape::ChannelInfo* info = it.getChannelInfo();

		Img<> img;
		if(deserializeImage(it->data, info->type, img)) {
			Time timestamp = startTime + it.getTimeOffset();

			if (i >= skip)
				plugin->encode(img, timestamp);
			++i;
		} else {
			//std::cout << "No image type" << std::endl;
		}
	}
	printFinishedProgressBar();

	plugin->close();
	return 0;
}

int writeJSON(const std::string& tapefile, const std::string& outputFile)
{
	Tape intape;

	if (boost::filesystem::exists( outputFile )) {
		std::cout << "ERROR: File " << outputFile << " does already exist! Abort.eog" << std::endl;
		return -1;
	}

	intape.open(tapefile, Tape::READ);

	MIRA_CMDLINE.getDescriptions().add_options()
			("channels", boost::program_options::value<std::string>(), "")
			("channels_not", boost::program_options::value<std::string>(), "")
			("first", boost::program_options::value<int64>()->default_value(std::numeric_limits<int64>::min()/1000), "")
			("last", boost::program_options::value<int64>()->default_value(std::numeric_limits<int64>::max()/1000), "")
			("matlab", "");

	ProgramOptions::VariableMap vmap = MIRA_CMDLINE.getOptions();

	Duration first = Duration::microseconds(vmap["first"].as<int64>());
	Duration last = Duration::microseconds(vmap["last"].as<int64>());

	bool matlab = vmap.count("matlab") != 0;

	std::vector<std::string> channels;
	if ( vmap.count("channels") )
	{
		std::string txt = vmap["channels"].as<std::string>();
		boost::split( channels, txt, boost::is_from_range(',',',') );
	}
	else if  ( vmap.count("channels_not") )
	{
		std::string txt = vmap["channels_not"].as<std::string>();
		std::vector<std::string> channels_not;
		boost::split( channels_not, txt, boost::is_from_range(',',',') );

		foreach(auto ch, intape.getChannels()) {
			if (find(channels_not.begin(), channels_not.end(), ch.first) == channels_not.end())
				channels.push_back(ch.first);
		}
	}
	else {
		foreach( auto ch, intape.getChannels() )
				channels.push_back( ch.first );
	}

	std::fstream fs;
	fs.open(outputFile, std::fstream::out);
	// begin JSON-Object
	fs << "{\n";

	int cId = 0;
	printProgressBarHeader();
	foreach( std::string const& channel, channels ) {
		// begin JSON-Array
		fs << (cId > 0 ? "," : "") << "\"" << channel << "\": [\n";
		TapeVisitor visitor;
		visitor.visit(&intape, std::vector<std::string>(1,channel), first, last);

		Time startTime = visitor.getStartTime();
		Duration totalTime = visitor.getLastMessageTimeOffset()-visitor.getFirstMessageTimeOffset();

		int i=0;
		int messageCount = 0;
		int totalMessages = (int)visitor.getMessageCount();
		Time copyStart = Time::now();
		if ( !matlab ) {
			// write channel value, timestamp and frameID as JSON-Object
			for(TapeVisitor::iterator it = visitor.begin(); it!=visitor.end(); ++it,++messageCount)
			{
				printProgressBar(((float(messageCount)/totalMessages)*100+0.5)+cId*100, channels.size()*100, copyStart, 10);

				Tape::ChannelInfo const* info = it.getChannelInfo();

				Stamped<JSONValue> value;
				it.readJSON( value );

				// write timestamp, value and frameID
				fs << (i++ > 0 ? ",\n" : "") <<
						"{ \"Timestamp\":" << value.timestamp.toUnixNS() <<
						", \"Value\":" << value.value() <<
						", \"FrameID\":\"" << value.frameID << "\"}";
			}
		}
		else {
			// begin JSON Object
			fs << "{";
			for ( int elem = 0; elem<3; ++elem) {
				// write channel value, timestamp and frameID to separate JSON-Arrays
				switch ( elem ) {
					case 0: fs << "\"Timestamp\":";break;
					case 1: fs << ",\"Value\":";break;
					case 2: fs << ",\"FrameID\":";break;
				}
				// begin JSON Array
				fs << "[";
				int j=0;
				for(TapeVisitor::iterator it = visitor.begin(); it!=visitor.end(); ++it,++messageCount)
				{
					printProgressBar(((float(messageCount)/(totalMessages*3))*100.0+0.5)+cId*100, channels.size()*100, copyStart, 10);

					Tape::ChannelInfo const* info = it.getChannelInfo();

					Stamped<JSONValue> value;
					it.readJSON( value );

					// write timestamp or value or frameID
					fs << (j++ > 0 ? ",\n" : "");
					switch ( elem ) {
						case 0: fs << value.timestamp.toUnixNS();break;
						case 1: fs << value.value();break;
						case 2: fs << "\"" <<value.frameID << "\"";break;
					}
				}
				// end JSON Array
				fs << "]";
			}
			// end JSON Object
			fs << "}";
		}
		// end JSON-Array
		fs << "]";
		++cId;
	}
	printFinishedProgressBar();
	// close JSON-Object
	fs << "}";
	fs.close();

	return 0;
}

int transcode(const std::string& infile, const std::string& outfile)
{
	Tape intape;
	Tape outtape;
	TapeVisitor visitor;

	intape.open(infile, Tape::READ);

	MIRA_CMDLINE.getDescriptions().add_options()
			("channels", boost::program_options::value<std::string>(), "")
			("messageblocksize", boost::program_options::value<uint32>()->default_value(64*1024), "");

	ProgramOptions::VariableMap vmap = MIRA_CMDLINE.getOptions();

	uint32 blocksize = vmap["messageblocksize"].as<uint32>();



	// get avail codecs:
	std::map<std::string, ClassProxy> availCodecs;
	std::string availCodecsStr;
	auto codecClasses = BinarySerializerCodec::CLASS().getDerivedClasses();

	foreach(auto i, codecClasses)
	{
		if(i.second.isAbstract())
			continue;

		std::string name = i.second.getMetaInfo("Name");
		if(name.empty())
			name = i.first; // use class name, if no meta name was specified

		availCodecs[name] = i.second;
		availCodecsStr+= name + ", ";
	}

	std::map<std::string,BinarySerializerCodecPtr> transcodeCodecMap;
	if ( vmap.count("channels") )
	{
		std::string txt = vmap["channels"].as<std::string>();
		std::vector<std::string> vals;
		boost::split( vals, txt, boost::is_from_range(',',',') );
		foreach(std::string& v, vals)
		{
			std::vector<std::string> p;
			boost::split(p, v, boost::is_from_range('=','=') );
			if(p.size()!=2) {
				MIRA_LOG(ERROR) << "Invalid syntax '" << v << "'. Expected 'channel=codec' or 'channel='";
				return -1;
			}

			// create an instance of the codec
			if(p[1].empty()) {
				transcodeCodecMap[p[0]].reset(); // NULL to mark that no codec is to be used
			} else {
				auto f = availCodecs.find(p[1]);
				if(f==availCodecs.end()) {
					MIRA_LOG(ERROR) << "Codec '" << p[1] << "' does not exist. Available codecs are: " + availCodecsStr;
					return -1;
				}
				transcodeCodecMap[p[0]].reset(f->second.newInstance<BinarySerializerCodec>());
			}
		}
	}



	visitor.visit(&intape);

	Time startTime = visitor.getStartTime();
	Duration totalTime = visitor.getLastMessageTimeOffset()-visitor.getFirstMessageTimeOffset();

	// prepare the output tape
	outtape.open(outfile, Tape::WRITE);
	outtape.setMaxMessageBlockSize(blocksize*1024);
	outtape.setStartTime(startTime);

	int i=0;
	int messageCount = 0;
	int totalMessages = (int)visitor.getMessageCount();
	Time copyStart = Time::now();
	printProgressBarHeader();
	for(TapeVisitor::iterator it = visitor.begin(); it!=visitor.end(); ++it,++messageCount)
	{
		printProgressBar(messageCount, totalMessages, copyStart, 100);

		const Tape::ChannelInfo* info = it.getChannelInfo();
		Time t = startTime + it.getTimeOffset();


		auto f = transcodeCodecMap.find(info->name);
		if(f!=transcodeCodecMap.end()) {
			// supporting images only at the moment, try to deserialize the image
			Img<> img;
			if(!deserializeImage(it->data, info->type, img)) {
				MIRA_LOG(ERROR) << "Currently images are supported for transcoding only, but '" <<
						info->name << "' does not contain images. Abort!";
				return -1;
			}

			Buffer<uint8> serializedValue;
			BinaryBufferSerializer bs(&serializedValue);

			// register codec
			if(f->second)
				bs.registerCodec(f->second);

			// serialize the data
			bs.serialize(img,false);

			outtape.write(info->name, info->type, t, it->frameID, it->sequenceID,
			              serializedValue, it->compressed, info->meta, info->metaDB);
		} else { // just copy
			outtape.write(info->name, info->type, t, it->frameID, it->sequenceID,
		                  it->data, it->compressed, info->meta, info->metaDB);
		}
	}
	printFinishedProgressBar();

	return 0;
}

int main(int argc, char** argv)
{
	try {

		std::vector<std::string> fwArgs;
		for (int i=3; i<argc; ++i)
			fwArgs.push_back(argv[i]);
		// create a framework
		Framework fw(fwArgs);

		if ( argc < 3 )  // must happen after the fw is created,
			usage();	 // otherwise querying the video codecs won't work

		std::string cmd(argv[1]);

		// COMMAND: info
		if ( cmd == "info" )
		{
			Tape t;
			t.open(argv[2], Tape::INFO);

			Tape::ChannelMap channels = t.getChannels();

			uint32 v = t.getVersion();
			uint32 major = MIRA_MAJOR_VERSION(v);
			uint32 minor = MIRA_MINOR_VERSION(v);
			uint64 startTime = t.getStartTime().toUnixNS() / 1000;
			std::cout << "Tape file:\n"
					<< "  version    : " << major << "." << minor << "\n"
					<< "  recorded   : " << t.getLocalStartTime() << " (" << startTime  << ")" << "\n"
					<< "  channels   : " << channels.size() << "\n" << std::endl;

			foreach(auto c, channels) {
				Duration d = c.second.lastMessageOffset - c.second.firstMessageOffset;
				std::cout << " Channel " << c.first << "\n"
				<< "   type      : " << c.second.type << "\n";
				if (c.second.meta)
					std::cout << "   meta      : " << c.second.meta->toString() << "\n";
				std::cout << "   messages  : " << c.second.messages.size() << "\n"
				<< "   first     : " << c.second.firstMessageOffset << "\n"
				<< "   last      : " << c.second.lastMessageOffset << "\n";
				if(d.totalMilliseconds()>0 && c.second.messages.size()>1) {
					float rate = c.second.messages.size() / ((float)d.totalMilliseconds()/1000.0f);
					std::cout << "   avg.rate  : " << rate << " Hz\n";
				}
				std::stringstream ss;

				if (c.second.dataSize > 0 && c.second.messages.size()>0) {
					if (c.second.dataSize < 1000)
						std::cout << "   total size: " << c.second.dataSize << " Byte\n";
					else if (c.second.dataSize < 1000000)
						std::cout << "   total size: " << boost::format("%.1f KByte\n") % (c.second.dataSize/1000.0);
					else if (c.second.dataSize < 1000000000)
						std::cout << "   total size: " << boost::format("%.2f MByte\n") % (c.second.dataSize/1000000.0);
					else
						std::cout << "   total size: " << boost::format("%.2f GByte\n") % (c.second.dataSize/1000000000.0);
					double avgSize = c.second.dataSize / c.second.messages.size();
					if (avgSize < 1000)
						std::cout << "   avg.size  : " << avgSize << " Byte\n";
					else if (avgSize < 1000000)
						std::cout << "   avg.size  : " << boost::format("%.1f KByte\n") % (avgSize/1000.0);
				}
				std::cout << std::endl;
			}
			return 0;
		}

		if ( cmd == "repair" )
		{
			if ( argc < 4 )
				usage();
			Tape t;
			t.repair(argv[2], argv[3]);
			return 0;
		}

		// COMMAND: copy
		if ( cmd == "copy" )
		{
			if ( argc < 4 )
				usage();
			return copy(argv[2], argv[3]);
		}
		// COMMAND: copy
		if ( cmd == "transcode" )
		{
			if ( argc < 4 )
				usage();
			return transcode(argv[2], argv[3]);
		}
		// COMMAND: merge
		if ( cmd == "merge" )
		{
			if ( argc < 5 )
				usage();
			return merge(argv[2], argv[3], argv[4]);
		}
		// COMMAND: extractimages
		else if ( cmd == "extractimages" )
		{
			if ( argc < 4 )
				usage();
			return extractimages(argv[2], argv[3]);
		}
		// COMMAND: extractvideo
		else if ( cmd == "extractvideo" )
		{
			if ( argc < 4 )
				usage();
			return extractvideo(argv[2], argv[3]);
		}
		else if ( cmd == "writeJSON" )
		{
			if ( argc < 4 )
				usage();
			return writeJSON(argv[2], argv[3]);
		}

		int res;

		// COMMAND: rec
		if ( cmd == "rec" )
			res = record(&fw, argv[2]);

		// COMMAND: play
		else if ( cmd == "play" )
			res = play(&fw, argv[2]);
		else
			usage();

		MIRA_LOG(NOTICE) << "Exiting application";
		return res;
	}
	catch(Exception& ex)
	{
		std::cerr << std::endl;
		std::cerr << "An exception has occurred: " << std::endl;
		std::cerr << ex.what() << std::endl;
		std::cerr << "[CALL STACK]: " << std::endl;
		std::cerr << ex.callStack();
		return -1;
	}
	catch(std::exception& ex)
	{
		std::cerr << std::endl;
		std::cerr << "An exception has occurred: " << std::endl;
		std::cerr << ex.what() << std::endl;
		return -1;
	}

	// we should never reach here
	return 0;
}
