/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

#include "core/MIRAPackage.h"
#include "core/Tools.h"

#include <iostream>

#include <QDir>

#include <boost/algorithm/string.hpp>
#include <boost/filesystem/operations.hpp>

#include <platform/Platform.h>
// for touching the CMakeLists.txt
#ifdef MIRA_WINDOWS
#include <sys/utime.h>
#define utime _utime
#else
#include <sys/types.h>
#include <utime.h>
#endif

#include <utils/PathFinder.h>

#include "core/Url.h"
#include "core/Repository.h"
#include "core/Package.h"
#include "core/PackageParser.h"

using namespace std;

namespace mira {

///////////////////////////////////////////////////////////////////////////////

MIRAPackage::MIRAPackage()
{
}

MIRAPackage::~MIRAPackage()
{
}

void MIRAPackage::load( std::string path )
{
	Path dbPath = path.empty() ? (getAppDataDirectory() / DATABASEXML_FILENAME) : path;

	// load the content of the configuration file into a XML node
	XMLDom config;
	config.loadFromFile(dbPath);

	// get the content from the XML node
	loadContent(config);
}

void MIRAPackage::save( std::string path )
{
	Path dbPath = path.empty() ? (getAppDataDirectory() / DATABASEXML_FILENAME) : path;

	// Load the original configuration: This way the content written by
	// MIRAPackageGUI keeps there, even if we're running only MIRAPackage
	// (by using the command line).
	XMLDom config;
	if (boost::filesystem::exists(dbPath))
		config.loadFromFile(dbPath);

	// save the content to the XML node
	saveContent(config);

	// save the configuration to the file
	config.saveToFile(dbPath);
}

void MIRAPackage::exportDB( string const& path )
{
	Database exportDB;
	map<Path,string> miraPaths;
	mDB.exportInstalledPackages( exportDB, miraPaths );

	confirmExportPackages( exportDB, miraPaths );
	XMLDom content;
	exportDB.store( content );
	content.saveToFile(path);
}

void MIRAPackage::importDB( std::string const& path )
{
	mDB.importRepositories( path );
	reindexWebRepositories(false);
	checkForInstalled();

	Database tmpDB;
	tmpDB.load( path );

	foreach( PackageGroup const* group, tmpDB.getRootPackage()->mSubPackages ) {
		PackageGroup* localGrp = mDB.getIdentical( group );
		Package* localPkg = dynamic_cast<Package*>( localGrp );
		if ( localPkg ) {
			// add package to install plan (do not check for dependencies here!)
			addPackageToCheckoutPlan( localPkg, false );
		}
		else {
			// we cannot find the package
			vector<string> tOptions;
			tOptions.push_back("yes");
			tOptions.push_back("no");
			int selection = errorMessage( "Cannot find package \""
					+ group->mName + "\" during import!",
					"Do you wan't to proceed?",&tOptions);
			if ( selection == 1 )
				return;
		}
	}
	// now we can check if all dependencies are fulfilled (which should be the case)
	foreach( PackageGroup const* group, tmpDB.getRootPackage()->mSubPackages ) {
		PackageGroup* localGrp = mDB.getIdentical( group );
		Package* localPkg = dynamic_cast<Package*>( localGrp );
		if ( localPkg ) {
			// check the dependencies
			Package* tDep = mDB.getInstallDependencies( *localPkg, true );
			if ( tDep ) {
				resolveDependencies( tDep );
			}
		}
		// no else case: do not bother the user with the same error message
		// already shown in the first loop
	}
}

void MIRAPackage::resolveMIRAPath( bool forceSelection )
{
	// try to set mira working path
	if ( PathProvider::miraPaths().size() > 1 || forceSelection ) {
		// try to select one out of the different paths present
		string selected = selectMIRAPath();
		if ( selected.empty() )
			MIRA_THROW( XLogical, "No MIRA_PATH selected" );
		mInstallRoot = selected;
	}
	else if ( PathProvider::miraPaths().size() == 0 ) {
		// we don't have a mira path at all -> thats no good
		MIRA_THROW( XLogical, "No MIRA_PATH found" );
	}
	else
		mInstallRoot = PathProvider::miraPaths()[0].string();
}

void MIRAPackage::clearRepositories()
{
	mDB.clearRepositories();
}

void MIRAPackage::addRepository( RepositoryPtr repo  )
{
	mDB.addRepository( repo );
}

void MIRAPackage::addRepository(std::string const& iName,
                                std::string const& iDescription,
                                std::string const& iUrl,
                                std::string const& iIndexFile,
                                uint priority, std::string const& iType )
{
	RepositoryPtr tRepository = Repository::create(iName, iDescription,
	                                               iUrl,
	                                               iIndexFile, priority, iType);
	addRepository( tRepository );
}

void MIRAPackage::reindexWebRepositories( bool thorough)
{
	mDB.clearPackages();
	typedef std::pair<std::string, RepositoryPtr> repoType;
	foreach( repoType const& r, mDB.repos) {
		reindexWebRepository(r.second, thorough);
	}
	mDB.buildTagIndex();
}

void MIRAPackage::reindexWebRepository(RepositoryPtr repo, bool thorough)
{
	if(repo->url.empty())
		return;
	if(!repo->enabled)
		return;

	statusProgress( "Examining: " + repo->url, "Reindexing ...", 0, 100);
	vector<Package*> packages;
	repo->examine( packages, thorough, boost::bind(&MIRAPackage::progressReceiver,this, _1, _2, _3) );

	foreach(Package* package, packages)
	{
//		statusProgress( "Indexing: " + s, "Reindexing ...", i*100/repo->packageList.size(), 100);
//		indexWebPackage(repo, s, mDB);
		mDB.addPackage( package );
	}
	
	mDB.buildTagIndex();
	statusProgress( "done", "Reindexing ...", 100, 100);
}


void MIRAPackage::checkForInstalled()
{
	statusMessage( "Indexing local packages" );
	// try to find all installed packages
	vector<Path> packages = findProjectFiles( "*.package", true );
	vector<Path> extraPackages = findProjectFiles( "*.extpackage", true );
	// append extra packages to package list
	packages.insert(packages.end(), extraPackages.begin(),extraPackages.end());

	foreach( Path const& path, packages ) {
		RepositoryPtr tRepo = mDB.getRepositoryForLocalFile( path.string() );
		Package* tPackage = indexLocalPackage(path, tRepo.get());
		if ( tPackage ) {
			mDB.addPackage( tPackage );
			// the package might have been merged with a previous one ->
			// obtain the right package and mark as installed
			Package* dbPackage = dynamic_cast<Package*>(
					mDB.getIdentical( tPackage ) );
			mDB.setInstalled( dbPackage );
		}
	}

	mDB.cleanUp();

	mDB.buildTagIndex();
	statusMessage( "done" );
}

void MIRAPackage::touchCmakeLists( Path const& path ) const
{
	utime( (path / Path("CMakeLists.txt")).string().c_str(),NULL);
}

void MIRAPackage::statusMessage( std::string const& message )
{
	MIRA_LOG(NOTICE) << message;
}

int MIRAPackage::errorMessage( string const& message, string const& actionText,
			std::vector<std::string>* actions )
{
	cout << message << endl;
	if ( actionText.empty() )
		return -1;
	cout << actionText << endl;
	if ( actions ) {
		string answer;
		while ( find(actions->begin(),actions->end(),answer) == actions->end() ) {
			foreach( string const& action, *actions ) {
				cout << action << " ";
			}
			cout << "\n";
			cin >> answer;
		}
		for ( uint32 i=0; i<actions->size();++i) {
			if ( answer == (*actions)[i] )
				return i;
		}
	}
	return -1;
}

void MIRAPackage::statusProgress( std::string const& message, std::string const& title, int value, int maximum )
{
	MIRA_LOG(NOTICE) << message;
}

void MIRAPackage::progressReceiver( string const& file, int step, int total )
{
	statusProgress( "Indexing: " + file, "Reindexing ...", step, total );
}

void MIRAPackage::addPackageToCheckoutPlan( Package* package,
                                            bool checkDependencies /* = true */)
{
	if ( !mDB.isInstalled( package ) && !(mDB.getFlag( package ) & Database::INSTALL ) ) {
		try {
			if (checkDependencies) {
				Package* tDep = mDB.getInstallDependencies( *package, true );
				if ( tDep )
					resolveDependencies( tDep );
			}
			mDB.setFlag( package, Database::INSTALL );
		}
		catch ( Exception& ex ) {
			mDB.getPromptProvider()->showErrorMessage( ex.message() );
		}
	}
	else if ( mDB.getFlag( package ) & Database::UNINSTALL ) {
		mDB.setFlag( package, Database::NONE );
	}
}

void MIRAPackage::addPackageForRemoval( Package* package )
{
	if ( mDB.isInstalled( package ) ) {
		if ( mDB.getRepoFromUrl( package->mCurrentRepo ) )
			mDB.setFlag( package, Database::UNINSTALL );
		else
			MIRA_THROW(XLogical,"Cannot find associated repository for package "
					+ package->mName );
	}
	else if ( mDB.getFlag( package ) & Database::INSTALL ) {
		mDB.setFlag( package, Database::NONE );
	}
}

PackageGroup* MIRAPackage::checkForPrimaryPackageError()
{
	typedef Database::ActionPlan::value_type mapType;

	map<string,int> primaryCount;
	// cycle through install plan and check if more than one package will
	// be installed if the checkout plan is applied
	foreach( mapType const& item, mDB.mActionPlan ) {
		if ( item.second & Database::INSTALL ) {
			string name = item.first->mName;
			// get all packages with that name
			vector<Package*> packages =
				mDB.getRootPackage()->findPackages( name, true );
			int count = 0;
			// count if more than one primary package with that name will be
			// installed
			foreach( Package* package, packages ) {
				if ( package->isPrimary() && mDB.isInstalled( package, true ) ) {
					++count;
					if ( count > 1 )
						return package;
				}
			}
		}
	}
	return NULL;
}

void recursiveAdd( Package* package, vector<Package*>& installList,
		map<string,Package*>& remainingPackages, uint packageCount )
{
	typedef map<string,Package*> mapType;

	foreach( Package* dep, package->mDependencies ) {
		mapType::iterator it = remainingPackages.find( dep->mName );
		// we cannot add the package up to now, since it depends on one of the
		// package scheduled for install -> add dependency first of all
		if ( it != remainingPackages.end() ) {
			recursiveAdd( it->second, installList, remainingPackages, packageCount );
		}
	}

	installList.push_back( package );
	remainingPackages.erase( package->mName );

	if ( installList.size() > packageCount ) {
		MIRA_THROW(XLogical, "Detected at least two packages which are depending"
				" on each other! Cannot proceed!");
	}
}

vector<Package*> MIRAPackage::getInstallSequence( vector<Package*> const& packages )
{
	vector<Package*> installSequence;

	// create map of involved (primary) packages
	// extra packages are added at the end of the list (they should never be a dependency)
	map<string,Package*> remainingPackages;
	vector<Package*> extraPackages;
	foreach( Package* package, packages ) {
		assert( package );
		if ( package->isPrimary() )
			remainingPackages[ package->mName ] = package;
		else
			extraPackages.push_back( package );
	}

	int numOfPackages = remainingPackages.size();
	while( remainingPackages.size() > 0 ) {
		recursiveAdd( remainingPackages.begin()->second, installSequence,
				remainingPackages, numOfPackages );
	}

	installSequence.insert( installSequence.end(),
			extraPackages.begin(), extraPackages.end() );

	return installSequence;
}

vector<Package*> MIRAPackage::getPackageSequence( Database::Action action )
{
	typedef Database::ActionPlan::value_type mapType;
	// create install or uninstall sequence depending on the package dependencies

	// assume install action first of all
	vector<Package*> installSequence;

	// create list of involved packages
	vector<Package*> involvedPackages;
	foreach( mapType const& installFlag, mDB.mActionPlan ) {
		Package* package = installFlag.first;
		if ( installFlag.second & action ) {
			involvedPackages.push_back( package );
		}
	}

	// don't know how to handle other action than install and uninstall
	// simply return the list of items
	if ( action != Database::INSTALL && action != Database::UNINSTALL )
		return involvedPackages;

	installSequence = getInstallSequence( involvedPackages );

	// reverse sequence for uninstall
	if ( action == Database::UNINSTALL ) {
		vector<Package*> uninstallSequence;
		vector<Package*>::const_reverse_iterator it = installSequence.rbegin();
		for(; it != installSequence.rend(); it++ ) {
			uninstallSequence.push_back( *it );
		}
		return uninstallSequence;
	}
	return installSequence;
}

Path MIRAPackage::getPathForPackageUpdate(const Package* package) const
{
	if (package == NULL)
		MIRA_THROW(XLogical, "Package must not be NULL.");

	RepositoryPtr tRepo = mDB.getRepoFromUrl( package->mCurrentRepo );
	if (tRepo == NULL) {
		MIRA_LOG(NOTICE) << "Package does not belong to a repository.";
		return Path();
	}

	// If mLocalPath is empty, we can't do anything.
	if (package->mLocalPath.empty())
		return Path();

	// get MIRA_PATH for the installed package
	Path miraPath = PathProvider::getAssociatedMiraPath(package->mLocalPath);
	// the relative path is extracted from the mount dir files
	Path subPath = tRepo->getMountDirForPackage(package->mRepoSubPath);

	return miraPath / subPath.make_preferred();
}

void MIRAPackage::doIt()
{
	typedef Database::ActionPlan::value_type mapType;

	// enforce only one primary package for every package name
	PackageGroup* errorPackage = checkForPrimaryPackageError();
	if ( errorPackage ) {
		MIRA_THROW( XLogical, "You are going to have more than one primary "
				"package with name " + errorPackage->mName + "! Please modify your selection." );
	}

	// determine local path for packages if they do not already have one
	foreach( mapType const& installFlag, mDB.mActionPlan ) {
		Package* package = installFlag.first;
		if ( installFlag.second & Database::INSTALL ) {
			RepositoryPtr tRepo = mDB.getRepoFromUrl( package->mCurrentRepo );
			assert( tRepo );
			if ( package->mLocalPath.empty() ) {
				// select root path first of all
				// if package is a dependency and the depInstallPath is set
				// the package will be installed to this location. installRoot
				// will be used otherwise
				if ( mDB.getFlag( package ) & Database::DEPENDENCY )
					package->mLocalPath = mDepInstallPath.empty() ? mInstallRoot : mDepInstallPath;
				else
					package->mLocalPath = mInstallRoot;
				// the relative path is extracted from the mount dir files
				package->mLocalPath /= Path( Path(tRepo->getMountDirForPackage(
						package->mRepoSubPath ) ) ).make_preferred();
				MIRA_LOG(NOTICE) << "path for " << package->mName << " (determined) : " << package->mLocalPath;
			}
			else {
				MIRA_LOG(NOTICE) << "path for " << package->mName << " (known) : " << package->mLocalPath;
			}
		}
	}

	// confirm the checkout plan
	if ( confirmCheckoutPlan() ) {

		vector<Package*> uninstallSequence = getPackageSequence( Database::UNINSTALL );
		vector<Package*> installSequence = getPackageSequence( Database::INSTALL );

		// count install / uninstall actions (for progress bar)
		int maxSteps = installSequence.size() + uninstallSequence.size();

		set<Path> affectedMiraPaths;

		int step=0;
		// process deinstall actions first of all (important if another package
		// type should replace an existing one
		foreach( Package* package, uninstallSequence ) {
			statusProgress( "uninstalling " + package->mName, "Uninstalling ...", step, maxSteps);
				RepositoryPtr tRepo = mDB.getRepoFromUrl( package->mCurrentRepo );
				assert( tRepo );
				try {
					tRepo->uninstall( *package );
				}
				catch ( Exception& ex) {
					if (!mDB.getPromptProvider()->showYesNoErrorMessage("Cannot uninstall package \""
							+ package->mName + "\"!\n\n("+ex.message()+")\n\nDo you want to proceed with the remaining packages?")) {
						statusProgress( "aborted.", "Uninstalling ...", maxSteps, maxSteps);
						return;
					}
				}
				package->mLocalPath = Path();

				Path tMiraPath = PathProvider::getAssociatedMiraPath( package->mLocalPath );
				if ( !tMiraPath.empty() )
					affectedMiraPaths.insert( tMiraPath );

				++step;
		}

		foreach( Package* package, installSequence ) {
			statusProgress( "installing " + package->mName, "Installing ...", step, maxSteps);
				RepositoryPtr tRepo = mDB.getRepoFromUrl( package->mCurrentRepo );
				assert( tRepo );
				try {
					tRepo->install( *package, package->mLocalPath );
				}
				catch ( Exception& ex ) {
					if (!mDB.getPromptProvider()->showYesNoErrorMessage("Cannot install package \""
							+ package->mName + "\"!\n\n(" + ex.message() +")\n\nDo you want to proceed with the remaining packages?")) {
						statusProgress( "aborted.", "Installing ...", maxSteps, maxSteps);
						return;
					}
				}

				Path tMiraPath = PathProvider::getAssociatedMiraPath( package->mLocalPath );
				if ( !tMiraPath.empty() )
					affectedMiraPaths.insert( tMiraPath );

				++step;
		}

		statusProgress( "done.", "Installing ...", maxSteps, maxSteps);

		// touch CMakeLists of all affected mira paths
		foreach( Path const& tMiraPath, affectedMiraPaths ) {
			touchCmakeLists( tMiraPath );
		}

		checkForInstalled();

	}
}

vector<pair<Package*,Package*> > generateUpdatePlan( PackageGroup const* group, Database* db )
{
	vector<pair<Package*,Package*> > updatePlan;

	foreach( PackageGroup* subGroup, group->mSubPackages ) {
		// process groups only, since single packages cannot be updated
		foreach( PackageGroup* candidateGrp, subGroup->mSubPackages ) {
			Package* candidate = dynamic_cast<Package*>( candidateGrp );
			if ( candidate && db->isInstalled( candidate ) ) {
				Package* newVersion = db->newVersionAvailable( candidate );
				if ( newVersion ) {
					updatePlan.push_back( make_pair( candidate, newVersion ) );
				}
			}
		}
	}

	return updatePlan;
}

void MIRAPackage::update()
{
	vector<pair<Package*,Package*> > updatePlan = generateUpdatePlan( mDB.getRootPackage(), &mDB );
	if ( confirmUpdatePlan( updatePlan ) ) {
		// transfer the changes made in the dialog to the database
		foreach( auto& update, updatePlan ) {
			// Use the old MIRA_PATH for the new version.
			update.second->mLocalPath = getPathForPackageUpdate(update.first);

			addPackageForRemoval( update.first );
			// "False" for "Don't check dependencies" here!
			addPackageToCheckoutPlan( update.second, false );
		}
	}
}

///////////////////////////////////////////////////////////////////////////////

void MIRAPackage::loadContent(const XMLDom& content)
{
	mDB.load(content);
}

void MIRAPackage::saveContent(XMLDom& content)
{
	mDB.store(content);
}

void MIRAPackage::resolveDependencies( Package* depTree )
{
	Database depDatabase;
	depDatabase.repos = mDB.repos;
	depDatabase.recursiveSelectStdSource( depTree );
	if ( depTree->mSubPackages.size() > 0 && confirmDependencies( depTree, &depDatabase ) ) {
		// transfer the changes made in the dialog to the database
		typedef std::map<Package*,Database::Action>::value_type MapType;
		foreach( MapType const& flag, depDatabase.mActionPlan ) {
			PackageGroup* identicalGroup = mDB.getIdentical( flag.first );
			Package* identical = dynamic_cast<Package*>( identicalGroup );
			assert( identical );
			mDB.setFlag( identical, flag.second | Database::DEPENDENCY );
			identical->mCurrentRepo = flag.first->mCurrentRepo;
		}
	}
	depDatabase.repos.clear();

	delete depTree;
}

bool MIRAPackage::confirmCheckoutPlan()
{
	return true;
}

bool MIRAPackage::confirmExportPackages( Database& ioDB, map<Path,string>& oPathMap )
{
	return true;
}

bool MIRAPackage::confirmUpdatePlan( vector<pair<Package*,Package*> >& updatePlan )
{
	return true;
}

bool MIRAPackage::confirmDependencies( PackageGroup* rootPackage, Database* database )
{
	return true;
}

///////////////////////////////////////////////////////////////////////////////

}
