/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file MIRAPackageGUI.h
 *    $short description$.
 *
 * @author Ronny Stricker, Christian Martin
 * @date   2011/07/29
 */

#ifndef MIRAPACKAGEGUI_H_
#define MIRAPACKAGEGUI_H_

#include <ui_MIRAPackageGUI.h>

#include <QMainWindow>
#include <QProgressDialog>

#include <utils/PathFinder.h>

#include <core/MIRAPackage.h>
#include <gui/PackageListWidget.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * @brief Add GUI to the core functionality of the MIRAPackage class.
 */
class MIRAPackageGUI :
	public QMainWindow, public MIRAPackage,
	protected Ui::MIRAPackageGUI
{
	Q_OBJECT

public:
	/**
	 * @brief Setup GUI.
	 */
	MIRAPackageGUI();

	virtual ~MIRAPackageGUI();

	/**
	 * @brief Display dialog to select active MIRA Path.
	 * See MIRAPackage::resolveMIRAPath() for further details.
	 */
	virtual void resolveMIRAPath( bool forceSelection = false );

//signals:
//	void dependenciesResolved( Package* dep );

public:
	/**
	 * @brief Reindex all repositories.
	 * See MIRAPackage::reindexWebRepository() for further details.
	 * In addition the function prepares the tag and package
	 * model for updates.
	 */
	virtual void reindexWebRepository(RepositoryPtr repo, bool thorough);

	/**
	 * @brief Check for packages installed on the local system.
	 * See MIRAPackage::resolveMIRAPath() for further details.
	 * In addition the function prepares the tag and package
	 * model for updates.
	 */
	virtual void checkForInstalled();

	/**
	 * @brief Display status message in the main window status bar.
	 * See MIRAPackage::statusMessage() for further details.
	 */
	virtual void statusMessage( std::string const& message );

	/**
	 * @brief Display error message in a popup dialog.
	 * The dialog will have as many buttons as the actions vector element count.
	 * The functions returns the id of the selected action (element position) or
	 * -1 if no actions where provided.
	 * The maximum number of possible actions is 3
	 */
	virtual int errorMessage( std::string const& message,
	                          std::string const& actionText = "",
	                          std::vector<std::string>* actions = NULL );

	virtual void statusProgress( std::string const& message,
	                             std::string const& title,
	                             int value, int maximum );

protected slots:

	void exportDB();

	void importDB();

	/**
	 * @brief Reindex web repositories and look for local packages.
	 * Use repository index file if possible.
	 */
	void reindexDatabase();

	/**
	 * @brief Reindex web repositories and look for local packages.
	 * Do never use repository index file.
	 */
	void deepReindexDatabase();

	/**
	 * @brief Slot for storing the database.
	 * Calls function save().
	 */
	void storeDatabase();

	/**
	 * @brief Show dialog with the list of repositories.
	 */
	void showRepositoryList();

	/**
	 * @brief Add package for installation.
	 * Calls addPackageToCheckoutPlan() and informs the package
	 * view of the update
	 */
	void addPackageForInstall( Package* package );

	/**
	 * @brief Add package for uninstall.
	 * Calls addPackageForRemoval() and informs the package view
	 * of the update.
	 */
	void addPackageForUninstall( Package* package );

	/**
	 * @brief Simple slot which calls resolveMIRAPath( true )
	 */
	void resolveMIRAPathSlot();

	/**
	 * @brief Wrapper for the MIRAPackage::doIt() function.
	 * Catches exeptions of doIt() and show MessageBox if necessary.
	 */
	void doIt();

	/**
	 * @brief Try to select all packages for update.
	 * Asks for reindex of repositories first of all.
	 */
	void update();

protected:

	/**
	 * @brief Overwrite close Event to inform user about changes not applied yet.
	 */
	virtual void closeEvent(QCloseEvent *event);

	/**
	 * @brief Show PathSelectDialog for user interaction.
	 */
	virtual std::string selectMIRAPath();

	/**
	 * @brief Show InstallOverviewDialog and let the user decide.
	 */
	virtual bool confirmCheckoutPlan();

	/**
	 * @brief Show ExportDialog and let the user decide.
	 * The packages which are not selected will be delete from the database.
	 * @param ioDB Database to work with
	 * @return true if the user has accepted the dialog.
	 */
	virtual bool confirmExportPackages( Database& ioDB, std::map<Path,std::string>& oPathMap );

	/**
	 * @brief Show UpdateDialog and let the user decide.
	 * @param updatePlan updatePlan which should be reviewed
	 * @return true if update should be done
	 */
	virtual bool confirmUpdatePlan( std::vector<std::pair<Package*,Package*> >& updatePlan );

	/**
	 * @brief Show DependencyDialog and let the user decide.
	 * @see MIRAPackage::confirmDependencies()
	 */
	virtual bool confirmDependencies( PackageGroup* rootPackage, Database* database );

protected:
	virtual void loadContent(const XMLDom& content);
	virtual void saveContent(XMLDom& content);

protected:
	PackageListWidget* mListWidget; ///< the main GUI Widget

	QProgressDialog* mProgressDialog;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif /* MIRAPACKAGEGUI_H_ */
