/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RangeScanVisualization.C
 *    2D visualization plugin for miracenter to visualize RangeScan data.
 *
 * @author Christian Reuther
 * @date   2014/08/27
 */

#include <visualization/Visualization2DBasic.h>

#include <serialization/Serialization.h>
#include <serialization/SetterNotify.h>
#include <serialization/adapters/Qt/QColor>

#include <transform/Pose.h>
#include <robot/RangeScan.h>

#include <QGraphicsScene>
#include <QGraphicsPathItem>
#include <QPolygon>
#include <QPoint>

namespace mira { namespace robot {

//////////////////////////////////////////////////////////////////////////////

/**
 * 2D visualization plugin for miracenter to visualize RangeScan data.
 */
class RangeScanVisualization2D :  public Visualization2DBasic<robot::RangeScan>
{
	MIRA_META_OBJECT(mira::robot::RangeScanVisualization2D,
			("Name", "Range Scan")
			("Description", "Visualization of 2D range scans")
			("Category", "Robot Datatypes"))

	typedef Visualization2DBasic<robot::RangeScan> Base;

public:
	enum RenderMode {
		FAN = 0,
		POINTS
	};

	enum ColorMode {
		DEFAULT_COLOR = 0,
		VALID_COLOR
	};

	enum RangeScanValidity {
		VALID = 0,
		ABOVEMAX = 1,
		BELOWMIN = 2,
		MASKED = 3,
		INVALID = 4,
		INVALIDUSER = 5,
		RANGESCANVALIDITY_MAX = 6
	};

public:
	RangeScanVisualization2D() : Base("Range"), mRenderMode(FAN), mColorMode(VALID_COLOR) {
		mColors[VALID] = QColor(51, 128, 178, 128);
		mColors[ABOVEMAX] = QColor(255, 255, 0, 128);
		mColors[BELOWMIN] = QColor(0, 255, 255, 128);
		mColors[MASKED] = QColor(255, 255, 255, 128);
		mColors[INVALID] = QColor(0, 0, 0, 128);
		mColors[INVALIDUSER] = QColor(255, 0, 0, 128);

		mItem = NULL;
		mPathItems[VALID] = NULL;
		mPathItems[ABOVEMAX] = NULL;
		mPathItems[BELOWMIN] = NULL;
		mPathItems[MASKED] = NULL;
		mPathItems[INVALID] = NULL;
		mPathItems[INVALIDUSER] = NULL;

		mMaxRange = 0.0f;
		mPointScale = 1.0f;
	}

	virtual ~RangeScanVisualization2D() {
		auto site = this->getSite();
		// For some reason it is possible that the view gets destructed and re-constructed
		// when miracenter starts, so we need to make sure that we can access the items
		if(!mItem || !site || !site->getSceneManager())
			return;

		site->getSceneManager()->removeItem(mItem);
		for(std::size_t i = 0; i < RANGESCANVALIDITY_MAX; ++i) {
			mItem->removeFromGroup(mPathItems[i]);
			delete mPathItems[i];
		}
		// Do not delete mItem as the base class destructor does this.
	}

public:
	template <typename Reflector> void reflect(Reflector& r) {
		Base::reflect(r);

		r.property("RenderMode", mRenderMode, setterNotify(mRenderMode, &RangeScanVisualization2D::drawLast, this),
				"Specify the render mode", FAN, mira::PropertyHints::enumeration("FAN;POINTS"));
		r.property("ColorMode", mColorMode, setterNotify(mColorMode, &RangeScanVisualization2D::drawLast, this), "Specify the color mode",
				VALID_COLOR, mira::PropertyHints::enumeration("DEFAULT;VALID"));

		r.property("Color", mColors[VALID], setter<QColor>(&RangeScanVisualization2D::setColor, this), "The color", QColor(51, 128, 178, 128));
		r.property("ColorInvalid", mColors[INVALID], setterNotify(mColors[INVALID], &RangeScanVisualization2D::drawLast, this),
			"The color for invalid values.", QColor(0, 0, 0, 128));
		r.property("ColorAboveMax", mColors[ABOVEMAX], setterNotify(mColors[ABOVEMAX], &RangeScanVisualization2D::drawLast, this),
			"The color for values above maximum.", QColor(255, 255, 0, 128));
		r.property("ColorBelowMin", mColors[BELOWMIN], setterNotify(mColors[BELOWMIN], &RangeScanVisualization2D::drawLast, this),
			"The color for values below minimum.", QColor(0, 255, 255, 128));
		r.property("ColorMasked", mColors[MASKED], setterNotify(mColors[MASKED], &RangeScanVisualization2D::drawLast, this),
			"The color for masked values.", QColor(255, 255, 255, 128));
		r.property("Max Range", mMaxRange, setterNotify(mMaxRange, &RangeScanVisualization2D::drawLast, this),
			"The max. range that is visualized. The transparency is increased up to this range to fade the scan out (disabled if <=0)", 0.0f);
		r.property("Scale", mPointScale, setterNotify(mPointScale, &RangeScanVisualization2D::drawLast, this), "Scaling factor for POINTS drawing mode", 1.0f);
	}

public:
	virtual QGraphicsItem* setupScene(QGraphicsScene* mgr) {
		mItem = new QGraphicsItemGroup();
		for(std::size_t i = 0; i < RANGESCANVALIDITY_MAX; ++i) {
			mPathItems[i] = new QGraphicsPathItem();
			mItem->addToGroup(mPathItems[i]);
		}

		mgr->addItem(mItem);
		return mItem;
	}

	void setColor(const QColor& color) {
		mColors[VALID] = color;
		drawLast();
	}

protected:
	void dataChanged(ChannelRead<robot::RangeScan> scan);
	void drawLast();

protected:
	QPair<QPointF, QPointF> computeRangeScan(const robot::RangeScan& scan, Pose3 scanPose, size_t i, float maxRange = 0.0f);
	QPointF computeRangeScanCenter(const robot::RangeScan& scan, Pose3 scanPose, size_t i, float maxRange = 0.0f);
	QGraphicsPathItem* getItemForScanValidity(const robot::RangeScan& scan, size_t i);
	void updateItemColor(ColorMode color, RenderMode mode);

	void resetItems();
	void simplifyPaths();
	void drawPoints(const robot::RangeScan& scan, Pose3 scanPose);
	void drawFan(const robot::RangeScan& scan, Pose3 scanPose);
	QPolygonF drawFanSubset(const robot::RangeScan& scan, Pose3 scanPose, RangeScan::RangeCode values);

	RangeScan::RangeCode convertEnumFromHelper(RangeScanValidity value);
	RangeScanValidity convertEnumToHelper(RangeScan::RangeCode value);

protected:
	RenderMode mRenderMode;
	ColorMode mColorMode;
	float mMaxRange;
	float mPointScale;

	QColor mColors[RANGESCANVALIDITY_MAX];

	QGraphicsItemGroup* mItem;
	QGraphicsPathItem* mPathItems[RANGESCANVALIDITY_MAX];
};

////////////////////////////////////////////////////////////////////////////////

void RangeScanVisualization2D::dataChanged(ChannelRead<robot::RangeScan> scan) {
	if(!mItem)
		return;

	Pose3 scanPose = getTransform<Pose3>(scan->frameID, this->getSite()->getCameraFrame(), scan->timestamp);

	updateItemColor(mColorMode, mRenderMode);

	if(mRenderMode == FAN)
		drawFan(scan, scanPose);
	else if(mRenderMode == POINTS)
		drawPoints(scan, scanPose);

	simplifyPaths();
}

void RangeScanVisualization2D::drawLast() {
	try {
		if(!mDataChannel.isValid())
			return;
		dataChanged(mDataChannel.getChannel().read());
	} catch(Exception&) {}
}

////////////////////////////////////////////////////////////////////////////////

QPair<QPointF, QPointF> RangeScanVisualization2D::computeRangeScan(const robot::RangeScan& scan, Pose3 scanPose, size_t i, float maxRange) {
	float angle;
	float dangle;
	boost::tie(angle, dangle) = scan.getAnglesForOrientation(scanPose);

	float alpha0 = angle + i * dangle - scan.coneAngle.value() * 0.5f;
	float alpha = alpha0 + scan.coneAngle;
	float r = scan.range[i];
	if(mMaxRange > 0.0f && r > mMaxRange)
		r = mMaxRange;

	QPointF p1(r * cosf(alpha0), r * sinf(alpha0));
	QPointF p2(r * cosf(alpha), r * sinf(alpha));
	return QPair<QPointF, QPointF>(p1, p2);
}

QPointF RangeScanVisualization2D::computeRangeScanCenter(const robot::RangeScan& scan, Pose3 scanPose, size_t i, float maxRange) {
	float angle;
	float dangle;
	boost::tie(angle, dangle) = scan.getAnglesForOrientation(scanPose);

	const float alpha = angle + i * dangle;
	float r = scan.range[i];
	if(mMaxRange > 0.0f && r > mMaxRange)
		r = mMaxRange;

	return QPointF(r * cosf(alpha), r * sinf(alpha));
}

QGraphicsPathItem* RangeScanVisualization2D::getItemForScanValidity(const robot::RangeScan& scan, size_t i) {
	return mPathItems[convertEnumToHelper(static_cast<RangeScan::RangeCode>(scan.valid[i]))];
}

void RangeScanVisualization2D::updateItemColor(ColorMode color, RenderMode mode) {
	for(std::size_t i = 0; i < RANGESCANVALIDITY_MAX; ++i) {
		if(color == ColorMode::VALID_COLOR) {
			mPathItems[i]->setPen(Qt::NoPen);
			mPathItems[i]->setBrush(QBrush(mColors[i]));
		} else {  // Default, e.g. with color mode DEFAULT_COLOR
			mPathItems[i]->setPen(Qt::NoPen);
			mPathItems[i]->setBrush(QBrush(mColors[VALID]));
		}
	}
}

////////////////////////////////////////////////////////////////////////////////

void RangeScanVisualization2D::resetItems() {
	QPainterPath path;
	path.setFillRule(Qt::WindingFill);

	for(std::size_t i = 0; i < RangeScanValidity::RANGESCANVALIDITY_MAX; ++i)
		mPathItems[i]->setPath(path);
}

void RangeScanVisualization2D::simplifyPaths() {
	for(std::size_t i = 0; i < RangeScanValidity::RANGESCANVALIDITY_MAX; ++i)
		mPathItems[i]->setPath(mPathItems[i]->path().simplified());
}

////////////////////////////////////////////////////////////////////////////////

void RangeScanVisualization2D::drawPoints(const robot::RangeScan& scan, Pose3 scanPose) {
	resetItems();

	const auto size = scan.range.size();
	for(size_t i = 0;i < size; ++i) {
		auto item = getItemForScanValidity(scan, i);

		const auto p = computeRangeScan(scan, scanPose, i);
		const QPointF center = QPointF(p.first + p.second) * 0.5;
		const QPointF delta = QPointF(p.first - p.second);

		// Add to path of the item
		auto path = item->path();
		path.addEllipse(center, mPointScale * delta.x(), mPointScale * delta.y());
		item->setPath(path);
	}
}

void RangeScanVisualization2D::drawFan(const RangeScan& scan, Pose3 scanPose) {
	resetItems();

	for(std::size_t i = 0; i < RangeScanValidity::RANGESCANVALIDITY_MAX; ++i) {
		auto poly = drawFanSubset(scan, scanPose, convertEnumFromHelper(static_cast<RangeScanValidity>(i)));
		auto path = mPathItems[i]->path();
		path.addPolygon(poly);
		mPathItems[i]->setPath(path);
	}
}

QPolygonF RangeScanVisualization2D::drawFanSubset(const robot::RangeScan& scan, Pose3 scanPose, RangeScan::RangeCode values) {
	const auto size = scan.range.size();
	QPolygonF poly;
	poly.append(QPointF(0.0, 0.0));

	for(std::size_t i = 0; i < size; ++i) {
		if(scan.valid[i] != values) {
			poly.append(QPointF(0.0, 0.0));
			continue;
		}

		const auto endpoint = computeRangeScan(scan, scanPose, i, mMaxRange);
		poly.append(endpoint.first);
		poly.append(endpoint.second);
	}

	return poly;
}

////////////////////////////////////////////////////////////////////////////////

RangeScan::RangeCode RangeScanVisualization2D::convertEnumFromHelper(RangeScanValidity value) {
	if(value == RangeScanValidity::VALID)
		return RangeScan::RangeCode::Valid;
	else if(value == RangeScanValidity::ABOVEMAX)
		return RangeScan::RangeCode::AboveMaximum;
	else if(value == RangeScanValidity::BELOWMIN)
		return RangeScan::RangeCode::BelowMinimum;
	else if(value == RangeScanValidity::MASKED)
		return RangeScan::RangeCode::Masked;
	else if(value == RangeScanValidity::INVALIDUSER)
		return RangeScan::RangeCode::InvalidUser;
	return RangeScan::RangeCode::Invalid;
}

RangeScanVisualization2D::RangeScanValidity RangeScanVisualization2D::convertEnumToHelper(RangeScan::RangeCode value) {
	if(value == RangeScan::RangeCode::Valid)
		return RangeScanValidity::VALID;
	else if(value == RangeScan::RangeCode::AboveMaximum)
		return RangeScanValidity::ABOVEMAX;
	else if(value == RangeScan::RangeCode::BelowMinimum)
		return RangeScanValidity::BELOWMIN;
	else if(value == RangeScan::RangeCode::Masked)
		return RangeScanValidity::MASKED;
	else if(value == RangeScan::RangeCode::InvalidUser)
		return RangeScanValidity::INVALIDUSER;
	return RangeScanValidity::INVALID;
}

////////////////////////////////////////////////////////////////////////////////

}}

MIRA_CLASS_SERIALIZATION(mira::robot::RangeScanVisualization2D, mira::Visualization2D);
