/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file MarkerVisualization.C
 *    Visualization for markers.
 *
 * @author Tim Langner
 * @date   2013/03/02
 */


#include <transform/Pose.h>
#include <serialization/Serialization.h>
#include <visualization/3d/ArrowObject.h>
#include <visualization/3d/LineStripObject.h>
#include <visualization/3d/MeshObject.h>
#include <visualization/3d/PolygonObject.h>
#include <visualization/3d/TextObject.h>
#include <visualization/Visualization3DBasic.h>
#include <widgets/OgreUtils.h>
#include <model/Marker.h>

namespace mira { namespace  model {

///////////////////////////////////////////////////////////////////////////////

class MarkerVisualization3D : public Visualization3DBasic<mira::model::Marker>
{
	MIRA_META_OBJECT(MarkerVisualization3D,
			("Name", "Marker")
			("Description", "3D Visualization of markers")
			("Category", "Models"))

public:

	struct ObjectRepresentation
	{
		ObjectRepresentation() : node(NULL) {}
		void clear(Ogre::SceneManager* mgr)
		{
			objects.clear();
			if (node)
				mgr->destroySceneNode(node);
		}
		std::string frame;
		Ogre::SceneNode* node;
		std::vector<boost::shared_ptr<VisualizationObject>> objects;
	};

	MarkerVisualization3D() :
		Visualization3DBasic<mira::model::Marker>("Markers")
	{}

	~MarkerVisualization3D()
	{
		if (getSite() != NULL && getSite()->getSceneManager() != NULL)
			clearObjects();
	}

	void setupScene(Ogre::SceneManager* mgr, Ogre::SceneNode* node) {};

	virtual void setEnabled(bool enabled);

	virtual void update(Duration dt);

	void dataChanged(ChannelRead<mira::model::Marker> boxes);

protected:
	typedef std::map<uint32, ObjectRepresentation> IDObjects;
	typedef std::map<std::string, IDObjects> NSObjects;

	void clearObjects();
	void clearObjects(IDObjects& ids, uint32 id);

protected:

	NSObjects mObjects;
};

///////////////////////////////////////////////////////////////////////////////

boost::shared_ptr<VisualizationObject> createRepresentation(mira::model::MarkerBasePtr marker,
                                                            Ogre::SceneNode* node,
                                                            Ogre::SceneManager* sceneManager,
                                                            bool visible)
{
	Ogre::Vector3 scale(marker->scale.x(), marker->scale.y(), marker->scale.z());
	Ogre::Vector3 offsetPosition(marker->position.x(), marker->position.y(), marker->position.z());
	Ogre::Quaternion offsetOrientation(marker->position.r.w(),
	                                   marker->position.r.x(),
	                                   marker->position.r.y(),
	                                   marker->position.r.z());

	boost::shared_ptr<VisualizationObject> object;
	std::string identifier = marker->getClass().getIdentifier();
	if (identifier == "mira::model::BoxMarker")
	{
		object.reset(new MeshObject("Box.mesh", sceneManager, node));
		offsetPosition.y -= scale.y/2.0f;
	}
	else if (identifier == "mira::model::ConeMarker")
	{
		std::swap(scale.y, scale.z);
		// cone has 1 meter radius by default
		// as the user specifies the diameter we need to scale it down
		scale.z *= 0.5f;
		scale.x *= 0.5f;
		object.reset(new MeshObject("Cone.mesh", sceneManager, node));
		Ogre::Quaternion rotX;
		rotX.FromAngleAxis(Ogre::Degree(90), Ogre::Vector3::UNIT_X);
		offsetOrientation = offsetOrientation * rotX;
	}
	else if (identifier == "mira::model::CylinderMarker")
	{
		std::swap(scale.y, scale.z);
		// cylinder has 1 meter radius by default
		// as the user specifies the diameter we need to scale it down
		scale.z *= 0.5f;
		scale.x *= 0.5f;
		object.reset(new MeshObject("CylinderCentered.mesh", sceneManager, node));
		Ogre::Quaternion rotX;
		rotX.FromAngleAxis(Ogre::Degree(90), Ogre::Vector3::UNIT_X);
		offsetOrientation = offsetOrientation * rotX;
	}
	else if (identifier == "mira::model::SphereMarker")
	{
		// sphere has 1 meter radius by default
		// as the user specifies the diameter we need to scale it down
		scale *= 0.5f;
		object.reset(new MeshObject("Sphere.mesh", sceneManager, node));
	}
	else if (identifier == "mira::model::TextMarker")
	{
		mira::model::TextMarkerPtr tm = boost::dynamic_pointer_cast<mira::model::TextMarker>(marker);
		object.reset(new TextObject(tm->text.c_str(), sceneManager, node));
	}
	else if (identifier == "mira::model::ArrowMarker")
	{
		std::swap(scale.x, scale.y);
		object.reset(new ArrowObject(sceneManager, node));
		Ogre::Quaternion rotX;
		rotX.FromAngleAxis(Ogre::Degree(-90), Ogre::Vector3::UNIT_Z);
		offsetOrientation = offsetOrientation * rotX;
	}
	else if (identifier == "mira::model::PointedArrowMarker")
	{
		// set scale to 1 and ignore user specified value as we compute length
		// from distance between p1 and p2
		scale.y = 1.0f;
		mira::model::PointedArrowMarkerPtr pam = boost::dynamic_pointer_cast<mira::model::PointedArrowMarker>(marker);
		ArrowObject* arrow = new ArrowObject(sceneManager, node);
		object.reset(arrow);

		Ogre::Vector3 point1(pam->start.x(), pam->start.y(), pam->start.z());
		Ogre::Vector3 point2(pam->end.x(), pam->end.y(), pam->end.z());

		Ogre::Vector3 direction = point2 - point1;
		float distance = direction.length();
		arrow->setLength(distance);
		direction.normalise();

		Ogre::Quaternion orient = Ogre::Vector3::UNIT_Y.getRotationTo( direction );
		offsetPosition = point1;
		offsetOrientation = orient;
	}
	else if (identifier == "mira::model::PolygonMarker")
	{
		mira::model::PolygonMarkerPtr pm = boost::dynamic_pointer_cast<mira::model::PolygonMarker>(marker);
		PolygonObject<float>* pobject = new PolygonObject<float>(sceneManager, node);
		pobject->setPolygon(pm->polygon);
		object.reset(pobject);
	}
	else if (identifier == "mira::model::LineListMarker")
	{
		mira::model::LineListMarkerPtr llm = boost::dynamic_pointer_cast<mira::model::LineListMarker>(marker);
		LineStripObject* lobject = new LineStripObject(sceneManager, node);
		lobject->setLineWidth(llm->lineWidth);
		foreach(const Line3f& line, llm->lines)
		{
			lobject->begin();
			lobject->point(Ogre::Vector3(line.first.x(), line.first.y(), line.first.z()));
			lobject->point(Ogre::Vector3(line.second.x(), line.second.y(), line.second.z()));
			lobject->end();
		}
		// ignore scale values
		scale.x = 1.0f;
		scale.y = 1.0f;
		scale.z = 1.0f;
		object.reset(lobject);
	}
	else if (identifier == "mira::model::LineStripMarker")
	{
		mira::model::LineStripMarkerPtr lsm = boost::dynamic_pointer_cast<mira::model::LineStripMarker>(marker);
		LineStripObject* lobject = new LineStripObject(sceneManager, node);
		lobject->setLineWidth(lsm->lineWidth);
		lobject->begin();
		foreach(const Point3f& point, lsm->points)
			lobject->point(Ogre::Vector3(point.x(), point.y(), point.z()));
		lobject->end();
		// ignore scale values
		scale.x = 1.0f;
		scale.y = 1.0f;
		scale.z = 1.0f;
		object.reset(lobject);
	}
	else
		MIRA_THROW(XInvalidConfig, "'" << identifier << "' is not supported by MarkerVisualization3D")
	object->setScale(scale);
	object->setPosition(offsetPosition);
	object->setOrientation(offsetOrientation);
	object->setColor(marker->color);
	object->setVisible(visible);
	return object;
}

void MarkerVisualization3D::setEnabled(bool enabled)
{
	Visualization3D::setEnabled(enabled);
	foreach(auto ns, mObjects)
		foreach(auto id, ns.second)
			foreach(auto o, id.second.objects)
				o->setVisible(enabled);
}

void MarkerVisualization3D::update(Duration dt)
{
	foreach(auto ns, mObjects)
		foreach(auto id, ns.second)
		{
			Pose3 d = getAuthority().getTransform<RigidTransform3f>(id.second.frame,
			                                                        getSite()->getCameraFrame(),
			                                                        Time::now());
			OgreUtils::setTransform(id.second.node, d);
		}
}

void MarkerVisualization3D::dataChanged(ChannelRead<mira::model::Marker> markers)
{
	IDObjects& ids = mObjects[markers->ns];
	clearObjects(ids, markers->value().id);
	if (markers->value().command == Marker::REMOVE)
		return;

	Ogre::SceneManager* sceneManager = getSite()->getSceneManager();

	ObjectRepresentation r;
	r.node = mNode->createChildSceneNode();
	r.frame = markers->frameID;
	foreach(MarkerBasePtr m, markers->value().objects)
	{
		auto object = createRepresentation(m, r.node, sceneManager,isEnabled());
		r.objects.push_back(object);
	}
	ids[markers->value().id] = r;
}

void MarkerVisualization3D::clearObjects(IDObjects& ids, uint32 id)
{
	Ogre::SceneManager* sceneManager = getSite()->getSceneManager();
	if (ids.count(id) == 0)
		return;
	ids[id].clear(sceneManager);
	ids.erase(id);
}

void MarkerVisualization3D::clearObjects()
{
	Ogre::SceneManager* sceneManager = getSite()->getSceneManager();
	foreach(auto ns, mObjects)
		foreach(auto id, ns.second)
			id.second.clear(sceneManager);
	mObjects.clear();
}

//////////////////////////////////////////////////////////////////////////////

}}

MIRA_CLASS_SERIALIZATION(mira::model::MarkerVisualization3D, mira::Visualization3D);
