/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Marker.h
 *    Markers for 3D view.
 *
 * @author Tim
 * @date   2013/03/02
 */

#ifndef _MIRA_MARKER_H_
#define _MIRA_MARKER_H_

#include <image/Color.h>
#include <transform/Pose.h>
#include <geometry/Line.h>
#include <geometry/Point.h>
#include <geometry/Size.h>
#include <geometry/Polygon.h>

#include <model/RigidModelExports.h>

namespace mira { namespace model {

///////////////////////////////////////////////////////////////////////////////

/**
 * Base class for all marker types
 */
class MIRA_RIGID_MODEL_EXPORT MarkerBase : public Object
{
	MIRA_OBJECT(MarkerBase)
public:

	EIGEN_MAKE_ALIGNED_OPERATOR_NEW

	MarkerBase() :
		color(Color::White),
		scale(1.0f, 1.0f, 1.0f) {}

	MarkerBase(const Pose3& pos, const Color::RGBA& col=Color::White, const Size3f& s=Size3f(1.0f,1.0f,1.0f)) :
		position(pos),
		color(col),
		scale(s) {}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("Position", position, "");
		r.member("Color", color, "");
		r.member("Scale", scale, "");
	}

	Pose3 position;
	Color::RGBA color;
	Size3f scale;
};

typedef boost::shared_ptr<MarkerBase> MarkerBasePtr;

///////////////////////////////////////////////////////////////////////////////

/**
 * Box marker. By default its size is 1x1x1 meter.
 * Origin is in the center of the box
 */
class MIRA_RIGID_MODEL_EXPORT BoxMarker : public MarkerBase
{
	MIRA_OBJECT(BoxMarker)
public:
	BoxMarker() : MarkerBase() {}
	BoxMarker(const Pose3& position, const Color::RGBA& color=Color::White, const Size3f& scale=Size3f(1.0f,1.0f,1.0f)) :
		MarkerBase(position, color, scale) {}
};

typedef boost::shared_ptr<BoxMarker> BoxMarkerPtr;

///////////////////////////////////////////////////////////////////////////////

/**
 * Sphere marker. By default it has a diameter of 1 meter.
 * Origin is in the center of the box
 */
class MIRA_RIGID_MODEL_EXPORT SphereMarker : public MarkerBase
{
	MIRA_OBJECT(SphereMarker)
public:
	SphereMarker() : MarkerBase() {}
	SphereMarker(const Pose3& position, const Color::RGBA& color=Color::White, const Size3f& scale=Size3f(1.0f,1.0f,1.0f)) :
		MarkerBase(position, color, scale) {}
};

typedef boost::shared_ptr<SphereMarker> SphereMarkerPtr;

///////////////////////////////////////////////////////////////////////////////

/**
 * Cylinder marker. By default it has height 1 meter and a diameter of 1 meter.
 * Origin is in the center of the cylinder
 */
class MIRA_RIGID_MODEL_EXPORT CylinderMarker : public MarkerBase
{
	MIRA_OBJECT(CylinderMarker)
public:
	CylinderMarker() : MarkerBase() {}
	CylinderMarker(const Pose3& position, const Color::RGBA& color=Color::White, const Size3f& scale=Size3f(1.0f,1.0f,1.0f)) :
		MarkerBase(position, color, scale) {}
};

typedef boost::shared_ptr<CylinderMarker> CylinderMarkerPtr;

///////////////////////////////////////////////////////////////////////////////

/**
 * Cone marker. By default it has height 1 meter and a diameter of 1 meter.
 * Origin is in the center of the base plane
 */
class MIRA_RIGID_MODEL_EXPORT ConeMarker : public MarkerBase
{
	MIRA_OBJECT(ConeMarker)
public:
	ConeMarker() : MarkerBase() {}
	ConeMarker(const Pose3& position, const Color::RGBA& color=Color::White, const Size3f& scale=Size3f(1.0f,1.0f,1.0f)) :
		MarkerBase(position, color, scale) {}
};

typedef boost::shared_ptr<ConeMarker> ConeMarkerPtr;

///////////////////////////////////////////////////////////////////////////////

/**
 * Text marker. Origin is in the upper left of the text.
 */
class MIRA_RIGID_MODEL_EXPORT TextMarker : public MarkerBase
{
	MIRA_OBJECT(TextMarker)
public:
	TextMarker() : MarkerBase() {}
	TextMarker(const std::string& str, const Pose3& position, const Color::RGBA& color=Color::White, const Size3f& scale=Size3f(1.0f,1.0f,1.0f)) :
		MarkerBase(position, color, scale), text(str) {}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MarkerBase::reflect(r);
		r.member("Text", text, "");
	}
	std::string text;
};

typedef boost::shared_ptr<TextMarker> TextMarkerPtr;

///////////////////////////////////////////////////////////////////////////////

/**
 * Arrow marker. By default it has length 1 meter and points along the x-axis.
 */
class MIRA_RIGID_MODEL_EXPORT ArrowMarker : public MarkerBase
{
	MIRA_OBJECT(ArrowMarker)
};

typedef boost::shared_ptr<ArrowMarker> ArrowMarkerPtr;

/**
 * Arrow marker that points from start to end point.
 * Position and scale.x parameters of base class have no effect.
 */
class MIRA_RIGID_MODEL_EXPORT PointedArrowMarker : public MarkerBase
{
	MIRA_OBJECT(PointedArrowMarker)
public:

	EIGEN_MAKE_ALIGNED_OPERATOR_NEW

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MarkerBase::reflect(r);
		r.member("Start", start, "");
		r.member("End", end, "");
	}

	Point3f start;
	Point3f end;
};

typedef boost::shared_ptr<PointedArrowMarker> PointedArrowMarkerPtr;

///////////////////////////////////////////////////////////////////////////////

/**
 * Polygon marker.
 */
class MIRA_RIGID_MODEL_EXPORT PolygonMarker : public MarkerBase
{
	MIRA_OBJECT(PolygonMarker)
public:
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MarkerBase::reflect(r);
		r.member("Polygon", polygon, "");
	}

	Polygon3f polygon;
};

typedef boost::shared_ptr<PolygonMarker> PolygonMarkerPtr;

///////////////////////////////////////////////////////////////////////////////

/**
 * Line list marker. It will draw a line between each pair of points.
 * The scale parameter of the base class is ignored. By default the lines are 1 cm wide.
 * Each point of the line will be transformed by the position parameter of the
 * base class.
 */
class MIRA_RIGID_MODEL_EXPORT LineListMarker : public MarkerBase
{
	MIRA_OBJECT(LineListMarker)
public:
	LineListMarker() : lineWidth(0.01f) {}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MarkerBase::reflect(r);
		r.member("LineWidth", lineWidth, "", 0.01f);
		r.member("Lines", lines, "");
	}
	float lineWidth;
	std::vector<Line3f> lines;
};

typedef boost::shared_ptr<LineListMarker> LineListMarkerPtr;

///////////////////////////////////////////////////////////////////////////////

/**
 * Line strip marker. It will draw lines between every two consecutive points (0-1,1-2,2-3,..)
 * The scale parameter of the base class is ignored. By default the lines are 1 cm wide.
 * Each point of the line will be transformed by the position parameter of the
 * base class.
 */
class MIRA_RIGID_MODEL_EXPORT LineStripMarker : public MarkerBase
{
	MIRA_OBJECT(LineStripMarker)
public:
	LineStripMarker() : lineWidth(0.01f) {}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MarkerBase::reflect(r);
		r.member("LineWidth", lineWidth, "", 0.01f);
		r.member("Points", points, "");
	}
	float lineWidth;
	std::vector<Point3f> points;
};

typedef boost::shared_ptr<LineStripMarker> LineStripMarkerPtr;
///////////////////////////////////////////////////////////////////////////////

/**
 * Marker message.
 * Has an id a namespace and a command.
 * Contains multiple marker objects for rendering.
 */
struct Marker
{
	enum Command
	{
		CREATE_MODIFY = 0, /// Create or modify marker for namespace and id
		REMOVE = 1         /// Remove marker with namespace and id
	};
	Marker() :
		id(0),
		ns(""),
		command(CREATE_MODIFY) {}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("ID", id, "");
		r.member("Namespace", ns, "");
		r.member("Command", command, "");
		r.member("Objects", objects, "");
	}
	uint32 id;
	std::string ns;
	Command command;
	std::vector<MarkerBasePtr> objects;
};

///////////////////////////////////////////////////////////////////////////////

}} // namespace

#endif
