/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file CollisionTest.h
 *    Class used for collision testing of rigid model footprints with maps.
 *
 * @author Tim Langner
 * @date   2011/05/09
 */

#ifndef _MIRA_COLLISIONTEST_H_
#define _MIRA_COLLISIONTEST_H_

#include <serialization/DefaultInitializer.h>
#include <image/Img.h>
#include <math/Angle.h>
#include <utils/Path.h>
#include <geometry/Point.h>

#include <model/Footprint.h>
#include <model/RigidModelExports.h>

namespace mira { namespace model {

///////////////////////////////////////////////////////////////////////////////

/**
 * Class that provides several methods for running collision tests of a rigid
 * models footprint with images.
 * For optimization a LUT is build for a provided footprint. 
 * After initializing the LUT collision checks can be done for an arbitrary 
 * rotation of the footprint. This class is used for collision detection and 
 * avoidance in navigation tasks.
 * @note Images and maps (e.g. GridMap) have different coordinate systems.
 *       When using this class with images note that the upper left corner
 *       is 0,0 and the lower right corner is xsize, ysize.
 *       The phi parameter used in the methods of this class rotates the
 *       shape clockwise starting with 0° at 3 o'clock.
 *       Maps are internally images where y is flipped so the upper left corner 
 *       is 0, ysize and the lower right is xsize, 0. So phi rotates 
 *       counterclockwise starting with 0° at 3 o'clock.
 *       (This is handled automatically so no need to worry)
 */
class MIRA_RIGID_MODEL_EXPORT CollisionTest
{
public:

	CollisionTest(float cellSize = 0.02f, uint32 segments=360)
	{
		MIRA_INITIALIZE_THIS;
		mCellSize = cellSize;
		mSegments = segments;
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("CellSize", mCellSize, 
		         "Resolution of the collision map [m]", 0.02f);
		r.member("Segments", mSegments, 
		         "Size of the collision LUT (360 deg is divided in #segments)",
		         360);
		r.property("ObstacleThreshold", mObstacleThreshold,
		           "Occupancy grid values above this threshold are counted as obstacles",
		           140, PropertyHints::limits(0, 255) | PropertyHints::step(1));
	}

	/**
	 * Tests collision of the enclosed footprint with a given map for rotation phi
	 * @param map The map in which the collision check is done
	 * @param p The position of the model in the map
	 * @param phi The orientation of the model in the map
	 * @return true if the footprint rotated about phi, at position p collides with an 
	 *         obstacle in the map (footprint contains 255 at a pixel where the map contains
	 *         a value > mObstacleThreshold)
	 */
	bool testCollision(const Img8U1& map, const Point2i& p, float phi) const;

	/**
	 * Calculates the distance of the enclosed footprint to the nearest obstacle in the
	 * given map
	 * @param distanceMap The distance transformed map in which the collision check is done
	 * @param p The position of the model in the map
	 * @param phi The orientation of the model in the map
	 * @param oClosestModelPoint Optionally returns the position of the point on
	 *             the model with closest obstacle proximity. Can be used to find the actual
	 *             obstacle position using \ref obstaclePosition.
	 * @return distance in pixel between the shape pixel in the footprint rotated 
	 *         about phi, at position p that is closest to an obstacle in the map.
	 */
	float distanceToObstacle(const Img32F1& distanceMap, const Point2i& p, float phi,
	                         Point2i* oClosestModelPoint = NULL) const;

	/**
	 * Determine the position of the obstacle
	 * @param distanceMap The distance transformed map in which the collision check was done
	 * @param labelMap The label map returned by the distance transformation
	 * @param closestModelPoint The position on the model with closest obstacle proximity
	 *        (returned by distanceToObstacle())
	 * @param ioCache Optionally, obstacle points corresponding to labels can be cached
	 *        for speedup. The cache is associated with the map (not shape) and
	 *        therefore held outside. It is queried and updated inside this function.
	 *        Cache must be cleared when distance transformation is recomputed.
	 * @return Position of the obstacle in the map (or empty if there are no
	 *         obstacles in the map at all)
	 * @exception XRuntime Obstacle should be there but position was not found in map
	 */
	static boost::optional<Point2i> obstaclePosition(
	                                  const Img32F1& distanceMap, const Img<int32>& labelMap,
	                                  const Point2i& closestModelPoint,
	                                  std::map<int, Point2i>* ioCache = NULL);

	/**
	 * Calculates a distance transformation on the given map.
	 * In the distance transformed map each pixel contains the distance value to
	 * the closest obstacle (denoted by values > mObstacleThreshold in the map).
	 * This allows fast retrieval of distances between the robots footprint
	 * (a shape in the LUT) and an obstacle by using the distanceToObstacle()
	 * method.
	 * @param map The map to transform
	 * @param oDistanceMap The resulting distance transformed map 
	 */
	void distanceTransform(const Img8U1& map, Img32F1& oDistanceMap) const;

	/**
	 * Same as distanceTransform(const Img8U1&, Img32F1&) const,
	 * but also returns a label map which allows to determine the position of the obstacle in
	 * addition to its distance.
	 * @param map The map to transform
	 * @param oDistanceMap The resulting distance transformed map
	 * @param oLabelMap The resulting label map
	 * @exception XNotImplemented Not implemented (available OpenCV version is < 2.4)
	 */
	void distanceTransform(const Img8U1& map, Img32F1& oDistanceMap,
	                       Img<int32>& oLabelMap) const;

	/**
	 * Calculates a distance transformation on the given map and clears 
	 * the rotated footprint at position p in the map.
	 * In the distance transformed map each pixel contains the distance value to
	 * the closest obstacle (denoted by values > mObstacleThreshold in the map).
	 * This allows fast retrieval of distances between the robots footprint
	 * (a shape in the LUT) and an obstacle by using the distanceToObstacle()
	 * method.
	 * @param map The map to transform
	 * @param oDistanceMap The resulting distance transformed map
	 * @param p The position of the model in the map
	 * @param phi The orientation of the model in the map
	 */
	void distanceTransformAndClearShape(const Img8U1& map, Img32F1& oDistanceMap, 
	                                    const Point2i& p, float phi) const;

	/**
	 * Same as distanceTransformAndClearShape(const Img8U1&, Img32F1&, const Point2i&, float) const,
	 * but also returns a label map which allows to determine the position of the obstacle in
	 * addition to its distance.
	 * @param map The map to transform
	 * @param oDistanceMap The resulting distance transformed map
	 * @param oLabelMap The resulting label map
	 * @param p The position of the model in the map
	 * @param phi The orientation of the model in the map
	 * @exception XNotImplemented Not implemented (available OpenCV version is < 2.4)
	 */
	void distanceTransformAndClearShape(const Img8U1& map, Img32F1& oDistanceMap,
	                                    Img<int32>& oLabelMap,
	                                    const Point2i& p, float phi) const;

	/**
	 * Return the entry of the LUT that is closest to the representation of the 
	 * enclosed footprint rotated about phi.
	 */
	const Img8U1& getShape(float phi) const;

	/**
	 * Initialize collision testing with the given footprint. (Re)builds the internal
	 * LUT
	 * @param footprint The new footprint to use.
	 */
	
	void initialize(const Footprint& footprint);

	/**
	 * Initialize collision testing with the given footprint and cell size.
	 * (Re)builds the internal LUT
	 * @param footprint The new footprint to use.
	 * @param cellSize  The cell size of the maps that are later used for
	 *                  collision test (spatial precision)
	 */
	void initialize(const Footprint& footprint, float cellSize);

	/**
	 * Initialize collision testing with the given footprint, cell size and
	 * angular precision.
	 * (Re)builds the internal LUT
	 * @param footprint The new footprint to use.
	 * @param cellSize  The cell size of the maps that are later used for
	 *                  collision test (spatial precision)
	 * @param segments  The number of rotation segments, that are used when
	 *                  building the LUT. Each segment represents a certain
	 *                  orientation. The higher the number of segments, the
	 *                  higher is the angular precision in the collision tests.
	 */
	void initialize(const Footprint& footprint, float cellSize, uint32 segments);

	/**
	 * Clears the map region that is covered by the rotated footprint
	 * at position p in the given map.
	 * @param map The map to clear
	 * @param p The position of the model in the map
	 * @param phi Orientation of the model in the map
	 * @param freeValue the value that is used to clear the map (default: 0)
	 */
	void clearShape(Img8U1& map, const Point2i& p, float phi,
	                uint8 freeValue = 0) const;

protected:

	float mCellSize;  ///< Resolution of the collision map [m]
	uint32 mSegments; ///< Size of the collision LUT (360° is divided in #segments)

	std::vector<Img8U1> mShapeLUT; ///< Look up table for faster collision checking

	/// Each LUT entry has this dimension (e.g. is a dim x dim image)
	int mShapeDimension;

	/// Occupancy grid values above this threshold are counted as obstacles
	int mObstacleThreshold;

	/// Denotes a step in rad between each LUT entry (2pi / mSegments)
	float mSegmentStep;
};

///////////////////////////////////////////////////////////////////////////////

}}

#endif
