/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file VisualizationPlotView.h
 *    Abstract base class of a visualization view for plotting data in 2d using Qwt.
 *
 * @author Tim Langner
 * @date   2011/08/23
 */

#ifndef _MIRA_VISUALIZATIONPLOTVIEW_H_
#define _MIRA_VISUALIZATIONPLOTVIEW_H_

#include <qwt_plot.h>

#include <QtGui/QDragMoveEvent>

#include <visualization/VisualizationPlot.h>
#include <views/VisualizationView.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Abstract base class of a 2D plot view of a scene that may contain different visualization objects which
 * show the actual content of the scene.
 */
class VisualizationPlotView : public VisualizationViewTransformable, public IVisualizationPlotSite
{

public:
	VisualizationPlotView();

	virtual ~VisualizationPlotView()
	{
		destroyVisualizations();
	}

public:

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		VisualizationViewTransformable::reflect(r);
		r.property("YLeftAutoScale", getter<bool>(boost::bind(&VisualizationPlotView::getAutoScaleY, this, QwtPlot::yLeft)),
			                         setter<bool>(boost::bind(&VisualizationPlotView::setAutoScaleY, this, QwtPlot::yLeft, _1)),
			                         "Enable automatic scaling of left y axis", true);
		r.property("YLeftMin", getter<float>(boost::bind(&VisualizationPlotView::getYMin, this, QwtPlot::yLeft)),
			                   setter<float>(boost::bind(&VisualizationPlotView::setYMin, this, QwtPlot::yLeft, _1)),
			                   "Minimum value of left y axis", 0.0f);
		r.property("YLeftMax", getter<float>(boost::bind(&VisualizationPlotView::getYMax, this, QwtPlot::yLeft)),
			                   setter<float>(boost::bind(&VisualizationPlotView::setYMax, this, QwtPlot::yLeft, _1)),
			                   "Maximum value of left y axis", 1.0f);
		r.property("YRightAutoScale", getter<bool>(boost::bind(&VisualizationPlotView::getAutoScaleY, this, QwtPlot::yRight)),
			                          setter<bool>(boost::bind(&VisualizationPlotView::setAutoScaleY, this, QwtPlot::yRight, _1)),
			                          "Enable automatic scaling of right y axis", true);
		r.property("YRightMin", getter<float>(boost::bind(&VisualizationPlotView::getYMin, this, QwtPlot::yRight)),
			                    setter<float>(boost::bind(&VisualizationPlotView::setYMin, this, QwtPlot::yRight, _1)),
			                    "Minimum value of right y axis", 0.0f);
		r.property("YRightMax", getter<float>(boost::bind(&VisualizationPlotView::getYMax, this, QwtPlot::yRight)),
			                    setter<float>(boost::bind(&VisualizationPlotView::setYMax, this, QwtPlot::yRight, _1)),
			                    "Maximum value of right y axis", 1.0f);
		r.property("ShowLegend", getter(&VisualizationPlotView::legendEnabled, this), setter(&VisualizationPlotView::enableLegend,this),"Show legend above plot",true);
		r.property("MaxPlotHistory", mMaxPlotHistory, "Maximum history of the plot in milliseconds (0 means infinite)", Duration::seconds(0));
	}

public: // implementation of VisualizationView

	virtual QwtPlot* getPlot();
	virtual Duration getMaxPlotHistory() { return mMaxPlotHistory; }

	virtual void updateAxes() = 0;

	virtual void addVisualization(Visualization* vis);

	virtual void moveUpVisualization(Visualization* vis);
	virtual void moveDownVisualization(Visualization* vis);

	virtual const Class& supportedVisualizationClass() const = 0;

	virtual const Class& defaultVisualizationClass() const = 0;

	virtual const std::string& getFixedFrame() const {
		return mFixedFrame.getID();
	}

	virtual const std::string& getCameraFrame() const {
		return mCameraFrame.getID();
	}

	virtual bool isAcceptingUpdates() { return mAcceptingUpdates; }

protected:

	virtual QWidget* createVisualizationPart();

	void setupVisualizations();

	virtual void update(Duration dt);

	void setAutoScaleY(int axis, bool enabled);
	bool getAutoScaleY(int axis) { return mAutoScaleY[axis]; }

	void setYMin(int axis, float y);
	float getYMin(int axis) { return mYMin[axis]; }

	void setYMax(int axis, float y);
	float getYMax(int axis) { return mYMax[axis]; }

	void updateScale(int axis);

	void enableLegend( bool enabled );
	bool legendEnabled() const;

	bool mAutoScaleY[2];

	float mYMax[2];

	float mYMin[2];

	bool mLegendEnabled;

	Duration mMaxPlotHistory;

	bool mAcceptingUpdates;

protected:

	class UI;
	UI* ui;

	/**
	 * Overwrite this method in derived classes if you have a derived UI class.
	 */
	virtual UI* createUI();

private:
	friend class UI;

	void reattachVisualizationItems();

protected:
	void createDefaultVisualization(const std::string& className, const std::string& displayName);
	virtual void populateDefaultVisualizations() = 0;
};


///////////////////////////////////////////////////////////////////////////////

class VisualizationPlotView::UI : public QwtPlot
{
	Q_OBJECT
public:

	UI(VisualizationPlotView* parent);

	virtual ~UI()
	{
	}

public slots:
	void popupMenu(const QPoint& pos);

public:
	// make public when it is not (qwt 5.1?)
	using QwtPlot::updateAxes;

protected:

	virtual void populatePopupMenu(QMenu& menu) {}
	virtual void onPopupMenuAction(QAction* action) {}

	virtual void dragEnterEvent(QDragEnterEvent *event) {
		setAcceptDrops(true);
		vis->dragEnterEventImpl(event);
	}

	virtual void dropEvent(QDropEvent *event) {
		vis->dropEventImpl(event, this);
	}

	// we need to overwrite this to prevent the QGraphicsView from disturbing our drag n drop
	virtual void dragMoveEvent(QDragMoveEvent* event) {
		event->accept();
	}

	// we need to overwrite this to prevent the QGraphicsView from disturbing our mouse move event
	virtual void mouseMoveEvent(QMouseEvent * event)
	{
		event->accept();
	}

	void freeze(bool freeze);

	VisualizationPlotView* vis;
};


///////////////////////////////////////////////////////////////////////////////

}

#endif 
