/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TaskVisualization.C
 *    Visualization for navigation tasks.
 *
 * @author Tim Langner
 * @date   2012/02/23
 */

#include <visualization/Visualization3DBasic.h>
#include <visualization/3d/AxesCovObject.h>
#include <visualization/3d/CovObject.h>

#include <serialization/DefaultInitializer.h>
#include <serialization/SetterNotify.h>
#include <fw/TransformProperty.h>

#include <navigation/Task.h>
#include <navigation/tasks/DockingTask.h>
#include <navigation/tasks/OrientationTask.h>
#include <navigation/tasks/PositionTask.h>

namespace mira { namespace navigation {

///////////////////////////////////////////////////////////////////////////////

/// Visualization for navigation tasks.
class TaskVisualization :  public Visualization3DBasic<TaskPtr>
{
	MIRA_META_OBJECT(TaskVisualization,
			("Name", "Task")
			("Description", "3D Visualization of a navigation task")
			("Category", "Navigation"))

	typedef Visualization3DBasic<TaskPtr> Base;

public:

	TaskVisualization() {
		MIRA_INITIALIZE_THIS;
	}

	template <typename Reflector>
	void reflect(Reflector& r) {
		Base::reflect(r);
		r.property("Fallback Draw Frame", mDrawFrame,
			       "Frame to use as fallback for visualizing tasks without position",
			       "/robot/RobotFrame");
		r.property("Show Orientation Tolerance", mShowOrientationCov,
		            setterNotify(mShowOrientationCov, &TaskVisualization::drawLast, this),
		           "Enable/disable drawing of orientation covariance", true);
	}

	virtual void setupScene(Ogre::SceneManager* mgr, Ogre::SceneNode* node)
	{
		mCovObject.reset(new CovObject(mgr, node));
		mCovObject->setCovariance(Eigen::Matrix2f());
		mCovObject->setVisible(false);
		mPositionVisible = false;
		mAxesObject.reset(new AxesObject(mgr, node));
		mAxesObject->showZAxis(false);
		mAxesObject->setVisible(false);
		mAxisVisible = false;
		mAxesCovObject.reset(new AxesCovObject(mgr, mAxesObject.get()));
		mAxesCovObject->setVisible(false);
		mAxesCovObject->setCovariance(0.0f);
		mAxisCovVisible = false;
	}

	void setEnabled(bool enabled) {
		Visualization3D::setEnabled(enabled);
		mAxesObject->setVisible(enabled & mAxisVisible);
		mCovObject->setVisible(enabled & mPositionVisible);
		mAxesCovObject->setVisible(enabled & mAxisCovVisible);
	}

	virtual void update(Duration dt) {
		drawLast();
	}

	void dataChanged(ChannelRead<TaskPtr> task)
	{
		if (task->value())
			draw(task->value());
	}

	void draw(TaskPtr task)
	{
		PoseCov2 pose;
		mPositionVisible = false;
		mAxisVisible = false;
		mAxisCovVisible = false;
		// handle null task
		if (task)
		{
			auto positionTask = task->getSubTask<navigation::PositionTask>();
			if (positionTask)
			{
				RigidTransform2f d = getAuthority().getTransform<RigidTransform2f>(
						positionTask->frameID, Time::now(),
						getSite()->getCameraFrame(), Time(),
						getSite()->getFixedFrame());
				Pose2 transformedPosition = (d * Pose2(positionTask->position, 0.0));
				pose.x() = transformedPosition.x();
				pose.y() = transformedPosition.y();
				pose.cov(0,0) = positionTask->maxTolerance;
				pose.cov(1,1) = positionTask->maxTolerance;

				mPositionVisible = true;
			}
			auto orientationTask = task->getSubTask<navigation::OrientationTask>();
			if (orientationTask)
			{
				if (positionTask)
				{
					RigidTransform2f d = getAuthority().getTransform<RigidTransform2f>(
							positionTask->frameID, Time::now(),
							getSite()->getCameraFrame(), Time(),
							getSite()->getFixedFrame());
					Pose2 transformedPosition = (d * Pose2(0.0, 0.0, orientationTask->phi));
					pose.phi() = transformedPosition.phi();
				}
				else
				{
					// no position known, bind the task visualization to DrawFrame property,
					// and (only in this case) store the task to update in case DrawFrame moves
					RigidTransform2f d = getAuthority().getTransform<RigidTransform2f>(
							mDrawFrame, Time::now(),
							getSite()->getCameraFrame(), Time(),
							getSite()->getFixedFrame());
					pose.x() = d.x();
					pose.y() = d.y();

					pose.cov(0,0) = 0.1;
					pose.cov(1,1) = 0.1;

					d = getAuthority().getTransform<RigidTransform2f>(
							orientationTask->frameID, Time::now(),
							getSite()->getCameraFrame(), Time(),
							getSite()->getFixedFrame());
					Pose2 transformedPosition = (d * Pose2(0.0, 0.0, orientationTask->phi));

					pose.phi() = transformedPosition.phi();
				}
				pose.cov(2,2) = orientationTask->tolerance;
				mAxisVisible = true;
				mAxisCovVisible = mShowOrientationCov;
			}
			auto dockingTask = task->getSubTask<navigation::DockingTask>();
			if (dockingTask)
			{
				pose = dockingTask->pose;
				pose.cov(0,0) = dockingTask->tolerance.x();
				pose.cov(1,1) = dockingTask->tolerance.y();
				pose.cov(2,2) = dockingTask->tolerance.phi();

				mPositionVisible = true;
				mAxisVisible = true;
				mAxisCovVisible = mShowOrientationCov;
			}
		}
		Ogre::Vector3 v(pose.x(), pose.y(), 0.0f);
		mAxesObject->setOrientation(Eigen::Quaternionf(
				Eigen::AngleAxisf(pose.phi(), Eigen::Vector3f::UnitZ())));
		mAxesObject->setPosition(v);
		mCovObject->setPosition(v);
		mCovObject->setCovariance((Eigen::Matrix2f)pose.cov.block<2,2>(0,0));
		mAxesCovObject->setCovariance(pose.cov(2,2));
		mAxesObject->setVisible(isEnabled() & mAxisVisible);
		mCovObject->setVisible(isEnabled() & mPositionVisible);
		mAxesCovObject->setVisible(isEnabled() & mAxisCovVisible);
	}

	void drawLast()
	{
		try
		{
			if (!mDataChannel.isValid())
				return;
			dataChanged(mDataChannel.getChannel().read());
		}
		catch(Exception&)
		{
		}
	}

protected:

	boost::shared_ptr<AxesObject> mAxesObject;
	boost::shared_ptr<AxesCovObject> mAxesCovObject;
	boost::shared_ptr<CovObject> mCovObject;

	bool mPositionVisible;
	bool mAxisVisible;
	bool mAxisCovVisible;

	bool mShowOrientationCov;

	TransformProperty mDrawFrame;
};

///////////////////////////////////////////////////////////////////////////////

}}

MIRA_CLASS_SERIALIZATION(mira::navigation::TaskVisualization, mira::Visualization3D);

