/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file INavigation.h
 *    Interface for navigation services.
 *
 * @author Tim Langner
 * @date   2012/02/27
 */

#ifndef _MIRA_INAVIGATION_H_
#define _MIRA_INAVIGATION_H_

#include <transform/Pose.h>
#include <navigation/Task.h>

namespace mira { namespace navigation {

///////////////////////////////////////////////////////////////////////////////

/**
 * Interface for navigation services
 */
class INavigation
{
public:
	/// The reflect method.
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.interface("INavigation");
		r.method("setGoal", &INavigation::setGoal, this, "Set a goal");
		r.method("setTask", &INavigation::setTask, this,
		         "Set a new task. NULL resets task and stops navigation.");
		r.method("repeatTask", &INavigation::repeatTask, this,
		         "Repeat the last task (if any)");
	}

	/// The destructor
	virtual ~INavigation() {};

public:

	/**
	 * Set a goal with given tolerances
	 * @param[in] goal   The goal position
	 * @param[in] transT The translational tolerance for reaching the goal in [m].
	 * @param[in] rotT   The rotational tolerance for reaching the goal in [rad].
	 */
	virtual void setGoal(const Pose2& goal, float transT, float rotT) = 0;

	/**
	 * Set a task. If task is NULL it act as a reset of the currently set task
	 * and should stop navigation.
	 * @param[in] task The task. NULL resets current task.
	 */
	virtual void setTask(TaskPtr task) = 0;

	/**
	 * Repeat the last task (if any).
	 */
	virtual void repeatTask() = 0;
};

///////////////////////////////////////////////////////////////////////////////

}}

#endif
