/*
 * Copyright (C) 2014 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file FramePublisherView.C
 *    Implementation of FramePublisherView.h.
 *
 * @author Erik Einhorn
 * @date   2014/04/03
 */

#include "private/FramePublisherView.h"

#include <math/Angle.h>
#include <math/YawPitchRoll.h>

#include <QTimer>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

FramePublisherView::FramePublisherView() :
	ui(NULL)
{
}

FramePublisherView::~FramePublisherView()
{
}

QWidget* FramePublisherView::createPartControl()
{
	ui = new UI(this);
	return ui;
}

static QDoubleSpinBox* initSpinBox(QDoubleSpinBox* sb)
{
	sb->setMinimum(-10.0e8);
	sb->setMaximum(10.0e8);
	sb->setSingleStep(0.1);
	sb->setDecimals(3);
	return sb;
}

static QDoubleSpinBox* initSpinBoxAngle(QDoubleSpinBox* sb)
{
	sb->setMinimum(-180);
	sb->setMaximum(180);
	return sb;
}

FramePublisherView::UI::UI(FramePublisherView* parent) :
	QWidget(parent),
	authority("/", "FramePublisherView", Authority::ANONYMOUS | Authority::INTERNAL),
	mParent(parent)
{
	mLayout = new QFormLayout(this);


	mSbX = initSpinBox(new QDoubleSpinBox(this));
	mSbY = initSpinBox(new QDoubleSpinBox(this));
	mSbZ = initSpinBox(new QDoubleSpinBox(this));
	mSbYaw   = initSpinBoxAngle(new QDoubleSpinBox(this));
	mSbPitch = initSpinBoxAngle(new QDoubleSpinBox(this));
	mSbRoll  = initSpinBoxAngle(new QDoubleSpinBox(this));

	connect(mSbX, SIGNAL(valueChanged(double)), this, SLOT(updateFrame()));
	connect(mSbY, SIGNAL(valueChanged(double)), this, SLOT(updateFrame()));
	connect(mSbZ, SIGNAL(valueChanged(double)), this, SLOT(updateFrame()));
	connect(mSbYaw,   SIGNAL(valueChanged(double)), this, SLOT(updateFrame()));
	connect(mSbPitch, SIGNAL(valueChanged(double)), this, SLOT(updateFrame()));
	connect(mSbRoll,  SIGNAL(valueChanged(double)), this, SLOT(updateFrame()));

	mLeFrame = new QLineEdit(this);
	connect(mLeFrame, SIGNAL(textChanged(const QString &)), this, SLOT(updateControls()));

	mLayout->addRow("Frame ID:", mLeFrame);
	mLayout->addRow("X:", mSbX);
	mLayout->addRow("Y:", mSbY);
	mLayout->addRow("Z:", mSbZ);
	mLayout->addRow("Yaw:", mSbYaw);
	mLayout->addRow("Pitch:", mSbPitch);
	mLayout->addRow("Roll:", mSbRoll);

	QTimer* timer = new QTimer(this);
	timer->setInterval(2000);
	timer->setSingleShot(false);

	connect(timer, SIGNAL(timeout()), this, SLOT(updateControls()));
	timer->start();


}

void FramePublisherView::UI::updateControls()
{
	std::string frameID = mLeFrame->text().toStdString();
	if(frameID.empty())
		return;

	try {
		authority.getTransformNode(frameID);
	}
	catch(XInvalidRead&) {
		 // set default values in UI for non-existent frame
		mSbX->setValue(0.f);
		mSbY->setValue(0.f);
		mSbZ->setValue(0.f);

		mSbYaw->setValue(0.f);
		mSbPitch->setValue(0.f);
		mSbRoll->setValue(0.f);
		return;
	}

	try {
		auto ch = authority.subscribe<RigidTransform<float,3>>(frameID);
		auto r = ch.read();

		RigidTransform<float,3> tf = r->value();

		mSbX->setValue(tf.x());
		mSbY->setValue(tf.y());
		mSbZ->setValue(tf.z());

		mSbYaw->setValue(rad2deg(tf.yaw()));
		mSbPitch->setValue(rad2deg(tf.pitch()));
		mSbRoll->setValue(rad2deg(tf.roll()));

	} catch(...) {}

}

void FramePublisherView::UI::updateFrame()
{
	std::string frameID = mLeFrame->text().toStdString();
	if(frameID.empty())
		return;

	try {
		authority.getTransformNode(frameID);
	}
	catch(XInvalidRead&) { return; }  // ignore non-existent frame

	// extract the values into a transform

	RigidTransform<float,3> tf;

	tf.t.x() = mSbX->value();
	tf.t.y() = mSbY->value();
	tf.t.z() = mSbZ->value();

	float yaw = deg2rad(mSbYaw->value());
	float pitch = deg2rad(mSbPitch->value());
	float roll = deg2rad(mSbRoll->value());

	tf.r = quaternionFromYawPitchRoll(yaw,pitch,roll);

	auto ch = authority.subscribe<RigidTransform<float,3>>(frameID);
	Time timestamp = Time::now();

	try {
		auto r = ch.read();
		timestamp= r->timestamp + Duration::microseconds(1);
	} catch(...) {}

	authority.publishTransform<RigidTransform<float,3>>(frameID, tf, timestamp);


}


///////////////////////////////////////////////////////////////////////////////

}
MIRA_CLASS_SERIALIZATION(mira::FramePublisherView, mira::ViewPart);
