/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PoseVisualization.C
 *    A simple pose text visualization.
 *
 * @author Tim Langner
 * @date   2011/01/26
 */

#include <transform/Pose.h>
#include <visualization/text/VisualizationTextJSON.h>

namespace mira { namespace gui {

///////////////////////////////////////////////////////////////////////////////

template<typename Pose>
class PoseTextVisualizationBase : public VisualizationTextJSONBase<Pose>
{
	typedef VisualizationTextJSONBase<Pose> Base;
public:
	PoseTextVisualizationBase() :
		VisualizationTextJSONBase<Pose>("Pose")
	{
		mShowUntransformed = false;
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		Base::reflect(r);
		r.property("Show Untransformed", mShowUntransformed,
			setter(&PoseTextVisualizationBase::onShowUntransformedChanged, this),
			"If true, the original transformation stored in the channel is displayed, instead of transforming it into the camera frame.", false);
	}

	virtual void dataChanged(ChannelRead<Pose> data)
	{
		// if channel is no valid node in the transform tree we just plot it
		if (mShowUntransformed || MIRA_FW.getTransformer()->getNode(this->mChannel.getID()) == NULL)
			Base::dataChanged(data);
	}

	virtual void update(Duration dt)
	{
		if (mShowUntransformed)
			return; // nothing to do, update is handled in dataChanged()

		if (!this->mChannel.isValid())
			return;
		if (MIRA_FW.getTransformer()->getNode(this->mChannel.getID()) == NULL)
			return;


		try
		{
			Time n = Time::now();
			Pose d;
			d = this->getAuthority().template getTransform<Pose>(this->mChannel.getID(),
			                                                     this->getSite()->getCameraFrame(),
			                                                     n);
			JSONSerializer s;
			json::Value v = s.serialize(d);
			this->mLastValue.reset(v);
			this->mItem->setText(json::write(v,true,this->mPrecision));
			this->mItem->setStamped(n, this->mChannel.getID(), 0);
			this->ok("Text");
		}
		catch(Exception& ex)
		{
			this->error("Text", ex.message());
		}
	}

	void onShowUntransformedChanged(bool show)
	{
		mShowUntransformed = show;
		if (!this->mChannel.isValid())
			return;

		if(mShowUntransformed) {
			try{
				auto r = this->mChannel.getChannel().read();
				Base::dataChanged(r);
			}
			catch (XInvalidRead&) {}
		}
	}

	virtual void writeJSON(const Time& timestamp, const std::string& frameID,
	                       int sequenceID, const json::Value& value)
	{
		// writing back the shown value (default editor content) should not change the channel!
		// --> only write directly for non-transform tree poses or if "Show Untransformed" is selected
		std::string channelID = this->mChannel.getID();
		if (MIRA_FW.getTransformer()->getNode(channelID) == NULL) {
			VisualizationTextJSONBase<Pose>::writeJSON(timestamp, frameID, sequenceID, value);
			return;
		} else if (mShowUntransformed) {
			// ignore frameID/sequenceID from dialog for transform tree elements
			VisualizationTextJSONBase<Pose>::writeJSON(timestamp, channelID, 0, value);
			return;
		}

		// otherwise, publish transform relative to reference frame
		Pose d;
		JSONDeserializer ds(value);
		ds.deserialize(d);

		this->getAuthority().template publishTransformIndirect<Pose>(this->mChannel.getID(),
		                                                             this->mChannel.getID(),
		                                                             this->getSite()->getCameraFrame(),
		                                                             d, timestamp);
	}

	bool mShowUntransformed;
};

///////////////////////////////////////////////////////////////////////////////

class Pose2TextVisualization : public PoseTextVisualizationBase<Pose2>
{
	MIRA_META_OBJECT(Pose2TextVisualization,
		("Name", "Pose 2D")
		("Description", "Visualization of 2D poses as text")
		("Category", "Pose"))
};

class Pose3TextVisualization : public PoseTextVisualizationBase<Pose3>
{
	MIRA_META_OBJECT(Pose3TextVisualization,
		("Name", "Pose 3D")
		("Description", "Visualization of 3D poses as text")
		("Category", "Pose"))
};

class PoseCov2TextVisualization : public PoseTextVisualizationBase<PoseCov2>
{
	MIRA_META_OBJECT(PoseCov2TextVisualization,
		("Name", "Pose 2D")
		("Description", "Visualization of 2D poses with covariance as text")
		("Category", "Pose"))
};

class PoseCov3TextVisualization : public PoseTextVisualizationBase<PoseCov3>
{
	MIRA_META_OBJECT(PoseCov3TextVisualization,
		("Name", "Pose 3D")
		("Description", "Visualization of 3D poses with covariance as text")
		("Category", "Pose"))
};

///////////////////////////////////////////////////////////////////////////////

}}

MIRA_CLASS_SERIALIZATION(mira::gui::Pose2TextVisualization,
                         mira::VisualizationText);
MIRA_CLASS_SERIALIZATION(mira::gui::Pose3TextVisualization,
                         mira::VisualizationText);
MIRA_CLASS_SERIALIZATION(mira::gui::PoseCov2TextVisualization,
                         mira::VisualizationText);
MIRA_CLASS_SERIALIZATION(mira::gui::PoseCov3TextVisualization,
                         mira::VisualizationText);
