/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PolygonVectorVisualization.C
 *    Visualization for vectors of polygons.
 *
 * @author Christian Reuther
 * @date   2017/04/26
 */

#include <serialization/Serialization.h>

#include <visualization/Visualization3DBasic.h>
#include <visualization/3d/PolygonObject.h>
#include <serialization/adapters/OGRE/OgreColourValue.h>

#include <widgets/OgreUtils.h>
#include <OGRE/OgreSceneManager.h>

namespace mira { namespace gui {

///////////////////////////////////////////////////////////////////////////////

/**
 * A visualization for std::vectors of polygons.
 */
template <typename T, int Dim>
class PolygonVectorVisualization3DBase : public Visualization3DBasic<std::vector<boost::geometry::model::ring<Point<T, Dim>>>>
{
public:
	typedef boost::geometry::model::ring<Point<T, Dim>> Poly;
	typedef std::vector<Poly> PolyVector;
	typedef Visualization3DBasic<PolyVector> Base;

public:
	PolygonVectorVisualization3DBase() : mColor(Ogre::ColourValue::Black), mManager(NULL), mSceneNode(NULL) {}
	virtual ~PolygonVectorVisualization3DBase()
	{
		foreach(auto polygon, mPolygons)
			delete polygon;
	}

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		Base::reflect(r);
		r.property("Color", mColor,
		           setter<Ogre::ColourValue>(&PolygonVectorVisualization3DBase::setColor, this),
		           "The color of the grid", Ogre::ColourValue::Black);
	}

	//@}

public:
	void setColor(const Ogre::ColourValue& c)
	{
		mColor = c;
		foreach(auto polygon, mPolygons)
			polygon->setColor(mColor);
	}

public: // Implementation of Visualization3D and Visualization
	virtual void setupScene(Ogre::SceneManager* mgr, Ogre::SceneNode* node)
	{
		mManager = mgr;
		mSceneNode = node;
	}

	virtual void setEnabled(bool enabled)
	{
		Base::setEnabled(enabled);
		foreach(auto polygon, mPolygons)
			polygon->setVisible(enabled);
	}

	void dataChanged(ChannelRead<PolyVector> data)
	{
		const PolyVector& polys = data->value();

		// Create necessary visualization objects (or release unused ones)
		while(mPolygons.size() != polys.size())
		{
			if(mPolygons.size() > polys.size())
			{
				auto* obj = mPolygons.back();
				delete obj;

				mPolygons.pop_back();
			}
			else
			{
				PolygonObject<T>* obj = new PolygonObject<T>(mManager, mSceneNode);
				obj->setColor(mColor);
				obj->setVisible(this->isEnabled());

				mPolygons.push_back(obj);
			}
		}

		// Update the data
		for(std::size_t i = 0; i < polys.size(); ++i)
			mPolygons.at(i)->setPolygon(polys.at(i));
	}

protected:
	std::vector<PolygonObject<T>*> mPolygons;
	Ogre::ColourValue mColor;

	Ogre::SceneManager* mManager;
	Ogre::SceneNode* mSceneNode;
};

///////////////////////////////////////////////////////////////////////////////

class Polygon2fVectorVisualization3D :  public PolygonVectorVisualization3DBase<float, 2>
{
	MIRA_META_OBJECT(Polygon2fVectorVisualization3D,
		("Name", "PolygonVector2f")
		("Description", "Visualizes a vector of 2D float polygons in the xy-plane")
		("Category", "Geometry"))
};

class Polygon2dVectorVisualization3D :  public PolygonVectorVisualization3DBase<double, 2>
{
	MIRA_META_OBJECT(Polygon2dVectorVisualization3D,
		("Name", "PolygonVector2d")
		("Description", "Visualizes a vector of 2D double polygons in the xy-plane")
		("Category", "Geometry"))
};

class Polygon3fVectorVisualization3D :  public PolygonVectorVisualization3DBase<float, 3>
{
	MIRA_META_OBJECT(Polygon3fVectorVisualization3D,
		("Name", "PolygonVector3f")
		("Description", "Visualizes a vector of 3D float polygons")
		("Category", "Geometry"))
};

class Polygon3dVectorVisualization3D :  public PolygonVectorVisualization3DBase<double, 3>
{
	MIRA_META_OBJECT(Polygon3dVectorVisualization3D,
		("Name", "PolygonVector3d")
		("Description", "Visualizes a vector of 3D double polygons")
		("Category", "Geometry"))
};

///////////////////////////////////////////////////////////////////////////////

}} // namespace

MIRA_CLASS_SERIALIZATION(mira::gui::Polygon2fVectorVisualization3D, mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::Polygon2dVectorVisualization3D, mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::Polygon3fVectorVisualization3D, mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::Polygon3dVectorVisualization3D, mira::Visualization3D);
