/*
 * Copyright (C) 2014 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PoseVisualization.C
 *    Visualization of poses in 2D space.
 *
 * @author Christian Reuther
 * @date   2014/08/27
 */

#include <serialization/Serialization.h>
#include <visualization/Visualization2D.h>

#include <QGraphicsScene>
#include <QGraphicsPathItem>

#include <Eigen/Eigen>
#include <Eigen/Eigenvalues>

namespace mira { namespace gui {

///////////////////////////////////////////////////////////////////////////////

// Shamelessly copied from the 3D counterpart (all credits to Erik Einhorn)
template <typename Pose> struct CovVisTrait {
	static const bool poseHasCovariance = false;
};

template<> struct CovVisTrait<Pose2> {
	static const bool poseHasCovariance = false;
};

template<> struct CovVisTrait<PoseCov2> {
	static const bool poseHasCovariance = true;
};

///////////////////////////////////////////////////////////////////////////////

template <typename Pose>
class PoseVisualization2D :  public Visualization2DBasic<Pose> {
public:
	EIGEN_MAKE_ALIGNED_OPERATOR_NEW

public:
	PoseVisualization2D() : mArrowLength(0.1f), mShowName(true), mShowYAxis(true), mItem(NULL),
		mLineXItem(NULL), mLineYItem(NULL), mCovItem(NULL), mNameItem(NULL) {
		mHasCovariance = CovVisTrait<Pose>::poseHasCovariance;
		mShowCovariance = mHasCovariance;
	}

	virtual ~PoseVisualization2D() {
		auto site = this->getSite();
		// For some reason it is possible that the view gets destructed and re-constructed
		// when miracenter starts, so we need to make sure that we can access the items
		if(!mItem || !mLineXItem || !mLineYItem || !mCovItem || !mNameItem || !site || !site->getSceneManager())
			return;

		site->getSceneManager()->removeItem(mItem);
		mItem->removeFromGroup(mLineXItem);
		mItem->removeFromGroup(mLineYItem);
		mItem->removeFromGroup(mCovItem);
		mItem->removeFromGroup(mNameItem);
		delete mLineXItem;
		delete mLineYItem;
		delete mCovItem;
		delete mNameItem;
		// Do not delete mItem as the base class destructor does this.
	}

	template <typename Reflector> void reflect(Reflector& r) {
		Visualization2D::reflect(r);
		this->channelProperty(r, "Pose", mPoseChannel, "");

		r.property("Length", mArrowLength, setter<float>(&PoseVisualization2D::setArrowLength, this),
			"The length of the axes in [m]", 0.1f);
		r.property("Show Name", mShowName, setter<bool>(&PoseVisualization2D::setShowName, this),
			"Should the name be shown", true);
		r.property("Show Covariance", mShowCovariance, setter<bool>(&PoseVisualization2D::setShowCovariance, this),
			"Whether or not to display the covariance error ellipsis for PoseCov2 types", true);
		r.property("Show Y axis", mShowYAxis, setter<bool>(&PoseVisualization2D::setShowYAxis, this),
			"Whether or not to display the (quite redundant) y axis of the 2D pose", true);
	}

public:
	void setArrowLength(float length) {
		mArrowLength = length;
	}

	void setShowName(bool show) {
		mShowName = show;
		if(mNameItem)
			mNameItem->setVisible(mShowName);
	}

	void setShowCovariance(bool show) {
		mShowCovariance = show;
		if(mCovItem)
			mCovItem->setVisible(mShowCovariance);
	}

	void setShowYAxis(bool show) {
		mShowYAxis = show;
		if(mLineYItem)
			mLineYItem->setVisible(mShowYAxis);
	}

public:
	virtual Visualization::DataConnection getDataConnection() {
		return Visualization::DataConnection(mPoseChannel);
	}

	virtual QGraphicsItem* setupScene(QGraphicsScene* mgr) {
		QPen xpen;
		xpen.setColor(Qt::red);
		xpen.setCosmetic(true);

		QPen ypen;
		ypen.setColor(Qt::green);
		ypen.setCosmetic(true);

		mLineXItem = new QGraphicsLineItem();
		mLineXItem->setPen(xpen);
		mLineYItem = new QGraphicsLineItem();
		mLineYItem->setPen(ypen);
		mCovItem = new QGraphicsEllipseItem();
		mCovItem->setPen(Qt::NoPen);
		mCovItem->setBrush(QColor(255, 255, 255, 128));
		mNameItem = new QGraphicsTextItem();
		mNameItem->setTransform(QTransform::fromScale(1, -1)); // Flipped otherwise
		mNameItem->setScale(0.005); // Approximates the text size as in the 3D view
		mNameItem->setDefaultTextColor(Qt::black);

		mItem = new QGraphicsItemGroup();
		mItem->addToGroup(mLineXItem);
		mItem->addToGroup(mLineYItem);
		mItem->addToGroup(mCovItem);
		mItem->addToGroup(mNameItem);
		mgr->addItem(mItem);

		return mItem;
	}

	virtual void update(Duration dt) {
		if(!mPoseChannel.isValid() || mPoseChannel.getDataUpdateCount() == 0)
			return;

		const std::string frameID = mPoseChannel.getID();
		if(frameID.empty())
			MIRA_THROW(TransformerBase::XTransform, "Data has no frame for transforming it into the views camera frame");

		// Redraw data
		auto r = mPoseChannel.getChannel().read();
		updatePose(r);
	}

protected:
	void updatePose(ChannelRead<Pose> data) {
		const Time now = Time::now();
		const std::string channelID = data.getChannel().getID();
		const std::string fixedFrame = this->getSite()->getFixedFrame();
		const std::string cameraFrame = this->getSite()->getCameraFrame();

		// Determine whether this is a "regular pose" or a node in the transformation tree
		const bool isTransform = (MIRA_FW.getTransformer()->getNode(channelID) != NULL);
		Pose pose = this->getAuthority().template getTransform<Pose>(isTransform ? channelID : data->frameID, now, cameraFrame, now, fixedFrame);
		//pose.t(1) = -pose.t(1);

		// If this is only a "regular" pose, then all we have right now is the global offset of the pose's frame.
		// We therefore need to add the actual pose to the transformation as well
		if(!isTransform)
			pose = pose * transform_cast<Pose>(data->value());

		setPose(pose);

		// Display the name if enabled
		mNameItem->setPlainText(QString::fromStdString(channelID));
		mNameItem->setPos(pose.x(), pose.y());
	}

	void setPose(Pose pose) {
		const float ax = pose.phi();
		const float ay = pose.phi() + pi<float>() * 0.5f;
		const float r = mArrowLength;

		// Calculate our offset points with the length
		const QPointF o(pose.x(), pose.y());
		const QPointF x(o.x() + r * cosf(ax), o.y() + r * sinf(ax));
		const QPointF y(o.x() + r * cosf(ay), o.y() + r * sinf(ay));

		mLineXItem->setLine(QLineF(o, x));
		mLineYItem->setLine(QLineF(o, y));

		if(mHasCovariance && mShowCovariance)
			setCovariance(pose);
	}

	// Override this in derived classes thathave covariances.
	virtual void setCovariance(Pose pose) {}

	ChannelProperty<Pose> mPoseChannel;
	float mArrowLength;
	bool mHasCovariance;

	bool mShowName;
	bool mShowCovariance;
	bool mShowYAxis;

	QGraphicsItemGroup* mItem;
	QGraphicsLineItem* mLineXItem;
	QGraphicsLineItem* mLineYItem;
	QGraphicsEllipseItem* mCovItem;
	QGraphicsTextItem* mNameItem;
};

///////////////////////////////////////////////////////////////////////////////

class Pose2Visualization2D : public PoseVisualization2D<Pose2>
{
	MIRA_META_OBJECT(Pose2Visualization2D,
		("Name", "Pose2")
		("Description", "Visualization of 2D poses")
		("Category", "Pose"))

protected:
	virtual void setCovariance(Pose2 pose) {}
};

class PoseCov2Visualization2D : public PoseVisualization2D<PoseCov2>
{
	MIRA_META_OBJECT(PoseCov2Visualization2D,
		("Name", "PoseCov2")
		("Description", "Visualization of 2D poses with covariance")
		("Category", "Pose"))

protected:
	virtual void setCovariance(PoseCov2 pose) {
		mCovItem->setVisible(false);

		// Construct 2d matrix, we don't care about the phi variance right now
		Eigen::Matrix3d cov3d = pose.cov.cast<double>();
		Eigen::Matrix2d cov = cov3d.block<2,2>(0, 0);

		// Sanity check against infinitesimally small covariance matrices
		if(isApprox<double>(cov.trace(), 0.0, std::numeric_limits<double>::epsilon()))
			return;

		// See http://www.visiondummy.com/2014/04/draw-error-ellipse-representing-covariance-matrix/ for the math
		Eigen::SelfAdjointEigenSolver<Eigen::Matrix2d> eigensolver(cov);
		if(eigensolver.info() != Eigen::Success)
			return;

		const Eigen::Vector2d values = eigensolver.eigenvalues();
		const Eigen::Matrix2d vectors = eigensolver.eigenvectors();

		// Calculate the size of the minor and major axes and the angle of the ellipsis
		const double chi = 1.0f; // In line with the 3D cov visualization. I would rather use 2.4477 for a 95% confidence interval, hmm...
		const double angle = atan2(static_cast<double>(vectors(1,1)), static_cast<double>(vectors(0,1)));
		const double rx = chi * sqrt(static_cast<double>(values(1))); // We take the bigger one as the major axis
		const double ry = chi * sqrt(static_cast<double>(values(0)));

		// Finally set the ellipse
		mCovItem->setPos(static_cast<double>(pose.x()), static_cast<double>(pose.y()));
		mCovItem->setRect(-rx, -ry, 2.0 * rx, 2.0 * ry);
		mCovItem->setRotation(Radiand(pose.phi() + angle).deg());
		mCovItem->setVisible(true);
	}
};

//////////////////////////////////////////////////////////////////////////////

}}

MIRA_CLASS_SERIALIZATION(mira::gui::Pose2Visualization2D, mira::Visualization2D);
MIRA_CLASS_SERIALIZATION(mira::gui::PoseCov2Visualization2D, mira::Visualization2D);
