/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file AxesVisualization.C
 *    Visualization of 2D axes.
 *
 * @author Tim Langner
 * @date   2011/02/06
 */

#include <serialization/Serialization.h>
#include <visualization/Visualization2D.h>

#include <QGraphicsScene>
#include <QGraphicsPathItem>

namespace mira { namespace gui {

///////////////////////////////////////////////////////////////////////////////

/// A visualization of 2D axes.
class AxesVisualization2D :  public Visualization2D
{
	MIRA_META_OBJECT(AxesVisualization2D,
		("Name", "Axes")
		("Description", "Shows the coordinate axes in the origin")
		("Category", "General"))

public:
	/** @name Constructor, destructor and reflect */
	//@{

	/// The constructor.
	AxesVisualization2D() :
		mItem(NULL),
		mXAxis(NULL),
		mYAxis(NULL),
		mAxisLength(1.0f),
		mReferenceFrame("/GlobalFrame")
	{
	}

	/// The destructor.
	virtual ~AxesVisualization2D()
	{
		auto site = this->getSite();
		if(site && site->getSceneManager()) {
			mItem->removeFromGroup(mXAxis);
			mItem->removeFromGroup(mYAxis);
			delete mXAxis;
			delete mYAxis;
			delete mItem;
		}
	}

	/// The reflect method.
	template <typename Reflector>
	void reflect(Reflector& r)
	{
		Visualization2D::reflect(r);
		r.property("Length", mAxisLength,
		           setter<float>(&AxesVisualization2D::setAxisLength, this),
		           "The length of the axes in [m]", 1.0f);
		r.property("Frame", mReferenceFrame, "Frame of the coordinate system origin", "/GlobalFrame");
	}

public:
	/** @name Some setters */
	//@{

	/// Enable/disable the axes visualization
	virtual void setEnabled(bool enabled)
	{
		Visualization2D::setEnabled(enabled);
		mItem->setVisible(enabled);
	}

	/// Set the length of the axis.
	void setAxisLength(float length)
	{
		mAxisLength = length;
		updateAxes();
	}

	//@}

public:
	/** @name Public implementation of Visualization2D */
	//@{

	virtual void setupScene(IVisualization2DSite* site)
	{
		QGraphicsScene* mgr = site->getSceneManager();
		mXAxis = new QGraphicsLineItem();
		mXAxis->setPen(QPen(QBrush(Qt::red), 0.0));
		mYAxis = new QGraphicsLineItem();
		mYAxis->setPen(QPen(QBrush(Qt::green), 0.0));

		mItem = new QGraphicsItemGroup();
		// Above map but below "regular" visualizations
		mItem->setZValue(-0.1);
		mItem->addToGroup(mXAxis);
		mItem->addToGroup(mYAxis);

		updateAxes();

		mgr->addItem(mItem);
	}

	virtual QGraphicsItem* getItem() {
		return mItem;
	}

	//@}

protected:
	virtual void update(Duration dt)
	{
		const Time now = Time::now();
		RigidTransform2f d = getAuthority().getTransform<RigidTransform2f>(
				mReferenceFrame, now, getSite()->getCameraFrame(), now, getSite()->getFixedFrame());
		QtUtils::setTransform(mXAxis, d);
		QtUtils::setTransform(mYAxis, d);
	}

	void updateAxes()
	{
		if (mXAxis) {
			mXAxis->setLine(0, 0, mAxisLength, 0);
			mXAxis->update();
		}
		if (mYAxis) {
			mYAxis->setLine(0, 0, 0, mAxisLength);
			mYAxis->update();
		}
	}

protected:
	QGraphicsItemGroup* mItem;
	QGraphicsLineItem* mXAxis;
	QGraphicsLineItem* mYAxis;
	float mAxisLength;
	std::string mReferenceFrame;
};

///////////////////////////////////////////////////////////////////////////////

}} // namespace

MIRA_CLASS_SERIALIZATION(mira::gui::AxesVisualization2D, mira::Visualization2D);
