###############################################################################
# Copyright (C) 2012 by
#   MetraLabs GmbH (MLAB), GERMANY
# and
#   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
# All rights reserved.
#
# Contact: info@mira-project.org
#
# Commercial Usage:
#   Licensees holding valid commercial licenses may use this file in
#   accordance with the commercial license agreement provided with the
#   software or, alternatively, in accordance with the terms contained in
#   a written agreement between you and MLAB or NICR.
#
# GNU General Public License Usage:
#   Alternatively, this file may be used under the terms of the GNU
#   General Public License version 3.0 as published by the Free Software
#   Foundation and appearing in the file LICENSE.GPL3 included in the
#   packaging of this file. Please review the following information to
#   ensure the GNU General Public License version 3.0 requirements will be
#   met: http://www.gnu.org/copyleft/gpl.html.
#   Alternatively you may (at your option) use any later version of the GNU
#   General Public License if such license has been publicly approved by
#   MLAB and NICR (or its successors, if any).
#
# IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
# INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
# THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
# "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
# BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
# FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
# ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
# PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
#
###############################################################################
#
# Provide some macros for building binary installation packages from the
# MIRA packages.
#
# Author: Christian Martin
#
###############################################################################
#
# Usage:
#
#   MIRA_ADD_INSTALLERS
#     Put this macro at the end of the main CMakeLists.txt for the project
#     to generate the targets for building the binary installers. This macro
#     will create the following make targets:
#
#     ${package}-rpm
#         Build a RPM installers for ${package}. 
#     ${package}-deb
#         Build a DEB installers for ${package}. 
#     ${package}-tar
#         Build a TAR archive for ${package}.
#     ${package}-zip
#         Build a ZIP archive for ${package} for usage in mirapackage.
#
#     new-rpms
#         Build RPM installers based on InstallerBuildStatus (see below)
#     new-debs
#         Build DEB installers based on InstallerBuildStatus (see below)
#     new-tars
#         Build TAR archives based on InstallerBuildStatus (see below)
#     new-zips
#         Build ZIP archives based on InstallerBuildStatus (see below)
#
#     all-rpms
#         Build RPM installers for all(!) packages.
#     all-debs
#         Build RPM installers for all(!) packages.
#     all-tars
#         Build TAR archives for all(!) packages.
#     all-zips
#         Build ZIP archives for all(!) packages.
#
# !!! To use this feature, the directory ${InstallerRootDir} must exists !!!
#
# Since all installers types (RPM/DEB/TAR/ZIP) use the same status file, it is
# not recommended to mix them. Otherwise, not all necessary packages will be
# build for one type, since they might already updated for another type.
#
###############################################################################
#
# The information about the last build binary installation packages are store
# in a text file "${HOME}/MIRA-InstallerBuild/status.txt".
#
# The file consists of one line per package:
#
#   PkgName LastAPIVersion LastVersion LastBuildNr RebuildFlag
#
#     PkgName         The name of the package
#     LastAPIVersion  The API version of last built installer
#     LastVersion     The full version of the last built installer
#     LastBuildNr     The (automatically created) build number
#                     (!! This is NOT the patch number of MIRA_VERSION !!)
#     RebuildFlag     A flag, which indicated, whether a new installer for this
#                     package must be build or not.
#     NewAPIVersion   The new API version.
#     NewVersion      The new full version.
#     NewBuildId      The new build ID.
#
###############################################################################

if (WIN32)
	set(InstallerRootDir "$ENV{HOMEDRIVE}$ENV{HOMEPATH}/MIRA-InstallerBuild")
	set(InstallerStatusPath "${InstallerRootDir}/status.txt")
else()
	set(InstallerRootDir "$ENV{HOME}/MIRA-InstallerBuild")
	set(InstallerStatusPath "${InstallerRootDir}/status.txt")
endif()

###############################################################################
# Disable the whole installer support, if ${InstallerRootDir} does not exist.

if(NOT EXISTS ${InstallerRootDir})
	#file(MAKE_DIRECTORY ${InstallerRootDir})

	macro(MIRA_ADD_INSTALLERS)
		message(STATUS "Installer builds are disabled on this system, since the directory ${InstallerRootDir} does not exist.")
	endmacro(MIRA_ADD_INSTALLERS)

	return() # this will return from this file
endif()

message(STATUS "Storing installer status information in file ${InstallerStatusPath}")

###############################################################################
#
# A macro for getting the current date/time code based on the following post:
#   http://www.cmake.org/pipermail/cmake/2009-February/027014.html
#
macro(__MIRA_GET_DATE_CODE RESULT)
	if (WIN32)
		execute_process(COMMAND "cmd" " /C date /T" OUTPUT_VARIABLE ${RESULT})
		string(REGEX REPLACE "(..)\\.(..)\\.(....).*" "\\3\\2\\1" ${RESULT} ${${RESULT}})
	elseif(UNIX)
		execute_process(COMMAND "date" "+%d/%m/%Y" OUTPUT_VARIABLE ${RESULT})
		string(REGEX REPLACE "(..)/(..)/(....).*" "\\3\\2\\1" ${RESULT} ${${RESULT}})
	else (WIN32)
		message(SEND_ERROR "date not implemented")
		set(${RESULT} 000000)
	endif (WIN32)
endmacro (__MIRA_GET_DATE_CODE)

###############################################################################
#
# An internal macro, which determines the version of the last built installer
# of a package based on the information in "MIRA-InstallerBuild/status.txt"
#
macro(__MIRA_GET_LAST_INSTALLER_VERSION pkgName outAPIVersion outVersion outBuildNr outRebuildFlag)

	unset(${outAPIVersion})
	unset(${outVersion})
	unset(${outBuildNr})
	unset(${outRebuildFlag})

	if (EXISTS "${InstallerStatusPath}")

		file(READ "${InstallerStatusPath}" allBuildStatus)

		# Read the whole InstallerBuildStatus.txt file
		foreach(package_BuildStatus ${allBuildStatus})
			string(STRIP ${package_BuildStatus} package_BuildStatus)

			separate_arguments(package_BuildStatus)
			list(LENGTH package_BuildStatus listLen)

			if(listLen GREATER 4)
				list(GET package_BuildStatus 0 packageName)
				list(GET package_BuildStatus 1 packageAPIVersion)
				list(GET package_BuildStatus 2 packageVersion)
				list(GET package_BuildStatus 3 packageBuildNr)
				list(GET package_BuildStatus 4 packageRebuildFlag)

				if ("${packageName}" STREQUAL "${pkgName}")
					set(${outAPIVersion} ${packageAPIVersion})
					set(${outVersion} ${packageVersion})
					set(${outBuildNr} ${packageBuildNr})
					set(${outRebuildFlag} ${packageRebuildFlag})
				endif()
			endif()

		endforeach(package_BuildStatus)

	endif()

endmacro(__MIRA_GET_LAST_INSTALLER_VERSION)

###############################################################################
#
# An internal macro, which checks if a package requires a new installer-build.
# This is necessary in two cases:
#   a) the API version of a required package has changed
#   b) a required packaged needs a new installer-build
#
macro(__MIRA_CHECK_IF_NEEDS_NEW_INSTALLER pkgName outRebuildFlag)

	foreach(requires ${${pkgName}_REQUIREMENTS})

		list(FIND MIRA_EXTERNAL_PACKAGE_LIST ${requires} isExternalPackage)
		if(${isExternalPackage} EQUAL -1)
			# Get the last installer build information of ${requires}
			__MIRA_GET_LAST_INSTALLER_VERSION(${requires}
			                                  reqLastInstallerAPIVersion
			                                  reqLastInstallerVersion
			                                  reqLastInstallerBuildNr
			                                  reqInstallerRebuildFlag)

			# Case (a):
			#   A rebuild is necessary if the API version of ${requires} has
			#   changed since the last release build of ${requires}
			if ("${reqLastInstallerAPIVersion}" VERSION_LESS "${${requires}_APIVERSION}")
				set(${outRebuildFlag} "true")
			else()
				# Case (b):
				#   Do it recursive...
				unset(${outRebuildFlag}_1)
				__MIRA_CHECK_IF_NEEDS_NEW_INSTALLER(${requires} ${outRebuildFlag}_1)
				if (DEFINED ${outRebuildFlag}_1)
					set(${outRebuildFlag} "true")
				endif()
			endif()
		endif()

	endforeach()

endmacro(__MIRA_CHECK_IF_NEEDS_NEW_INSTALLER)

###############################################################################
#
# An internal macro for updating the information in the status file
# "MIRA-InstallerBuild/status.txt". This macro goes over all local packages of
# this project and checks their status within "MIRA-InstallerBuild/status.txt".
#
# New packages will be added to the file and updated packaged will be marked
# with "RebuildFlag" = true and an already updated BuildID.
#
macro(__MIRA_UPDATE_INSTALLER_STATUS)

	unset(listOfRequiredInstallerBuilds)

	# Get the current date code YYYYMMDD
	__MIRA_GET_DATE_CODE(newBuildID)

	#######################################################
	# Step 1: Check for new required installer-builds 

	foreach(package ${MIRA_INTERNAL_PACKAGE_LIST})

		# Only process this package, if this is a local package and the
		# package has version information
		__MIRA_IS_LOCAL_PACKAGE(${package} packageIsLocal)
		if ("${packageIsLocal}" STREQUAL "true" AND
		    DEFINED ${package}_APIVERSION AND DEFINED ${package}_PATCH)

			#message(STATUS "*** Checking installer update status for ${package}")

			###################################################################
			#
			# First we have to ensure, that we really know all requirements
			# of a package. Since not only package-files may contain a
			# MIRA_REQUIRE_PACKAGE, we have to go over all targets of the
			# current package and check, if one of the auto link libraries
			# belong to a package, which is not yet in ${package}_REQUIREMENTS.
			#
			###################################################################

			foreach(pkgTarget ${${package}_TARGETS})
				if (DEFINED ${pkgTarget}_AUTOLINKLIBS)
					foreach(autoLinkLib ${${pkgTarget}_AUTOLINKLIBS})
						# If this is a package and it is a different(!) package,
						# add it to the requirements list.
						if ((DEFINED ${autoLinkLib}_PACKAGE) AND
						   (NOT "${package}" STREQUAL "${${autoLinkLib}_PACKAGE}"))
							MIRA_ADD_REQUIREMENT_FOR_PACKAGE(${package} ${${autoLinkLib}_PACKAGE})
						endif()
					endforeach()
				endif()
			endforeach()

			#message(STATUS "***   Requirements ${package} = ${${package}_REQUIREMENTS}")

			#########################################################

			# Get the last release build information for ${package}
			unset(lastInstallerAPIVersion)
			unset(lastInstallerVersion)
			unset(lastInstallerBuildNr)
			unset(lastInstallerRebuildFlag)
			__MIRA_GET_LAST_INSTALLER_VERSION(${package}
			                                  lastInstallerAPIVersion
			                                  lastInstallerVersion
			                                  lastInstallerBuildNr
			                                  lastInstallerRebuildFlag)

			unset(requireInstallerBuild)

			# If the package is not in "InstallerBuildStatus.txt", add it now.
			if (NOT DEFINED lastInstallerVersion)
				# The package is not in "InstallerBuildStatus.txt". Therefore,
				# a new (in this case: the first) release-build is necessary.
				set(requireInstallerBuild "true")

				message(STATUS "Adding new package ${package} to InstallerBuildStatus.")
				file(APPEND "${InstallerStatusPath}"
				     "${package} 0.0 0.0.0 0 true ${${package}_APIVERSION} ${${package}_VERSION} ${newBuildID};\n")

			else()
				# The package is already in "InstallerBuildStatus.txt".
				# Therefore, we have to check for VERSION updates and the
				# API-VERSION changes in the the required packages.

				# If the package VERSION has changed since the last installer-
				# build, a new release-build is necessary
				if ("${lastInstallerVersion}" VERSION_LESS "${${package}_VERSION}")
					set(requireInstallerBuild "true")
				else()
					__MIRA_CHECK_IF_NEEDS_NEW_INSTALLER(${package} requireInstallerBuild)
				endif()
			endif()

			if (DEFINED requireInstallerBuild)
				message(STATUS "Package ${package} requires new installer build.")
				list(APPEND listOfRequiredInstallerBuilds "${package}")
			endif()

		endif()

	endforeach()

	#######################################################
	# Step 2: Update InstallerBuildStatus

	file(READ "${InstallerStatusPath}" oldInstallerStatus)

	file(WRITE "${InstallerStatusPath}.new")

	foreach(package_BuildStatus ${oldInstallerStatus})
		string(STRIP ${package_BuildStatus} package_BuildStatus)

		separate_arguments(package_BuildStatus)
		list(LENGTH package_BuildStatus listLen)

		if(listLen EQUAL 8)
			list(GET package_BuildStatus 0 packageName)
			list(GET package_BuildStatus 1 packageAPIVersion)
			list(GET package_BuildStatus 2 packageVersion)
			list(GET package_BuildStatus 3 packageBuildNr)
			list(GET package_BuildStatus 4 packageFlag)
			list(GET package_BuildStatus 5 newPackageAPIVersion)
			list(GET package_BuildStatus 6 newPackageVersion)
			list(GET package_BuildStatus 7 newPackageBuildNr)

			unset(pkgInNeedReleaseBuildList)
			list(FIND listOfRequiredInstallerBuilds ${packageName} pkgInNeedReleaseBuildList)
			if(NOT ${pkgInNeedReleaseBuildList} EQUAL -1)
				set(packageFlag "true")
				set(newPackageAPIVersion "${${packageName}_APIVERSION}")
				set(newPackageVersion "${${packageName}_VERSION}")
				set(newPackageBuildNr "${newBuildID}")
			endif()

			file(APPEND "${InstallerStatusPath}.new"
				"${packageName} ${packageAPIVersion} ${packageVersion} ${packageBuildNr} ${packageFlag} ${newPackageAPIVersion} ${newPackageVersion} ${newPackageBuildNr};\n")
		endif()

	endforeach(package_BuildStatus)

	file(RENAME "${InstallerStatusPath}.new" "${InstallerStatusPath}")

endmacro(__MIRA_UPDATE_INSTALLER_STATUS)

###############################################################################

macro(MIRA_ADD_INSTALLERS)

	# Update the InstallerBuildStatus
	__MIRA_UPDATE_INSTALLER_STATUS()

	# create some global "installer" targets
	add_custom_target(all-rpms)
	add_custom_target(all-debs)
	add_custom_target(all-tars)
	add_custom_target(all-zips)
	add_custom_target(new-rpms)
	add_custom_target(new-debs)
	add_custom_target(new-tars)
	add_custom_target(new-zips)

	foreach(package ${MIRA_INTERNAL_PACKAGE_LIST})

#		message(STATUS "* Processing package '${package}'")
#		message(STATUS "*   Version       : '${${package}_VERSION}'")
#		message(STATUS "*   Build         : '${${package}_PATCH}'")
#		message(STATUS "*   Components    : '${${package}_COMPONENTS}'")
#		message(STATUS "*   Requirements  : '${${package}_REQUIREMENTS}'")

		# Only process this package, if this is a local package
		__MIRA_IS_LOCAL_PACKAGE(${package} packageIsLocal)
		if("${packageIsLocal}" STREQUAL "true")

			#######################################################################
			# Collect all requirements for the current package recursivly

			unset(fullPkgList)

			unset(wrkList)
			list(APPEND wrkList ${${package}_REQUIREMENTS})
			list(LENGTH wrkList wrkListLen)
			while(wrkListLen GREATER 0)
				# Take the first element from the list and remove it
				list(GET wrkList 0 requires)
				list(REMOVE_AT wrkList 0)

				# Add the list element to the requirements, if it is not already there.
				list(FIND fullPkgList ${requires} found)
				if(${found} EQUAL -1)
					list(APPEND fullPkgList ${requires})
					list(APPEND wrkList ${${requires}_REQUIREMENTS})
				endif()

				list(LENGTH wrkList wrkListLen)
			endwhile()

#			message(STATUS "*   ResursivReq.  : '${fullPkgList}'")

			#######################################################################
			# Add version information to the requirements

			unset(fullRequirements)

			foreach(requires ${fullPkgList})
				list(FIND MIRA_EXTERNAL_PACKAGE_LIST ${requires} isExternalPackage)
				if(${isExternalPackage} EQUAL -1)
#					message(STATUS "*   RequireVersion: '${requires}' '${${requires}_VERSION}'")
					if(DEFINED ${requires}_APIVERSION)
						list(APPEND fullRequirements "${requires}=${${requires}_APIVERSION}")
					else()
						list(APPEND fullRequirements "${requires}")
					endif()
				ELSE()
#					message(STATUS "*   Skipping external requirement '${requires}'")
				endif()
			endforeach()

			#######################################################################
			# Collect some more information

			set(vendor "MLAB+NICR")
			set(license "MLAB+NICR")

			set(descriptionFileName "${CMAKE_BINARY_DIR}/description_${package}.txt")
			set(changelogFileName "${CMAKE_BINARY_DIR}/changelog_${package}.txt")

			#######################################################################

			if(NOT DEFINED ${package}_APIVERSION OR NOT DEFINED ${package}_PATCH)
				message("-- WARNING - Skipping build installer for ${package}, since version information are missing.")
			else()

				# Build a list corresponding development component names.
				set(componentsWithDevel "")
				foreach(componentName ${${package}_COMPONENTS})
					list(APPEND componentsWithDevel ${componentName}-devel)
				endforeach()

				# The devel component requires the corresponding component.
				set(develRequiresPackage
					"MIRA-${package}=${${package}_APIVERSION}-${${package}_PATCH}")

				# Get the current date code YYYYMMDD
				__MIRA_GET_DATE_CODE(ReleaseDateCode)

				# Get the new(!) release build number (old from "InstallerBuildStatus.txt"+1)
				__MIRA_GET_LAST_INSTALLER_VERSION(${package}
				                                  pkgLastInstallerAPIVersion
				                                  pkgLastInstallerVersion
				                                  pkgLastInstallerBuildNr
				                                  pkgInstallerRebuildFlag)
				math(EXPR pkgNewInstallerBuildNr ${pkgLastInstallerBuildNr}+1)

				###############################################################
				# RPM

				add_custom_target(${package}-rpm
					${CMAKE_COMMAND}
						-D "MIRA_INSTALL_PACKAGE=${package}"
						-D "MIRA_INSTALL_VERSION=${${package}_APIVERSION}"
						-D "MIRA_INSTALL_RELEASE=${${package}_PATCH}_${pkgNewInstallerBuildNr}"
						-D "MIRA_INSTALL_AUTHOR=${${package}_AUTHOR}"
						-D "MIRA_INSTALL_DESCRIPTION_FILE=${descriptionFileName}"
						-D "MIRA_INSTALL_CHANGELOG_FILE=${changelogFileName}"
						-D "MIRA_INSTALL_LICENSE=${vendor}"
						-D "MIRA_INSTALL_VENDOR=${license}"
						-D "MIRA_INSTALL_COMPONENTS=\"${${package}_COMPONENTS}\""
						-D "MIRA_INSTALL_REQUIREMENTS=\"${fullRequirements}\""
						-D "MIRA_ROOT_DIR=${MIRA_ROOT_DIR}"
						-D "BUILD_BINARY_DIR=${CMAKE_BINARY_DIR}"
						-P "${MIRA_ROOT_DIR}/make/BuildInstaller-RPM.cmake"

					${CMAKE_COMMAND}
						-D "MIRA_INSTALLERBUILD_PACKAGE=${package}"
						-D "MIRA_INSTALLERBUILD_APIVERSION=${${package}_APIVERSION}"
						-D "MIRA_INSTALLERBUILD_VERSION=${${package}_VERSION}"
						-D "MIRA_INSTALLERBUILD_BUILDNR=${pkgNewInstallerBuildNr}"
						-D "MIRA_INSTALLERBUILD_PATH=${InstallerStatusPath}"
						-P "${MIRA_ROOT_DIR}/make/UpdateInstallerBuildStatus.cmake"

				)
				add_dependencies(all-rpms ${package}-rpm)
				if ("${pkgInstallerRebuildFlag}" STREQUAL "true")
					add_dependencies(new-rpms ${package}-rpm)
				endif()

				###############################################################
				# DEB

				add_custom_target(${package}-deb
					${CMAKE_COMMAND}
						-D "MIRA_INSTALL_PACKAGE=${package}"
						-D "MIRA_INSTALL_VERSION=${${package}_VERSION}"
						-D "MIRA_INSTALL_BUILD=${pkgNewInstallerBuildNr}"
						-D "MIRA_INSTALL_AUTHOR=${${package}_AUTHOR}"
						-D "MIRA_INSTALL_DESCRIPTION_FILE=${descriptionFileName}"
						-D "MIRA_INSTALL_CHANGELOG_FILE=${changelogFileName}"
						-D "MIRA_INSTALL_LICENSE=${vendor}"
						-D "MIRA_INSTALL_VENDOR=${license}"
						-D "MIRA_INSTALL_COMPONENTS=\"${${package}_COMPONENTS}\""
						-D "MIRA_INSTALL_REQUIREMENTS=\"${fullRequirements}\""
						-D "MIRA_ROOT_DIR=${MIRA_ROOT_DIR}"
						-D "BUILD_BINARY_DIR=${CMAKE_BINARY_DIR}"
						-P "${MIRA_ROOT_DIR}/make/BuildInstaller-DEB.cmake"

					${CMAKE_COMMAND}
						-D "MIRA_INSTALLERBUILD_PACKAGE=${package}"
						-D "MIRA_INSTALLERBUILD_APIVERSION=${${package}_APIVERSION}"
						-D "MIRA_INSTALLERBUILD_VERSION=${${package}_VERSION}"
						-D "MIRA_INSTALLERBUILD_BUILDNR=${pkgNewInstallerBuildNr}"
						-D "MIRA_INSTALLERBUILD_PATH=${InstallerStatusPath}"
						-P "${MIRA_ROOT_DIR}/make/UpdateInstallerBuildStatus.cmake"

				)
				add_dependencies(all-debs ${package}-deb)
				if ("${pkgInstallerRebuildFlag}" STREQUAL "true")
					add_dependencies(new-debs ${package}-deb)
				endif()

				###############################################################
				# TAR

				add_custom_target(${package}-tar
					${CMAKE_COMMAND}
						-D "MIRA_INSTALL_PACKAGE=${package}"
						-D "MIRA_INSTALL_VERSION=${${package}_APIVERSION}"
						-D "MIRA_INSTALL_RELEASE=${${package}_PATCH}_${pkgNewInstallerBuildNr}"
						-D "MIRA_INSTALL_COMPONENTS=\"${${package}_COMPONENTS}\""
						-D "MIRA_ROOT_DIR=${MIRA_ROOT_DIR}"
						-D "BUILD_BINARY_DIR=${CMAKE_BINARY_DIR}"
						-P "${MIRA_ROOT_DIR}/make/BuildInstaller-TAR.cmake"

					${CMAKE_COMMAND}
						-D "MIRA_INSTALLERBUILD_PACKAGE=${package}"
						-D "MIRA_INSTALLERBUILD_APIVERSION=${${package}_APIVERSION}"
						-D "MIRA_INSTALLERBUILD_VERSION=${${package}_VERSION}"
						-D "MIRA_INSTALLERBUILD_BUILDNR=${pkgNewInstallerBuildNr}"
						-D "MIRA_INSTALLERBUILD_PATH=${InstallerStatusPath}"
						-P "${MIRA_ROOT_DIR}/make/UpdateInstallerBuildStatus.cmake"
				)
				add_dependencies(all-tars ${package}-tar)
				if ("${pkgInstallerRebuildFlag}" STREQUAL "true")
					add_dependencies(new-tars ${package}-tar)
				endif()

				###############################################################
				# ZIP for mirapackage

				add_custom_target(${package}-zip
					${CMAKE_COMMAND}
						-D "MIRA_INSTALL_PACKAGE=${package}"
						-D "MIRA_INSTALL_VERSION=${${package}_VERSION}"
						-D "MIRA_INSTALL_DATECODE=${ReleaseDateCode}"
						-D "MIRA_INSTALL_COMPONENTS=\"${${package}_COMPONENTS}\""
						-D "MIRA_INSTALL_REQUIREMENTS=\"${fullRequirements}\""
						-D "MIRA_INSTALL_DESTPATH=\"${InstallerRootDir}\""
						-D "MIRA_INSTALL_STATUS_FILE=\"${InstallerStatusPath}\""
						-D "MIRA_ROOT_DIR=${MIRA_ROOT_DIR}"
						-D "BUILD_BINARY_DIR=${CMAKE_BINARY_DIR}"
						-P "${MIRA_ROOT_DIR}/make/BuildInstaller-MIRA-ZIP.cmake"

					${CMAKE_COMMAND}
						-D "MIRA_INSTALLERBUILD_PACKAGE=${package}"
						-D "MIRA_INSTALLERBUILD_APIVERSION=${${package}_APIVERSION}"
						-D "MIRA_INSTALLERBUILD_VERSION=${${package}_VERSION}"
						-D "MIRA_INSTALLERBUILD_BUILDNR=${ReleaseDateCode}"
						-D "MIRA_INSTALLERBUILD_PATH=${InstallerStatusPath}"
						-P "${MIRA_ROOT_DIR}/make/UpdateInstallerBuildStatus.cmake"
				)
				add_dependencies(all-zips ${package}-zip)
				if ("${pkgInstallerRebuildFlag}" STREQUAL "true")
					add_dependencies(new-zips ${package}-zip)
				endif()

			endif()

		endif()

	endforeach()

endmacro(MIRA_ADD_INSTALLERS)
