/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file FlowLayout.C
 *    Implementation of FlowLayout.
 *
 * @author Erik Einhorn
 * @date   2012/12/11
 */

#include <widgets/FlowLayout.h>

#include <assert.h>
#include <iostream>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

FlowLayout::FlowLayout(QWidget* parent)
: QLayout(parent)
{
	init();
}

void FlowLayout::init()
{
	mVLayout = new QVBoxLayout();

	QHBoxLayout* hl = new QHBoxLayout();
	mVLayout->addLayout(hl);
	mHLayouts.push_back(hl);
}

void FlowLayout::clearLayouts()
{
	foreach(QHBoxLayout* hbox, mHLayouts)
	{
		while(hbox->count()>0)
			hbox->removeItem(hbox->itemAt(0));
	}
}

FlowLayout::~FlowLayout()
{
	clearLayouts();

	QLayoutItem *item;
	while ((item = takeAt(0)))
		delete item;

	foreach(QHBoxLayout* hbox, mHLayouts)
		delete hbox;
	delete mVLayout;
}

void FlowLayout::addItem(QLayoutItem *item, int stretch)
{
	mItemList.append(std::make_pair(item,stretch));
}

int FlowLayout::count() const
{
	return mItemList.size();
}

QLayoutItem *FlowLayout::itemAt(int index) const
{
	return mItemList.value(index).first;
}

QLayoutItem *FlowLayout::takeAt(int index)
{
	if (index >= 0 && index < mItemList.size())
		return mItemList.takeAt(index).first;
	else
		return 0;
}

Qt::Orientations FlowLayout::expandingDirections() const
{
	return 0;
}

bool FlowLayout::hasHeightForWidth() const
{
	return true;
}

int FlowLayout::heightForWidth(int width) const
{
	// Simply return the current height, independent from the size.
	// This gives desireable results and does not require to compute the
	// whole layout beforehand.
	return minimumSize().height();
}

void FlowLayout::setGeometry(const QRect &rect)
{
	QLayout::setGeometry(rect);

	// clear all hlayouts
	clearLayouts();

	int row = 0;
	QHBoxLayout* hbox = mHLayouts.at(row);

	// insert our items step by step
	for(int i=0; i<mItemList.size(); ++i)
	{
		QLayoutItem *item = mItemList.at(i).first;
		int itemStretch = mItemList.at(i).second;

		if(hbox->count()>0 && (hbox->sizeHint().width()+item->sizeHint().width()) > rect.width())
		{
			++row;
			// we need to put the remaining items into the next hboxlayout
			if(row>=mHLayouts.size()) {
				QHBoxLayout* hl = new QHBoxLayout();
				mHLayouts.push_back(hl);
				mVLayout->addLayout(hl);
			}
			hbox = mHLayouts.at(row);
		}
		hbox->addItem(item);
		hbox->setStretch(hbox->count()-1,itemStretch);
	}

	// remove exceeding hlayouts
	while(mHLayouts.size()>(row+1) && mHLayouts.size()>=2) {
		mVLayout->removeItem(mHLayouts.back());
		delete mHLayouts.back();
		mHLayouts.pop_back();
	}

	mVLayout->setGeometry(rect);
}

QSize FlowLayout::sizeHint() const
{
	QSize size = mVLayout->sizeHint();
	size.expandedTo(mHLayouts.back()->sizeHint());
	return size;

}

QSize FlowLayout::minimumSize() const
{
	QSize size;
	foreach (auto item, mItemList)
		size = size.expandedTo(item.first->minimumSize());
	size += QSize(2*margin(), 2*margin());

	size.setHeight(std::max(mVLayout->minimumSize().height(),
	                        mHLayouts.back()->sizeHint().height()));
	return size;
}

///////////////////////////////////////////////////////////////////////////////

}
