/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Visualization3DContainer.h
 *    A standalone container for 3D visualizations, similar to
 *    Visualization3DView but as lightweight standalone widget instead of a
 *    view within the RCP framework.
 *
 * @author Erik Einhorn
 * @date   2012/09/01
 */

#ifndef _MIRA_VISUALIZATION3DCONTAINER_H_
#define _MIRA_VISUALIZATION3DCONTAINER_H_

#include <list>

#include <widgets/OgreWidgetEx.h>

#include <visualization/Visualization3D.h>
#include <visualization/VisualizationTool3D.h>

namespace mira {

class PropertyEditor;

///////////////////////////////////////////////////////////////////////////////

/**
 * A standalone container for 3D visualizations, similar to
 * Visualization3DView but as lightweight standalone widget instead of a
 * view within the RCP framework.
 */
class MIRA_GUI_VISUALIZATION_EXPORT Visualization3DContainer : public OgreWidgetEx, public IVisualization3DSite
{
public:
	Visualization3DContainer(QWidget* parent=NULL);
	virtual ~Visualization3DContainer();

public:

	/**
	 * Creates an instance of the specified visualization class, adds it to the
	 * container and returns a pointer to it.
	 */
	Visualization3D* addVisualization(const std::string& visualizationClassName);

	/**
	 * Adds the specified visualization instance to the container.
	 * The container will take ownership of the visualization.
	 */
	void addVisualization(Visualization3D* vis);

	/// Returns the value of the specified property for the specified visualization as string.
	std::string getPropertyString(Visualization3D* vis, const std::string& property);

	/// Returns the value of the specified property for the specified visualization.
	template <typename T>
	T getProperty(Visualization3D* vis, const std::string& property) {
		return fromString<T>(getPropertyString(vis,property));
	}

	/// Sets the value of the specified property for the specified visualization from string.
	void setPropertyString(Visualization3D* vis, const std::string& property, const std::string& value);

	/// Sets the value of the specified property for the specified visualization.
	template <typename T>
	void setProperty(Visualization3D* vis, const std::string& property, const T& value) {
		setPropertyString(vis,property,toString<T>(value));
	}

	/**
	 * Creates a property editor (if not yet created) and returns the pointer
	 * to it. The widget is hidden by default. Hence, you should call its
	 * show() method. Moreover, the widget has no parent by default. The parent
	 * can be set in the first call of this method as optional parameter.
	 */
	PropertyEditor* getPropertyEditor(QWidget* parent = NULL);

public:
	// implementation of IVisualization3DSite
	virtual Ogre::SceneManager* getSceneManager();

	virtual void registerInteractionListener(InteractionListener3D* listener);
	virtual void removeInteractionListener(InteractionListener3D* listener);
	virtual std::set<InteractionListener3D*> getInteractionListeners();

	virtual const Camera& getCamera() const;

	virtual void setCamera(const Camera& camera);

	virtual Ogre::Camera* acquireCamera(boost::function<void()> lostCameraCallback = boost::function<void()>());

	virtual void releaseCamera();

	virtual const std::string& getFixedFrame() const;
	virtual const std::string& getCameraFrame() const;

	void setFixedFrame(const std::string& frame);
	void setCameraFrame(const std::string& frame);

public:

	virtual void mousePressEvent(QMouseEvent* e);
	virtual void mouseReleaseEvent(QMouseEvent* e);
	virtual void mouseMoveEvent(QMouseEvent* e);
	virtual void wheelEvent(QWheelEvent* e);

	virtual void timerEvent(QTimerEvent* e);

protected:

	virtual void initOgre();

private:

	void setupVisualizations();

private:

	mutable Camera mCameraSettings;

	boost::shared_ptr<VisualizationTool3D> mActiveTool;

	Time mLastUpdateTime;

	boost::shared_ptr<PropertyNode> mGeneralProperties;
	std::map<Visualization3D*, boost::shared_ptr<PropertyNode>> mVisualizations;

	PropertyEditor* mEditor;

	std::string mFixedFrame;
	std::string mCameraFrame;

	boost::mutex mListenerMutex;
	std::set<InteractionListener3D*> mListeners;
};

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
