/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Visualization2D.h
 *    Declaration of Visualization2D and related classes.
 *
 * @author Erik Einhorn
 * @date   2011/01/15
 */

#ifndef _MIRA_VISUALIZATION2D_H_
#define _MIRA_VISUALIZATION2D_H_

#include <geometry/Point.h>
#include <math/Angle.h>
#include <visualization/Visualization.h>

//////////////////////////////////////////////////////////////////////////////

class QGraphicsScene;
class QGraphicsView;
class QGraphicsItem;

namespace mira {

//////////////////////////////////////////////////////////////////////////////

class IVisualization2DSite : public IVisualizationSiteTransformable
{
public:

	virtual QGraphicsScene* getSceneManager() = 0;
	virtual QGraphicsView* getViewManager() = 0;

public:

	class MIRA_GUI_VISUALIZATION_EXPORT Camera
	{
	public:

		Camera();

		template <typename Reflector>
		void reflect(Reflector& r)
		{
			r.property("X", position.x(), "the position of the camera");
			r.property("Y", position.y(), "the position of the camera");
			r.property("Orientation", orientation, "Rotation of the camera w.r.t. the global coordinate system");
			r.property("Scale", scale, "the scale factor");
		}

		// The camera rotation and translation, specified by mouse interaction

		Point2f position;
		Degreef orientation;
		float scale;
	};

	/**
	 * Obtains the current camera settings (position, orientation, etc.) of the
	 * views own camera.
	 */
	virtual const Camera& getCamera() const = 0;

	/**
	 * Sets the current camera settings (position, orientation, etc.) of the
	 * views own camera.
	 */
	virtual void setCamera(const Camera& camera) = 0;

	/**
	 * Obtains the Z value all tools should adhere to to ensure that they will
	 * always be above regular visualizations.
	 */
	virtual float getToolBaseZValue() const = 0;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Abstract base class for all 2D visualization that are used by
 * Visualization2DView.
 */
class MIRA_GUI_VISUALIZATION_EXPORT Visualization2D :
	public Visualization
{
	MIRA_OBJECT(Visualization2D)
public:

	Visualization2D();
	virtual ~Visualization2D() {}

public:

	/**
	 * Is called by the visualization view when the visualization is attached.
	 * The default implementation stores the pointer to the IVisualization2DSite
	 * interface that can be retrieved using the getSite() method for later use.
	 * Moreover, it calls the setupScene() method.
	 * You should never need to overwrite this method in derived classes. Use
	 * the setupScene() method instead. However, if you overwrite this method
	 * you SHOULD call this default implementation.
	 *
	 * The passed interface must be of the type IVisualization2DSite.
	 */
	virtual void init(IVisualizationSite* site);

public:

	/**
	 * Returns a pointer to the IVisualization2DSite interface that was set
	 * in the init() method.
	 * Or returns NULL, if init() was not yet called.
	 */
	virtual IVisualization2DSite* getSite();

public:

	/**
	 * Must be implemented in derived classes. Is called within the
	 * visualization's init() method. You can use this method to create
	 * your scene and scene objects.
	 * The passed IVisualization2DSite interface is the same that is
	 * returned by getSite().
	 */
	virtual void setupScene(IVisualization2DSite* site) = 0;

	/**
	 * Must be implemented in derived classes to return a pointer to
	 * the QGraphicsItem used for drawing.
	 */
	virtual QGraphicsItem* getItem() = 0;

private:

	IVisualization2DSite* mSite;

};

//////////////////////////////////////////////////////////////////////////////

}

#endif
