/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file DynamicRenderable.h
 *    Base class providing dynamically growing hardware buffer
 *    Adapted from http://www.ogre3d.org/tikiwiki/DynamicGrowingBuffers&structure=Cookbook
 *
 *    Original file: DynamicRenderable.h
 */

#ifndef _MIRA_DYNAMICRENDERABLE_H_
#define _MIRA_DYNAMICRENDERABLE_H_

#include <visualization/GuiVisualizationExports.h>

#ifndef Q_MOC_RUN
#include <OGRE/OgreSimpleRenderable.h>
#endif

#include <utils/EnumToFlags.h>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

/**
 * Abstract base class for renderables based on dynamically growing
 * vertex and index buffers.
 *
 * The usage is as follows:
 *
 * -# call initialize() once, to create all necessary components and to provide
 *    the type of render operations, the layout of the vertex buffer and specify
 *    whether to use an index buffer or not.
 *
 * -# call prepareBuffers() to specify the size of the vertex and index buffer
 * -# fill in the vertex and index buffers, which can be obtained using the
 *    getVertexBuffer() and getIndexBuffer() methods.
 * -# set the bounding box of the Renderable using the setBoundingBox() method.
 * -# continue at 2 or 3, whenever the content needs to be updated.
 */
class MIRA_GUI_VISUALIZATION_EXPORT DynamicRenderable : public Ogre::SimpleRenderable
{
public:
	DynamicRenderable();
	virtual ~DynamicRenderable();

	/**
	 * Initializes the dynamic renderable.
	 * This function should only be only called once! It initializes all
	 * necessary structures and specifies the operation type (triangle list,
	 * fan, etc), the vertex declaration (vertex,normals,texturecoords)
	 * and specifies, whether to use an index buffer or not.
	 * @param operationType The type of render operation to perform.
	 * @param vertexDecl The vertex declaration describing the content of the vertex buffer
	 * @param useIndices Specifies whether to use an index buffer or not
	 */
	void initialize(Ogre::RenderOperation::OperationType operationType,
			const Ogre::VertexDeclaration& vertexDecl, bool useIndices);

	/// Allows to change the operation type after calling initialize
	void setOperationType(Ogre::RenderOperation::OperationType operationType);

	/**
	 * Allows to enable/disable the index buffer, after initialize() was
	 * already called.
	 * Note, that you can enable the index buffer only, is the parameter
	 * useIndices in the call to initialize() was set true. Otherwise, this
	 * method will throw an exception here.
	 */
	void enableIndexBuffer(bool enable=true);


	/**
	 * Flags that can be specified in prepareHardwareBuffers().
	 */
	enum Flags
	{
		NONE = 0,
		INDEX_BUFFER_32BIT = 1,  ///< create a 32bit index buffer instead of 16bit
	};
	MIRA_ENUM_TO_FLAGS_INCLASS(Flags);

	/**
	 * Prepares the hardware buffers for the requested vertex and index counts.
	 * This function must be called at least once before locking the buffers
	 * to fill in data. It guarantees that the hardware buffers are large
	 * enough to hold at least the requested number of vertices and indices
	 * (if using indices). The buffers are possibly reallocated to achieve this.
	 * This method can be called more than once to increase the size of the
	 * buffers, if necessary.
	 *
	 * @param vertexCount The number of vertices the buffer must hold.
	 * @param indexCount The number of indices the buffer must hold. This
	 *                    parameter is ignored if not using indices.
	 */
	void prepareBuffers(size_t vertexCount, size_t indexCount=0,
	                    Flags flags = NONE,
	                    Ogre::HardwareBuffer::Usage usage=Ogre::HardwareBuffer::HBU_DYNAMIC_WRITE_ONLY);


	/// Gets the vertex buffer bound to the given source index
	Ogre::HardwareVertexBufferSharedPtr getVertexBuffer(unsigned short index = 0);

	/// Gets the index buffer of this renderable (or NULL, if no index buffer was created)
	Ogre::HardwareIndexBufferSharedPtr getIndexBuffer();


public:

	/// Implementation of Ogre::SimpleRenderable
	virtual Ogre::Real getBoundingRadius(void) const;
	/// Implementation of Ogre::SimpleRenderable
	virtual Ogre::Real getSquaredViewDepth(const Ogre::Camera* cam) const;

#if OGRE_VERSION < 0x11000
	// workaround for buggy implementation not respecting mRenderQueuePriority,
	// fixed in OGRE 1.10
public:

	virtual void _updateRenderQueue(Ogre::RenderQueue* queue) {
		queue->addRenderable( this, mRenderQueueID, mRenderQueuePriority);
	}
#endif

protected:
	/// Maximum capacity of the currently allocated vertex buffer.
	size_t mVertexBufferCapacity;
	/// Maximum capacity of the currently allocated index buffer.
	size_t mIndexBufferCapacity;
};

//////////////////////////////////////////////////////////////////////////////

}

#endif
