/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file WorkbenchPart.h
 *    Declaration of WorkbenchPart and related interfaces.
 *
 * @author Erik Einhorn
 * @date   2010/12/01
 */

#ifndef _MIRA_WORKBENCHPART_H_
#define _MIRA_WORKBENCHPART_H_

#include <factory/Factory.h>
#include <rcp/Adaptable.h>
#include <rcp/GuiRCPExports.h>

class QWidget;

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class IPartListener;
class IWorkbenchPart;

///////////////////////////////////////////////////////////////////////////////

/**
 * The primary interface between a workbench part and the workbench.
 */
class IWorkbenchPartSite
{
public:
	virtual ~IWorkbenchPartSite() {}

	virtual void addPartListener(IPartListener* listener) = 0;
	virtual void removePartListener(IPartListener* listener) = 0;

	/// Returns the currently active editor part, or NULL, if no editor is active
	virtual IWorkbenchPart* getActiveEditor() = 0;

	/**
	 * Returns a list of all editors of this workbench site.
	 */
	virtual std::list<IWorkbenchPart*> getEditors() = 0;
};

///////////////////////////////////////////////////////////////////////////////

class IWorkbenchPart : public Object, public IAdaptable
{
public:
	virtual ~IWorkbenchPart() {}

public:

	/// Is called by the Workbench to inform the Part that is being activated by the user.
	virtual void activate() = 0;

	/// Is called by the Workbench to inform the Part that is being deactivated since another Part got the focus.
	virtual void deactivate() = 0;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * A workbench part is a component within the workbench. There are two main
 * subclasses: ViewPart and EditorPart.
 *
 * A view is typically used to display information or properties. Views are
 * shown as dock windows. Only one instance of a particular view type may
 * exist within the workbench. This policy is designed to simplify the view
 * management for the user.
 *
 * An editor is typically used to edit or browse data or objects. Editors are
 * shown as tabbed MDI windows. In contrast to views there may exist multiple
 * instances of a certain editor type.
 */
class MIRA_GUI_RCP_EXPORT WorkbenchPart : public IWorkbenchPart
{
	MIRA_OBJECT(WorkbenchPart)
public:

	WorkbenchPart();

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		// nothing to do here
	}

public: // implementation if IWorkbenchPart

	virtual void activate();
	virtual void deactivate();

	/// Returns true if this ViewPart currently has the focus and hence the users attention
	bool isActivated() const { return mIsActivated; }

public:

	/**
	 * Is called by the workbench.
	 * The WorkbenchPart should initialize itself and my store the
	 * pointer tothe IWorkbenchPartSite interface for later use.
	 */
	virtual void init(IWorkbenchPartSite* site) = 0;

public:

	virtual Object* getAdapter(const Class& adapter) { return NULL; }

private:

	/// Is true, if this is the view that is "selected" by the user, i.e. that has the focus.
	bool mIsActivated;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
