/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TitleBar.h
 *    Base class for all window title bars (EditorPartTitleBar,
 *    ViewPartTitleBar).
 *
 * @author Erik Einhorn
 * @date   2011/09/08
 */

#ifndef _MIRA_TITLEBAR_H_
#define _MIRA_TITLEBAR_H_

#include <QWidget>
#include <QToolButton>
#include <QHBoxLayout>
#include <QMenu>

#include <rcp/GuiRCPExports.h>

namespace mira
{
///////////////////////////////////////////////////////////////////////////////

/**
 * Base class for all window title bars (EditorPartTitleBar,
 * ViewPartTitleBar).
 * This class provides painting of the title bar, and adds several default
 * buttons to the title:
 * - left:  Dock, Undock
 * - right: Menu, Minimize, Maximize, Restore, Close
 */
class MIRA_GUI_RCP_EXPORT TitleBar : public QWidget
{

public:

	TitleBar(QWidget *parent = 0);
	virtual ~TitleBar();

public:

	/// enables/disables highlighted title bar
	void setHighlight(bool on);

	/// updates the TitleBar (e.g. necessary after the title was changed)
	void update();

public:

	/**
	 * Returns a widget containing all buttons of the title bars right side.
	 * The widget can be placed somewhere else and later put back using
	 * putBackRightButtons().
	 * This is useful, when the icons should be placed into a different
	 * widget than this title bar, e.g. when the window is maximized and the
	 * title bar is hidden.
	 */
	QWidget* takeRightButtons();

	/// Puts back the right buttons, that were taken using the above method
	void putBackRightButtons();

	/**
	 * Returns a widget containing all buttons of the title bars left side.
	 * The widget can be placed somewhere else and late put back using
	 * putBackLeftButtons().
	 * This is useful, when the icons should be placed into a different
	 * widget than this title bar, e.g. when the window is maximized and the
	 * title bar is hidden.
	 */
	QWidget* takeLeftButtons();

	/// Puts back the left buttons, that were taken using the above method
	void putBackLeftButtons();

	/**
	 * Inserts the specified button within the right button group at the given
	 * index. The default index is 0 which will add the button on the left side.
	 * Note that the parent of the button will be changed to the right button
	 * group widget.
	 */
	void insertRightButton(QToolButton* button, int index = 0);

	/**
	 * Inserts the specified button within the left button group at the given
	 * index. The default index is -1 which will add the button on the right
	 * side.
	 * Note that the parent of the button will be changed to the left button
	 * group widget.
	 */
	void insertLeftButton(QToolButton* button, int index = -1);

public:

	/**
	 * Returns the menu of the menu button in order to add new items to the
	 * menu. Calling this method also makes the menu button visible.
	 */
	QMenu* getMenu();

protected:

	bool eventFilter( QObject* obj, QEvent* event);

	void paintEvent(QPaintEvent*);
	void resizeEvent(QResizeEvent*);

protected:

	void updateBackground();

protected:

	QHBoxLayout* mLayout;
	QHBoxLayout* mLeftLayout;
	QHBoxLayout* mRightLayout;
	QToolButton* mMenuBtn;
	QToolButton* mClose;
	QToolButton* mDock;
	QToolButton* mUndock;
	QToolButton* mMaximize;
	QToolButton* mMinimize;
	QToolButton* mRestore;
	QWidget* mLeftButtons;
	QWidget* mRightButtons;

	QWidget* mParent;
	QPixmap *mBackground;

	bool mIsHighlighted;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
