/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file EditorPartArea.h
 *    Declaration of EditorPartArea and EditorPartAreaTabBar.
 *
 * @author Erik Einhorn
 * @date   2011/09/08
 */

#ifndef MIRA_EDITORPARTAREA_H_
#define MIRA_EDITORPARTAREA_H_

#include <set>
#include <iostream>

#include <QWidget>
#include <QTabBar>
#include <QMouseEvent>

#include <rcp/GuiRCPExports.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class EditorPartWindow;
class EditorPartAreaTabBar; // declared below in this file

///////////////////////////////////////////////////////////////////////////////

/**
 * This is a replacement for QMdiArea.
 * It is the MDI area that contains several EditorPartWindow widgets
 * as MDI subwindows.
 */
class MIRA_GUI_RCP_EXPORT EditorPartArea : public QWidget
{
	Q_OBJECT
public:

	EditorPartArea(QWidget* parent=0, Qt::WindowFlags flags=0) ;

public:

	void addSubWindow(EditorPartWindow* window);
	void removeSubWindow(EditorPartWindow* window);

	void activateSubWindow(EditorPartWindow* window);
	EditorPartWindow* activeSubWindow () const;

	void scrollToSubWindow(EditorPartWindow* window);

signals:

	void editorWindowActivated(EditorPartWindow* window);

public slots:

	void tileSubWindows();

public:

	void setBackground(const QBrush& brush);

	void setAutoHideDecorations(bool on);

public:

	virtual bool eventFilter(QObject* object, QEvent* event);
	virtual void resizeEvent(QResizeEvent* event);

protected:

	friend class EditorPartWindow;

	// to be called by EditorPartWindows
	void dockSubwindow(EditorPartWindow* window); // called by subwindow to dock back
	void undockSubwindow(EditorPartWindow* window); // called by subwindow to undock
	void maximizedSubwindow(EditorPartWindow* window); // called by subwindow when it was maximized
	void restoredSubwindow(EditorPartWindow* window); // called by subwindow when it was restored

	// must be called if area was resized, scrolled, or window was newly maximized
	void rearrangeMaximizedSubwindow(EditorPartWindow* window);

	void resizeMaximizedSubwindows(const QSize& size);
	void rearrangeMaximizedSubwindows();
	void updateMaximizedSubwindowsVisability();

	void scrolledAreaBy(int dx, int dy);

protected slots:

	void onFocusChanged(QWidget *old, QWidget *now);

	void onTabChanged(int);
	void onTabDoubleClicked(int);

private:

	void addWindowTab(EditorPartWindow* window);
	void removeWindowTab(EditorPartWindow* window);
	int  findWindowTab(EditorPartWindow* window) const;

	void internalRaise(EditorPartWindow* window) const;

private:

	class ScrollArea;
	friend class ScrollArea;
	ScrollArea* mArea;

	EditorPartAreaTabBar* mTabBar;

	std::set<EditorPartWindow*> mSubWindows;
	std::set<EditorPartWindow*> mDockedSubWindows;

	EditorPartWindow* mActiveSubWindow;

	bool mAutoHideDecorations;
};

///////////////////////////////////////////////////////////////////////////////

class EditorPartAreaTabBar : public QTabBar
{
	Q_OBJECT
public:

	EditorPartAreaTabBar (QWidget* parent) : QTabBar(parent) {
		setAcceptDrops(true);
	}

signals:

	void tabDoubleClicked(int index);
	void tabContextMenu(int index, QPoint globalPos);

protected:

	void dragEnterEvent(QDragEnterEvent* event)
	{
		event->acceptProposedAction();
	}

	void dragMoveEvent(QDragMoveEvent* event)
	{
		int tab = tabAt(event->pos());
		if(tab>=0 && tab!=currentIndex())
			setCurrentIndex(tab);
		event->ignore();
	}

	virtual void mouseDoubleClickEvent(QMouseEvent* e)
	{
		QTabBar::mouseDoubleClickEvent(e);

		if(e->button()==Qt::LeftButton)
			emit tabDoubleClicked(currentIndex());
	}

	virtual void contextMenuEvent(QContextMenuEvent* e)
	{
		emit tabContextMenu(tabAt(e->pos()), e->globalPos());
	}
};

///////////////////////////////////////////////////////////////////////////////
}

#endif
